/*
 * Copyright (C) 2010 Joseph Adams <joeyadams3.14159@gmail.com>
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#include "lambda.h"
#include <tigcclib.h>

// Create a popup menu listing all defined environment entries.
// If a name is selected, return it.  Otherwise, return NULL.
const char *envMenu(Environment env)
{
	size_t count = avl_count(env);
	size_t i;
	
	if (count == 0)
		return NULL;
	
	// Line up the strings so we can look them up by index
	const char **strings = HeapAllocPtr(count * sizeof(*strings));
	if (strings == NULL)
		return NULL;
	
	{
		AvlIter      iter;
		const char **p = strings;
		
		avl_foreach(iter, env)
			*p++ = iter.key;
		
		assert((size_t)(p - strings) == count);
	}
	
	// Build the popup menu
	HANDLE menu = PopupNew("CATALOG", 0);
	if (menu == H_NULL) {
		HeapFreePtr(strings);
		return NULL;
	}
	for (i = 0; i < count; i++) {
		if (PopupAddText(menu, -1, strings[i], i+1) == H_NULL) {
			HeapFree(menu);
			HeapFreePtr(strings);
			return NULL;
		}
	}
	
	// Run the menu, and free it.
	// Note that we need to hang on to the string table so we can decode the return value.
	short result = PopupDo(menu, CENTER, CENTER, 0);
	HeapFree(menu);
	
	// Decode the result, and free the string table.
	const char *ret = result > 0 ? strings[result-1] : NULL;
	HeapFreePtr(strings);
	
	return ret;
}