/*
 * Copyright (C) 2010 Joseph Adams <joeyadams3.14159@gmail.com>
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#include <tigcclib.h>

#include "lambda.h"

const char *file_to_run;

bool suppressError;

void error_(const char *fmt, ...)
{
	if (!suppressError) {
		va_list ap;
		va_start(ap, fmt);
		vprintf(fmt, ap);
	}
	ER_throw(ER_NO_MSG);
}

int long_cmp(const void *ap, const void *bp)
{
	return (long)ap - (long)bp;
}

volatile int break_pressed;
static INT_HANDLER old_on_key_handler;

DEFINE_INT_HANDLER (on_key_handler)
{
	break_pressed = 1;
}

void enable_break(void)
{
	SetIntVec(INT_VEC_ON_KEY_PRESS, on_key_handler);
}

void disable_break(void)
{
	SetIntVec(INT_VEC_ON_KEY_PRESS, old_on_key_handler);
}

FILE *openHistory(const char *fileName)
{
	FILE *f;
	
	// Open the file (if it exists) to read previous commands
	f = fopen(fileName, "rt");
	if (f != NULL) {
		while (fgets(prompt_buffer, sizeof(prompt_buffer), f)) {
			chomp(prompt_buffer);
			add_history(prompt_buffer);
		}
		fclose(f);
	}
	
	// Now open the file to add more commands
	f = fopen(fileName, "at");
	
	return f;
}

void main2(void)
{
	Environment  env;
	FILE        *history;
	
	old_on_key_handler = GetIntVec(INT_VEC_ON_KEY_PRESS);
	
	clrscr();
	unsigned char oldFont = FontGetSys();
	
	randomize();
	
	bindings = avl_new(long_cmp);
	add_binding(4372, gc_strdup("it"));      // ANS       -> it
	if (TI89)
		add_binding(16434, gc_strdup("\x89")); // DIAMOND+2 -> 
	
	env = newEnv();
	
	if (file_to_run != NULL) {
		if (file_to_run[0] != '\0') // allow lambda(""), which doesn't load anything
			env = import(file_to_run, env, true);
	} else {
		env = import("prelude", env, false);
	}
	
	history = openHistory("history");
	if (history == NULL)
		printf("WARNING: can't write to history file\n");
	
	const char *line;
	
	TRY
		while ((line = prompt("> ", true, env)) != NULL) {
			if (streq(line, "exit"))
				break;
			
			const char *s = line;
			skipSpace(&s);
			if (*s == '\0')
				continue;
			
			// Add to history file
			if (history) {
				fputs(line, history);
				putc('\n', history);
			}
			
			// Add to current session history
			add_history(line);
			
			env = runCommand(line, env);
		}
	FINALLY
		if (history)
			fclose(history);
		FontSetSys(oldFont);
		disable_break();
	ENDFINAL
}

// Main Function
void _main(void)
{
	ESI           argptr     = top_estack;
	unsigned char argtype    = GetArgType(argptr);
	bool          ignore_ram = false;
	
	if (argtype == STR_TAG) {
		file_to_run = GetStrnArg(argptr);
	} else if (argtype == POSINT_TAG) {
		gc_set_verbose_level(GetIntArg(argptr));
		ignore_ram = true;
	}
	
	if (!ignore_ram) {
		unsigned long avail = HeapAvail();
		
		if (avail < 75000) {
			char buffer[256];
			
			sprintf(buffer, "Need %lu more bytes of RAM", 75000 - avail);
			
			DlgMessage("\x89-calculator", buffer, BT_CANCEL, BT_NONE);
			
			return;
		}
	}
	
	gc_main(main2);
}