/*
 * Copyright (C) 2010 Joseph Adams <joeyadams3.14159@gmail.com>
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#include "lambda.h"

typedef struct VarStack VarStack;

struct VarStack {
	const VarStack *next;
	const char     *name; // NULL used for anonymous temporary variables
};

static LExpr *liftValue(Value *value)
{
	LExpr *ret = mkLExpr(L_VALUE);
	ret->value = value;
	return ret;
}

static LExpr *mkVar(unsigned int var)
{
	LExpr *ret = mkLExpr(L_VAR);
	ret->var = var;
	return ret;
}

static LExpr *mkLambda(LExpr *inner)
{
	LExpr *ret = mkLExpr(L_LAMBDA);
	ret->lambda = inner;
	return ret;
}

static LExpr *mkAp(LExpr *f, LExpr *x)
{
	LExpr *ret = mkLExpr(L_AP);
	ret->ap.f = f;
	ret->ap.x = x;
	return ret;
}

LExpr *lookupVar(const char *name, const VarStack *vars, Environment env)
{
	unsigned int  i = 0;
	LExpr        *ret;
	
	// First try variables
	for (; vars != NULL; vars = vars->next, i++) {
		if (vars->name != NULL && streq(vars->name, name)) {
			ret = mkLExpr(L_VAR);
			ret->var = i;
			return ret;
		}
	}
	
	// Now try the environment
	Value *v = lookupEnv(env, name);
	if (v != NULL)
		return liftValue(v);
	
	error("Not in scope: %s", name);
}

LExpr *process(Expression *expr, const VarStack *vars, Environment env)
{
	LExpr *ret;
	
	switch (expr->tag) {
		case E_VAR:
			return lookupVar(expr->var, vars, env);
		
		case E_VALUE:
			return liftValue(expr->value);
		
		case E_LIST:
			{
				List *list = expr->list,
				     *i;
				
				for (i = list; i != NULL; i = i->next)
					i->item = process(i->item, vars, env);
				
				ret = mkLExpr(L_LIST);
				ret->list = list;
				return ret;
			}
		
		case E_LAMBDA:
			{
				VarStack v   = {vars, expr->lambda.var};
				LExpr *inner = process(expr->lambda.expr, &v, env);
				
				return mkLambda(inner);
			}
		
		case E_AP:
			return mkAp(process(expr->ap.f, vars, env), process(expr->ap.x, vars, env));
		
		case E_INFIX:
			{
				LExpr      *op = lookupVar(expr->infix.op, vars, env);
				Expression *a  = expr->infix.a,
				           *b  = expr->infix.b;
				
				// a + b => (+) a b
				if (a != NULL && b != NULL)
					return mkAp(mkAp(op, process(a, vars, env)), process(b, vars, env));
				
				// (a +) => (+) a
				if (a != NULL)
					return mkAp(op, process(a, vars, env));
				
				// (+ b) => \x. (+) x b
				if (b != NULL) {
					VarStack  vars_with_x = {vars, NULL};
					LExpr    *x           = mkVar(0);
					return mkLambda(mkAp(mkAp(op, x), process(b, &vars_with_x, env)));
				}
				
				// (+) => (+)
				return op;
			}
		
		case E_IF_THEN:
			ret = mkLExpr(L_IF_THEN);
			ret->if_then.pred     = process(expr->if_then.pred,     vars, env);
			ret->if_then.on_true  = process(expr->if_then.on_true,  vars, env);
			ret->if_then.on_false = process(expr->if_then.on_false, vars, env);
			return ret;
		
		default:
			error("Corrupt data structure passed to lexpr.c:process() %d", expr->tag);
	}
}

LExpr *makeLExpr(Expression *expr, Environment env)
{
	return process(expr, NULL, env);
}
