/*
 * Copyright (C) 2010 Joseph Adams <joeyadams3.14159@gmail.com>
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#include "lambda.h"

unsigned char cclass[256];

__attribute__((constructor))
void build_charclass(void)
{
	const unsigned char *s;
	int i;
	
	memset(cclass, 0, sizeof(cclass));
	
	for (s = (void*)"\t\n\v\f\r "; *s != 0; s++)
		cclass(*s) = C_SPACE;
	for (s = (void*)"!#$%&*+./<=>?@\\^|-~:"; *s != '\0'; s++)
		cclass(*s) = C_SYMBOL;
	for (i = '0'; i <= '9'; i++)
		cclass[i] = C_DIGIT | C_HEX;
	for (i = 'A'; i <= 'Z'; i++)
		cclass[i] = C_UPPER;
	for (i = 'a'; i <= 'z'; i++)
		cclass[i] = C_LOWER;
	
	for (i = 'A'; i <= 'F'; i++)
		cclass[i] |= C_HEX;
	for (i = 'a'; i <= 'f'; i++)
		cclass[i] |= C_HEX;
	
	for (i = 128; i <= 255; i++)
		cclass[i] = C_LOWER;
	
	
	// The choice of C_SYMBOL versus alphabetical is somewhat arbitrary
	// See http://en.wikipedia.org/wiki/TI_calculator_character_sets#TI-89.2F92_Series
	cclass[7] = C_SYMBOL;
	cclass[8] = C_SYMBOL;
	for (i = 14; i <= 31; i++)
		cclass[i] = C_SYMBOL;
	cclass[127] = C_SYMBOL;
	for (i = 128; i <= 255; i++)
		cclass[i] = C_LOWER;
	for (i = 156; i <= 168; i++)
		cclass[i] = C_SYMBOL;
	for (i = 171; i <= 172; i++)
		cclass[i] = C_SYMBOL;
	for (i = 174; i <= 177; i++)
		cclass[i] = C_SYMBOL;
	for (i = 182; i <= 184; i++)
		cclass[i] = C_SYMBOL;
	for (i = 187; i <= 191; i++)
		cclass[i] = C_SYMBOL;
	cclass[215] = C_SYMBOL;
	cclass[247] = C_SYMBOL;
}

void skipSpace(const char **sptr)
{
	const char *s = *sptr;
	
	while (is_space(*s))
		s++;
	
	*sptr = s;
}

bool startsWithKeyword(const char *str, const char *start)
{
	for (; *start != '\0'; str++, start++) {
		if (*str != *start)
			return false;
	}
	
	return !is_var_char(*str);
}

bool startsWithSymbol(const char *str, const char *start)
{
	for (; *start != '\0'; str++, start++) {
		if (*str != *start)
			return false;
	}
	
	return !is_symbol(*str);
}

/*
int: negativeSign? unsignedInt

negativeSign: '\xAD'

unsignedInt:
	('0x' | '0X') [0-9A-Fa-f]+
	('0c' | '0C') [0-7]+
	('0b' | '0B') [01]+
	[0-9]+
*/
long long parseInt(const char **sptr)
{
	const char *s        = *sptr;
	long long   ret      = 0;
	bool        negative = false;
	
	assert(is_int_start(*s));
	
	if (*s == '\xAD') {
		negative = true;
		s++;
		
		if (!is_digit(*s))
			error("Syntax error: no number after \xAD sign");
	}
	
	if (*s == '0') {
		s++;
		
		if (*s == 'B' || *s == 'b') {
			s++;
			if (!is_binary(*s))
				error("Syntax error: no digits after binary prefix 0%c", s[-1]);
			do {
				ret += ret;
				ret += *s - '0';
				s++;
			} while (is_binary(*s));
			
			goto success;
		}
		
		if (*s == 'C' || *s == 'c') {
			s++;
			if (!is_octal(*s))
				error("Syntax error: no digits after octal prefix 0%c", s[-1]);
			do {
				ret <<= 3;
				ret += *s - '0';
				s++;
			} while (is_octal(*s));
			
			goto success;
		}
		
		if (*s == 'X' || *s == 'x') {
			s++;
			if (!is_hex(*s))
				error("Syntax error: no digits after hex prefix 0%c", s[-1]);
			do {
				ret <<= 4;
				
				if (*s >= 'a')
					ret += *s - 'a' + 10;
				else if (*s >= 'A')
					ret += *s - 'A' + 10;
				else
					ret += *s - '0';
				
				s++;
			} while (is_hex(*s));
			
			goto success;
		}
	}
	
	while (is_digit(*s)) {
		ret *= 10;
		ret += *s - '0';
		s++;
	}
	
success:
	*sptr = s;
	return negative ? -ret : ret;
}

char *showInt(long long i, char buffer[22])
{
	char  tmp[22];
	char *o = buffer;
	char *t = tmp;
	
	unsigned long long u;
	
	if (i >= 0) {
		u = i;
	} else {
		u = -i;
		*o++ = 0xAD;
	}
	
	do {
		*t++ = '0' + u % 10;
		u /= 10;
	} while (u != 0);
	
	do {
		*o++ = *--t;
	} while (t > tmp);
	
	*o = 0;
	return buffer;
}

char *parseSymbol(const char **sptr)
{
	const char *s = *sptr;
	const char *p = s;
	char       *ret;
	
	assert(is_name_start(*p));
	
	if (is_symbol(*p))
		do p++; while (is_symbol(*p));
	else
		do p++; while (is_var_char(*p));
	
	ret = gc_strdup_len(s, p - s);
	
	{
		static const char * const reserved[] = {
			// "..",
			// "|",
			// "~",
			// "@",
			
			"=",
			"\\",
			"<-",
			"->",
			"let",
			"if",
			"then",
			"else",
			
			NULL
		};
		const char * const *r = reserved;
		
		for (; *r != NULL; r++)
			if (streq(ret, *r))
				error("Syntax error: expected variable name, but found reserved symbol '%s'", ret);
	}
	
	*sptr = p;
	return ret;
}


// when printing a string, be sure to add \& chars when necessary
void showChar1(char c, char quoteChar, char **out)
{
	char *s = *out;
	
	if (c == quoteChar)
		goto backslash;
	
	switch (c) {
		case '\a': c = 'a'; goto backslash;
		case '\b': c = 'b'; goto backslash;
		case '\f': c = 'f'; goto backslash;
		case '\n': c = 'n'; goto backslash;
		case '\r': c = 'r'; goto backslash;
		case '\t': c = 't'; goto backslash;
		case '\v': c = 'v'; goto backslash;
		case '\0': c = '0'; goto backslash;
		case '\\':          goto backslash;
		
		default:
			*s++ = c;
			*out = s;
			return;
	}
	
backslash:
	*s++ = '\\';
	*s++ = c;
	*out = s;
}

char *showChar(char c, char buffer[5])
{
	char *s = buffer;
	
	*s++ = '\'';
	showChar1(c, '\'', &s);
	*s++ = '\'';
	*s   = 0;
	return buffer;
}

// returns -1 on \& (which means "no character")
// not implemented: gaps, \NUL \SOH \STX ..., \^A \^@ \^[ ...
int parseChar1(const char **sptr)
{
	const char    *s = *sptr;
	char           c = *s++;
	unsigned char  n = 0;
	int            ret;
	
	if (c != '\\') {
		ret = c;
	} else {
		c = *s++;
		if (c == 0)
			error("Unexpected EOF in string/char literal");
		switch (c) {
			case 'a':  ret = '\a'; break;
			case 'b':  ret = '\b'; break;
			case 'f':  ret = '\f'; break;
			case 'n':  ret = '\n'; break;
			case 'r':  ret = '\r'; break;
			case 't':  ret = '\t'; break;
			case 'v':  ret = '\v'; break;
			
			case '\\': ret = '\\'; break;
			case '\"': ret = '\"'; break;
			case '\'': ret = '\''; break;
			
			case '&':  ret = -1;   break;
			
			case 'o':
				{
					if (!is_octal(*s))
						error("Expected octal after \\o");
					do {
						n <<= 3;
						n += *s - '0';
						s++;
					} while (is_octal(*s));
					
					ret = n;
				}
				break;
			
			case 'x':
				{
					if (!is_hex(*s))
						error("expected hex after \\x");
					
					do {
						n <<= 4;
						
						if (*s >= 'a')
							n += *s - 'a' + 10;
						else if (*s >= 'A')
							n += *s - 'A' + 10;
						else
							n += *s - '0';
						
						s++;
					} while (is_hex(*s));
					
					ret = n;
				}
				break;
			
			default:
				if (is_digit(c)) {
					s--;
					
					do {
						n *= 10;
						n += *s - '0';
						s++;
					} while (is_digit(*s));
					
					ret = n;
				} else {
					error("Unrecognized escape \\%c", c);
				}
		}
	}
	
	*sptr = s;
	return (unsigned char)ret;
}

Value *parseChar(const char **sptr)
{
	const char *s = *sptr;
	int         c;
	
	assert(*s == '\'');
	s++;
	
	if (*s == 0)
		error("Missing char terminator");
	if (*s == '\'')
		error("Empty char literal");
	
	c = parseChar1(&s);
	
	if (*s == 0)
		error("Missing char terminator");
	if (*s != '\'')
		error("Char literal has more than one character");
	if (c < 0)
		error("\\& not allowed in char literal");
	
	s++;
	
	*sptr = s;
	return mkCharValue(c);
}

Value *parseString(const char **sptr)
{
	const char *s    = *sptr;
	Value      *head = nil_v, **tail = &head;
	int         c;
	
	assert(*s == '\"');
	s++;
	
	while (*s != '\"') {
		if (*s == '\0')
			error("Missing string terminator");
		
		c = parseChar1(&s);
		if (c >= 0) {
			*tail = mkConsValue(mkCharValue(c), *tail);
			tail = &(*tail)->cons.xs;
		}
	}
	s++;
	
	*sptr = s;
	return head;
}