/*
 * Copyright (C) 2010 Joseph Adams <joeyadams3.14159@gmail.com>
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#ifndef LAMBDA_H
#define LAMBDA_H

#include "gc.h"
#include "avl.h"

// #define alloc(type) ((type*) gc_alloc_labeled(sizeof(type), "(" #type "*)"))
#define alloc(type) ((type*) gc_alloc(sizeof(type)))

typedef struct Expression Expression;
typedef struct LExpr LExpr;
typedef struct Value Value;

typedef struct ExprInfix  ExprInfix;
typedef struct ExprIfThen ExprIfThen;

#define List(a) List
typedef struct List List;

typedef struct Stack Stack;

typedef Map(char*, Value*) Environment;

typedef enum {E_VAR, E_VALUE, E_LIST, E_LAMBDA, E_AP, E_INFIX, E_INFIX_CLOSED, E_SECTION, E_IF_THEN} ExpressionTag;
typedef enum {L_VAR, L_VALUE, L_LIST, L_LAMBDA, L_AP, L_IF_THEN} LExprTag;

typedef enum {V_INT, V_CHAR, V_BOOL, V_UNIT, V_NIL, V_CONS, V_LAMBDA,
              V_UNARY, V_BINARY, V_BINARY1, V_NARY,
              V_IO, V_BIND,
              
              V_RETURN,
              
              V_UNDEFINED,
              V_INFINITE_LOOP,
              
              // thunk types
              V_DEFER_AP, V_DEFER_IF_THEN, V_DEFER_CALL,
              V_DEFER_TO
	           } ValueTag;

typedef enum {T_INT = 1, T_CHAR, T_BOOL, T_UNIT, T_LIST, T_FUNC, T_IO, T_RETURN} ValueType;

#define IS_THUNK(value) ((value)->tag >= V_DEFER_AP && (value)->tag <= V_DEFER_TO)

typedef Value *(*UnaryFunc)(Value *a);
typedef Value *(*BinaryFunc)(Value *a, Value *b);
typedef Value *(*NaryFunc)(List(Value) *vs);
typedef Value *(*IOFunc)(void *ctx);
typedef Value *(*ThunkFunc)(void *ctx);

#include "fixity.h"
#include "lex.h"

struct Expression {
	ExpressionTag tag;
	
	union {
		// E_VAR
		char *var;
		
		// E_VALUE
		Value *value;
		
		// E_LIST
		List(Expression) *list; // items are in reverse order
		
		// E_LAMBDA
		struct {
			char       *var;
			Expression *expr;
		} lambda;
		
		// E_AP (application)
		struct {
			Expression *f;
			Expression *x;
		} ap;
		
		// E_INFIX
		// E_INFIX_CLOSED (used internally by parser)
		// E_SECTION      (used internally by parser)
		struct {
			char         *op;
			const Fixity *fixity;
			
			// Either or both of the following may be NULL.
			Expression   *a;
			Expression   *b;
		} infix;
		
		// E_IF_THEN
		struct {
			Expression *pred;
			Expression *on_true;
			Expression *on_false;
		} if_then;
	};
	
	// used internally by parser
	Expression *parent;
};

// Post-processed, "low-level" expression
struct LExpr {
	LExprTag tag;
	
	union {
		// L_VAR
		unsigned int var;
		
		// L_VALUE
		Value *value;
		
		// L_LIST
		List(LExpr) *list; // items are in reverse order
		
		// L_LAMBDA
		LExpr *lambda;
		
		// L_AP
		struct {
			LExpr *f;
			LExpr *x;
		} ap;
		
		// L_IF_THEN
		struct {
			LExpr *pred;
			LExpr *on_true;
			LExpr *on_false;
		} if_then;
	};
};

struct Value {
	ValueTag tag;
	
	union {
		// V_INT
		long long i;
		
		// V_CHAR
		char c;
		
		// V_BOOL
		bool b;
		
		// V_UNIT
		
		// V_NIL
		
		// V_CONS
		struct {
			Value *x;
			Value *xs;
		} cons;
		
		// V_LAMBDA
		struct {
			LExpr *expr;
			Stack *stack;
		} lambda;
		
		// V_UNARY
		UnaryFunc unary;
		
		// V_BINARY
		// V_BINARY1 (binary operator with one value applied)
		struct {
			BinaryFunc  func;
			Value      *v1;
		} binary;
		
		// V_NARY
		struct {
			NaryFunc      func;
			List(Value)  *applied;
			unsigned int  remaining;
		} nary;
		
		// V_IO
		struct {
			IOFunc  func;
			void   *ctx;
		} io;
		
		// V_BIND
		struct {
			Value *m;
			Value *k;
		} bind;
		
		// V_RETURN
		Value *return_;
		
		// V_UNDEFINED
		// V_INFINITE_LOOP
		
		union {
			// V_DEFER_AP
			struct {
				Value *f;
				Value *x;
			};
			
			// V_DEFER_IF_THEN
			struct {
				Value *pred;
				Value *on_true;
				Value *on_false;
			};
			
			// V_DEFER_CALL
			struct {
				ThunkFunc  func;
				void      *ctx;
			};
			
			// V_DEFER_TO
			Value *to;
		} thunk;
	};
};

// Generic list
struct List {
	List *next;
	void *item;
};

struct Stack {
	Stack *next;
	Value *value;
};


// environment.c
static inline Environment addEnv(Environment env, const char *name, Value *value) {
	return avl_insert(env, name, value);
}
static inline Value *lookupEnv(Environment env, const char *name) {
	return avl_lookup(env, name);
}

Environment  addLet(Environment env, const char *name, const char *code);
Environment  runCommand(const char *cmd, Environment env);
Environment  import(const char *filename, Environment env, bool printError);


// parse.c
Expression *parseExpression(const char *s);


// lexpr.c
LExpr *makeLExpr(Expression *expr, Environment env); // destroys expr


// evaluate.c
Value *evaluate(Expression *expr, Environment env);
Value *evaluateLazy(Expression *expr, Environment env);
Value *eval(LExpr *expr, Stack *stack);
Value *force(Value *v);
Value *apply(Value *f, Value *x);

#define forceApply(f, x) force(apply(f, x))


// op.c
Environment newEnv(void);
extern Environment defaultEnv; // created by newEnv

extern Value *false_v, *true_v;
extern Value *fix_v, *foldr_v;
extern Value *step_v, *stepInf_v;
extern Value *unit_v, *nil_v;
extern Value *undefined_v, *infinite_loop_v;

extern Value *bindFunc_v; // \m \k \x k (m x) x


// prompt.c
const char *prompt(const char *start, bool use_history, Environment env);
void        add_history(const char *str);

extern Map(long, char*) bindings;

void add_binding(long key, char *string);
static inline void remove_binding(long key)
{
	bindings = avl_remove(bindings, (void*)key);
}

// Buffers used by prompt() that can also be used outside for other things.
// Note that prompt() returns prompt_buffer on success, so exercise care.
extern char prompt_buffer[1024],
            prompt_cur_buffer[1024];


// menu.c
const char *envMenu(Environment env);

// misc.c
Value             *perform(Value *v);
void               print(Value *v);
void               printValueDebug(Value *v);
void               printLExpr(LExpr *expr);
const char        *typeStr(Value *v);
ValueType          getType(Value *v);
void               readString(Value *str, char *out, size_t max);
Value             *buildString(const char *str);
void               chomp(char *buffer);

static inline void printLn(Value *v)
{
	print(v);
	putchar('\n');
}


/*** Data constructors ***/

static inline Expression *mkExpression(ExpressionTag tag)
{
	Expression *expr = alloc(Expression);
	expr->tag = tag;
	return expr;
}

static inline LExpr *mkLExpr(LExprTag tag)
{
	LExpr *lexpr = alloc(LExpr);
	lexpr->tag = tag;
	return lexpr;
}

static inline Value *mkValue(ValueTag tag)
{
	Value *ret = alloc(Value);
	ret->tag = tag;
	return ret;
}

static inline List *listCons(void *x, List *xs)
{
	List *ret = alloc(List);
	ret->item = x;
	ret->next = xs;
	return ret;
}

static inline Value *mkIntValue(long long i)
{
	Value *ret = mkValue(V_INT);
	ret->i = i;
	return ret;
}

static inline Value *mkConsValue(Value *x, Value *xs)
{
	Value *ret = mkValue(V_CONS);
	ret->cons.x = x;
	ret->cons.xs = xs;
	return ret;
}

static inline Value *mkLambdaValue(LExpr *expr, Stack *stack)
{
	Value *ret = mkValue(V_LAMBDA);
	ret->lambda.expr = expr;
	ret->lambda.stack = stack;
	return ret;
}

static inline Value *mkIOValue(IOFunc func, void *ctx)
{
	Value *ret = mkValue(V_IO);
	ret->io.func = func;
	ret->io.ctx = ctx;
	return ret;
}

static inline Value *mkBindValue(Value *m, Value *k)
{
	Value *ret = mkValue(V_BIND);
	ret->bind.m = m;
	ret->bind.k = k;
	return ret;
}

static inline Value *mkCharValue(char c)
{
	Value *ret = mkValue(V_CHAR);
	ret->c = c;
	return ret;
}

static inline Value *mkBoolValue(bool b)
{
	Value *ret = mkValue(V_BOOL);
	ret->b = b;
	return ret;
}

static inline Value *mkUnit(void)
{
	Value *ret = mkValue(V_UNIT);
	return ret;
}

static inline Value *deferApply(Value *f, Value *x)
{
	Value *ret = mkValue(V_DEFER_AP);
	ret->thunk.f    = f;
	ret->thunk.x    = x;
	return ret;
}

static inline Value *deferIfThen(Value *pred, Value *on_true, Value *on_false)
{
	Value *ret = mkValue(V_DEFER_IF_THEN);
	ret->thunk.pred     = pred;
	ret->thunk.on_true  = on_true;
	ret->thunk.on_false = on_false;
	return ret;
}

/*
 * If a thunk's context is stateful, it must not let the following happen
 * in its critical section (between updating its state and returning its value):
 *  - Exception (other than ER_MEMORY, which is fatal)
 *  - Getting forced again.  A stateful thunk should avoid forcing
 *    any values in its critical section, as those values might point right back
 *    to the thunk still being forced.
 *
 * If one of those is allowed to happen, trying to force the thunk again will skip over
 * the value it was supposed to originally produce.
 *
 * Note that deferApply and deferIfThen can waive these restrictions, as their
 * values are supposed to be pure.  If they aren't (due to unsafePerformIO),
 * the state smear is to be expected :-)
 */
static inline Value *deferCall(ThunkFunc func, void *ctx)
{
	Value *ret = mkValue(V_DEFER_CALL);
	ret->thunk.func = func;
	ret->thunk.ctx  = ctx;
	return ret;
}


/*** Value identification ***/

// is_list: see "List management"

static inline bool is_func(Value *v)
{
	switch (getType(v)) {
		case T_FUNC:
		case T_RETURN:
			return true;
		
		default:
			return false;
	}
}


/*** List management ***/

/*
 * The call semantics of the list functions are a bit confusing:
 *
 *   Before using a list, call is_list() and throw a friendly error if it's not a list.
 *   Otherwise, it'll accuse the user of misusing cons when the user didn't pass a list at all.
 *
 *   That is, call is_list() before using either of:
 *    - list_foreach()
 *    - list_head()
 *    - list_pop()
 *
 *   Only call list_tail() on a value where list_head() returns non-NULL.
 *   list_tail() does not force the cons, list_head() does.
 */

#define INVALID_CONS() error("Invalid use of cons: right-hand argument must be a list")

// call is_list before using (unless you already know it's a list)
#define list_foreach(i, x) \
	for (; ((x) = list_head(i)) != NULL; i = list_tail(i))

#define is_list(v) is_list_(force(v))

static inline bool is_list_(Value *v)
{
	switch (v->tag) {
		case V_NIL:
		case V_CONS:
		case V_RETURN:
			return true;
		
		default:
			return false;
	}
}

static inline Value *list_head(Value *list)
{
	switch (force(list)->tag) {
		case V_NIL:
			return NULL;
		
		case V_CONS:
			return list->cons.x;
		
		case V_RETURN:
			return list->return_;
		
		default:
			// error("bug: list_head() called on %s", typeStr(list));
			INVALID_CONS();
	}
}

static inline Value *list_tail(Value *list)
{
	switch (list->tag) {
		case V_NIL:
			error("bug: list_tail() called on []");
		
		case V_CONS:
			list = list->cons.xs;
			
			// Don't do this, as it can force the tail before it's needed,
			// causing `let xs = 1 : map (+1) xs; xs` to stack overflow
			// if (!is_list(list))
			// 	INVALID_CONS();
			
			return list;
		
		case V_RETURN:
			return nil_v;
		
		default:
			error("bug: list_tail() called on %s", typeStr(list));
	}
}

static inline Value *list_pop(Value **list)
{
	Value *x = list_head(*list);
	if (x != NULL)
		*list = list_tail(*list);
	return x;
}

#endif
