/*
 * Copyright (C) 2010 Joseph Adams <joeyadams3.14159@gmail.com>
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#include <alloc.h>
#include <setjmp.h>
#include <statline.h>
#include <stdlib.h>

#define get_stack_ptr(out) asm("movea.l %%sp, %0" : "=a" (out))
extern void __ld_bss_start;
extern void __ld_bss_size;
#define __ld_bss_start (&__ld_bss_start)
#define __ld_bss_size ((unsigned long) &__ld_bss_size)

#define HEAP_START   ((char*) 0x5B00)
#define HEAP_END     ((char*) 0x40000)
#define IN_HEAP(ptr) ((char*)(ptr) >= HEAP_START && (char*)(ptr) < HEAP_END)

#define IN_STACK(ptr) ((char*)(ptr) >= (char*)0x400 && (char*)(ptr) < (char*)0x4200)
#define IN_BSS(ptr)   ((char*)(ptr) >= (char*)__ld_bss_start && (char*)(ptr) < (char*)__ld_bss_start + __ld_bss_size)

#define MIN_PAGE_SIZE 24
	// enough for a 12-byte block followed by an 8-byte sentinel block,
	// plus 4 more bytes in case the handle's pointer is not 4-byte-aligned.
#define MAX_PAGE_SIZE (MAX_SIZE + 12)
	// enough to hold a block of MAX_SIZE bytes of user data,
	// plus a header for that block and space for a sentinel block.

#define PAGE_MAX  16
#define ROOT_MAX  16
#define DEFER_MAX 4096

#define JUMP_TABLE_SHIFT 6
#define JUMP_TABLE_COUNT (262144 >> JUMP_TABLE_SHIFT)

/*
 * gc.free_lists[n] holds blocks of size (n*4 + 12)
 *
 * Except for the last one (when n == FREE_LIST_COUNT-1).
 * The last one holds all blocks too big to go in the other lists.
 */
#define FREE_LIST_COUNT 8
#define FREE_LIST_INDEX(size) min((unsigned short)(((size) >> 2) - 3), (unsigned short)(FREE_LIST_COUNT - 1))


/*
 * A "near pointer" crams a pointer to an object on the heap into a 16-bit value.
 * The pointer must be 4-byte aligned.
 *
 * This trick works because the heap ends at 0x3FFFF, and 0x3FFFF/4 is 0xFFFF.
 */
typedef unsigned short NearPtr;
#define toNear(ptr)    ((NearPtr)((unsigned long)(ptr) >> 2))
#define fromNear(near) ((void*)((unsigned long)(near) << 2))

typedef struct Block Block;
typedef struct Page Page;
typedef struct GCRange GCRange;
typedef struct GC GC;

#define BLOCK_HEADER_SIZE 4

struct Block {
	/*
	 * size of block, including the 4 bytes the block header occupies.
	 * if size == 0, this is a sentinel block, and nextPage becomes relevant.
	 */
	unsigned short    size;
	
	signed   char     type;
	
	union {
		unsigned char   bits;
		struct {
			unsigned char bit7:1, bit6:1, bit5:1, // unused bits
			              in_free_list:1,         // only used by gc_check_invariants()
			              
			              used:1,
			              marked:1,
			              need_to_scan:1, // only set if there was no room in the defer stack
			              dont_gc:1;
		};
	};
	
	union {
		char data[1];
		
		// only if current block is free
		Block *nextFree;
		
		// if size == 0, this "block" simply serves to point to the next page.
		// Pages are kept in order of memory position.
		//
		// if size == 0 and nextPage == NULL, there are no more blocks left.
		Block *nextPage;
	};
};

#define in_block(block, ptr) ((char*)(ptr) >= (block)->data && (char*)(ptr) < (char*)(block) + (block)->size)

static inline Block *next_block(Block *b)
{
	b = (Block*)((char*)b + b->size);
	if (b->size == 0)
		b = b->nextPage;
	return b;
}

struct Page {
	Block          *first;
	unsigned short  size;
	Block          *sentinel; // either links to next page, or terminates the entire block list.
	HANDLE          handle;
};

struct GCRange {
	void   *base;
	size_t  size;
};

struct GC {
	Block        *free_lists[FREE_LIST_COUNT];
	
	Page          pages[PAGE_MAX];
	unsigned int  page_count;
	
	GCRange       roots[ROOT_MAX];
	unsigned int  root_count;
	void         *stack_base;
	
	NearPtr       defer[DEFER_MAX]; // Blocks that have been marked, but still need to be scanned.
	unsigned int  defer_count;
	bool          need_to_scan;     // If the defer stack maxes out, this is set, and the blocks are traversed all over again.
	
	NearPtr       jump_table[JUMP_TABLE_COUNT];
	
	bool          collecting;
	
	/* Stats gathered by performGC() */
	unsigned long bytes_freed,
	              bytes_retained,
	              bytes_total;
};

extern GC gc_state;

Block *gc_alloc_(unsigned short size, BlockType type);

static void gc_init(void);
static void gc_finish(void);

static Block *claim(Block *b, unsigned short request, BlockType type);

static bool add_page(void);
static bool alloc_page(Page *page);

static void add_to_free_list(Block *b);
static Block *get_from_free_list(unsigned short size);

static Block *lookup(void *ptr);

static void mark(void *ptr);
static void defer_scan(Block *b);
