/*
 * Copyright (C) 2010 Joseph Adams <joeyadams3.14159@gmail.com>
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#ifndef GC_H
#define GC_H

#include <kbd.h>
#include "common.h"

/*
 * Largest size of block you can allocate with gc_alloc.
 *
 * Note that:
 *  - It corresponds directly to how big the GC (tries to) allocates its pages.
 *  - It must be <= 65520 so block sizes can be stored in an unsigned short.
 *  - The chosen value is rather arbitrary.
 */
#define MAX_SIZE 16366

/*
 * Number of bytes to reserve for non-GC use.
 *
 * When the GC allocates a page of memory, nobody else can use it
 * until the end of the program.  RESERVE_SIZE prevents the GC
 * from allocating all available memory, as that would impact file IO,
 * screen buffering, etc.
 */
#define RESERVE_SIZE 15000

extern volatile bool memoryError_reported;

#define memoryError(fmt, ...) do { \
        printf(fmt "\n", ##__VA_ARGS__); \
        ngetchx(); \
        memoryError_reported = true; \
        ER_throw(ER_MEMORY); \
    } while(0)

/*
 * To add your own finalizers, add entries to this enum,
 * implement the finalizers in gc/hooks.h, and allocate
 * blocks with gc_alloc_type.
 *
 * For blocks that never point to any GC-allocated blocks
 * (example: strings), use GC_ATOMIC so they won't get scanned.
 */
typedef enum {
	GC_ATOMIC = 0,
	GC_NORMAL = 1,
	
	GC_READFILE = 2,
	
	GC_LABELED = 57
} BlockType;

#define  gc_alloc(size)        gc_alloc_type(size, GC_NORMAL)
#define  gc_alloc_atomic(size) gc_alloc_type(size, GC_ATOMIC)

void    *gc_alloc_type(size_t size, BlockType type);

void *gc_alloc_labeled(size_t size, const char *label);
char *gc_show_ptr(void *ptr, char buffer[64]);

#define gc_free(ptr)             gc_free_(ptr, true)
#define gc_free_no_finalize(ptr) gc_free_(ptr, false)
void gc_free_(void *ptr, bool run_finalizer);

char *gc_strdup_len(const char *str, size_t len);
static inline char *gc_strdup(const char *str)
{
	return gc_strdup_len(str, strlen(str));
}

void performGC(void);
void gc_check_invariants(void);

void gc_main(void func(void));

#define GC_VERBOSE_COLLECTING (1 << 0) // print "Garbage collecting..."
#define GC_VERBOSE_BORDER     (1 << 1) // draw a text border around GC output
#define GC_VERBOSE_STATS      (1 << 2) // print number of bytes freed and retained
#define GC_VERBOSE_PAUSE      (1 << 3) // ngetchx()
#define GC_VERBOSE_RANGES     (1 << 4) // print hexadecimal ranges of roots that are scanned
#define GC_VERBOSE_TASKS      (1 << 5) // print "Building jump table...", "Marking...", and "Sweeping..."
#define GC_VERBOSE_FINALIZE   (1 << 6) // print finalizer messages

#define GC_VERBOSE_NEED_TO_SCAN (1 << 7)

#define GC_VERBOSE_TERSE      (1 << 8) // print garbage collector stats in the status line

#define GC_VERBOSE_0 0
#define GC_VERBOSE_1 GC_VERBOSE_TERSE
#define GC_VERBOSE_2 (GC_VERBOSE_1 | GC_VERBOSE_COLLECTING | GC_VERBOSE_FINALIZE)
#define GC_VERBOSE_3 (GC_VERBOSE_2 | GC_VERBOSE_BORDER | GC_VERBOSE_STATS)
#define GC_VERBOSE_4 (GC_VERBOSE_3 | GC_VERBOSE_TASKS | GC_VERBOSE_NEED_TO_SCAN)
#define GC_VERBOSE_5 (GC_VERBOSE_4 | GC_VERBOSE_RANGES)
#define GC_VERBOSE_6 (GC_VERBOSE_5 | GC_VERBOSE_PAUSE)

extern unsigned int gc_verbose;

#define gc_set_verbose_level(level) do { \
		gc_verbose = gc_verbose_level(level); \
	} while (0);

static inline unsigned int gc_verbose_level(int level)
{
	if (level < 0)
		return 0;
	
	switch (level) {
		case 0:  return GC_VERBOSE_0;
		case 1:  return GC_VERBOSE_1;
		case 2:  return GC_VERBOSE_2;
		case 3:  return GC_VERBOSE_3;
		case 4:  return GC_VERBOSE_4;
		case 5:  return GC_VERBOSE_5;
		case 6:  return GC_VERBOSE_6;
		default: return GC_VERBOSE_6;
	}
}

#endif
