/*
 * Copyright (C) 2010 Joseph Adams <joeyadams3.14159@gmail.com>
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#include "fixity.h"
#include "lambda.h"

static const Fixity default_var_fixity =
	{"",        PREFIX, 0};

static const Fixity default_op_fixity =
	{"",        INFIXL, 9};

static const Fixity fixity_tbl[] = {
	{"!!",      INFIXL, 9},
	{".",       INFIXR, 9}, {"\xb7", INFIXR, 9}, 
	{"^",       INFIXR, 8},
	{"^^",      INFIXR, 8},
	{"**",      INFIXR, 8},
	{"*",       INFIXL, 7},
	{"/",       INFIXL, 7},
	{"div",     INFIXL, 7},
	{"mod",     INFIXL, 7},
	{"quot",    INFIXL, 7},
	{"rem",     INFIXL, 7},
	{"+",       INFIXL, 6},
	{"-",       INFIXL, 6},
		// The negate symbol on the TI-89 is "\xAD", but that's a different story
	{":",       INFIXR, 5},
	{"++",      INFIXR, 5},
	{"~",       INFIX,  4}, // typeEq
	{"==",      INFIX,  4},
	{"/=",      INFIX,  4}, {"\x9d",  INFIX, 4},
	{"<",       INFIX,  4},
	{"<=",      INFIX,  4}, {"\x9c",  INFIX, 4},
	{">",       INFIX,  4},
	{">=",      INFIX,  4}, {"\x9e",  INFIX, 4},
	{"\x1f",    INFIX,  4}, // elem
	{"/\x1f",   INFIX,  4}, // notElem
	{"&&",      INFIXR, 3},
	{"||",      INFIXR, 2},
	{">>",      INFIXL, 1},
	{">>=",     INFIXL, 1},
	{"..",      INFIX,  1},
	{"...",     INFIX,  1}, {"\xa0", INFIX,  1},
	{"$",       INFIXR, 0},
	{"$!",      INFIXR, 0},
	// {"seq",     INFIXR, 0},
	
	{"</>",     INFIXR, 5},
	{"<$>",     INFIXL, 4},
	{"<$",      INFIXL, 4},
	{"<*>",     INFIXL, 4},
	{"<*",      INFIXL, 4},
	{"*>",      INFIXL, 4},
	{"<**>",    INFIXL, 4},
	{"<|>",     INFIXL, 3},
	{"=<<",     INFIXR, 1},
	
	{".<<.",    INFIXL, 8},
	{".>>.",    INFIXL, 8},
	{".&.",     INFIXL, 7},
	{".^.",     INFIXL, 6},
	{".|.",     INFIXL, 5}
};

const FixityList *customFixities;

const Fixity *getFixity(const char *name)
{
	const FixityList *cell;
	unsigned int      i;
	
	assert(is_name_start(*name));
	
	for (cell = customFixities; cell != NULL; cell = cell->next) {
		if (streq(name, cell->f.name))
			return &cell->f;
	}
	
	for (i = 0; i < sizeof(fixity_tbl) / sizeof(*fixity_tbl); i++) {
		if (streq(name, fixity_tbl[i].name))
			return &fixity_tbl[i];
	}
	
	if (is_symbol(*name))
		return &default_op_fixity;
	else
		return &default_var_fixity;
}

int compareFixity(const Fixity *a, const Fixity *b)
{
	assert(a->fixity != PREFIX && b->fixity != PREFIX);
	
	if (a->precedence != b->precedence)
		return a->precedence - b->precedence;
	
	if (a->fixity == INFIXL && b->fixity == INFIXL)
		return 1;
	
	if (a->fixity == INFIXR && b->fixity == INFIXR)
		return -1;
	
	return 0;
}

void setFixity(const char *name, FixityTag fixity, int precedence)
{
	FixityList *cell = alloc(FixityList);
	cell->next = customFixities;
	cell->f.name = name;
	cell->f.fixity = fixity;
	cell->f.precedence = precedence;
	
	customFixities = cell;
}