/*
 * Copyright (C) 2010 Joseph Adams <joeyadams3.14159@gmail.com>
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#include "lambda.h"

Value *evaluate(Expression *expr, Environment env)
{
	return force(eval(makeLExpr(expr, env), NULL));
}

Value *evaluateLazy(Expression *expr, Environment env)
{
	return eval(makeLExpr(expr, env), NULL);
}

Value *force(Value *v)
{
	while (IS_THUNK(v)) {
		Value *v2 = NULL;
		
		switch (v->tag) {
			case V_DEFER_TO:
				v2 = v->thunk.to;
				break;
			
			case V_DEFER_AP:
				v2 = apply(v->thunk.f, v->thunk.x);
				break;
			
			case V_DEFER_IF_THEN:
				if (force(v->thunk.pred)->tag != V_BOOL)
					error("if/then predicate must be a Bool");
				
				v2 = v->thunk.pred->b ? v->thunk.on_true : v->thunk.on_false;
				break;
			
			case V_DEFER_CALL:
				v2 = v->thunk.func(v->thunk.ctx);
				break;
			
			default:
				error("Corrupt data structure passed to force() %d", v->tag);
		}
		
		if (v != v2) {
			*v = *v2;
			
			if (IS_THUNK(v)) {
				memset(v2, 0, sizeof(*v2));
				v2->tag = V_DEFER_TO;
				v2->thunk.to = v;
			}
		}
	}
	
	if (v->tag == V_UNDEFINED)
		error("undefined");
	else if (v->tag == V_INFINITE_LOOP)
		error("infinite loop");
	
	return v;
}

static Stack *push(Stack *s, Value *v)
{
	Stack *ret = alloc(Stack);
	ret->next = s;
	ret->value = v;
	return ret;
}

Value *eval(LExpr *expr, Stack *stack)
{
	switch (expr->tag) {
		case L_VAR:
			{
				unsigned int i = expr->var;
				
				while (i-- != 0)
					stack = stack->next;
				
				return stack->value;
			}
		
		case L_VALUE:
			return expr->value;
		
		case L_LIST:
			{
				Value       *xs = nil_v;
				List(LExpr) *i  = expr->list;
				
				for (i = expr->list; i != NULL; i = i->next)
					xs = mkConsValue(eval(i->item, stack), xs);
				
				return xs;
			}
		
		case L_LAMBDA:
			return mkLambdaValue(expr->lambda, stack);
		
		case L_AP:
			return deferApply(eval(expr->ap.f, stack),
			                  eval(expr->ap.x, stack));
		
		case L_IF_THEN:
			return deferIfThen(eval(expr->if_then.pred,     stack),
			                   eval(expr->if_then.on_true,  stack),
			                   eval(expr->if_then.on_false, stack));
		
		default:
			error("Corrupt data structure passed to eval()");
	}
}

Value *apply(Value *f, Value *x)
{
	Value *v;
	
	check_break();
	
	switch (force(f)->tag) {
		case V_LAMBDA:
			return eval(f->lambda.expr, push(f->lambda.stack, x));
		
		case V_UNARY:
			return f->unary(x);
		
		case V_BINARY:
			v = mkValue(V_BINARY1);
			v->binary.func = f->binary.func;
			v->binary.v1 = x;
			return v;
			
		case V_BINARY1:
			return f->binary.func(f->binary.v1, x);
		
		case V_NARY:
			{
				List(Value) *vs = listCons(x, f->nary.applied);
				
				if (f->nary.remaining > 1) {
					v = mkValue(V_NARY);
					v->nary.func      = f->nary.func;
					v->nary.applied   = vs;
					v->nary.remaining = f->nary.remaining - 1;
					return v;
				} else {
					return f->nary.func(vs);
				}
			}
		
		case V_RETURN:
			return f->return_;
		
		default:
			error("Cannot apply argument to a non-function");
	}
}