/*
 * Copyright (C) 2010 Joseph Adams <joeyadams3.14159@gmail.com>
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#include "lambda.h"

#include <kbd.h>

Environment addLet(Environment env, const char *name, const char *code)
{
	Expression *expr = parseExpression(code);
	Value *v         = mkValue(V_INFINITE_LOOP);
	
	env = addEnv(env, name, v);
	*v = *evaluateLazy(expr, env);
	
	return env;
}

static void runFixity(FixityTag tag, const char *s)
{
	int   prec = 0;
	char *name;
	
	skipSpace(&s);
	
	if (tag != PREFIX) {
		if (!is_digit(*s))
			goto syntax_error;
		
		do {
			prec *= 10;
			prec += *s - '0';
			s++;
		} while (is_digit(*s));
		
		if (!is_space(*s))
			goto syntax_error;
		skipSpace(&s);
	}
	
	if (!is_name_start(*s))
		goto syntax_error;
	name = parseSymbol(&s);
	
	skipSpace(&s);
	if (*s != 0)
		goto syntax_error;
	
	if (!(prec >= 0 && prec <= 9))
		error("Precedence out of range");
	
	setFixity(name, tag, prec);
	return;

syntax_error:
	error("Syntax error in fixity declaration");
}

// x <- action
static Environment runBind(const char *s, Environment env)
{
	skipSpace(&s);
	if (!is_name_start(*s))
		return NULL;
	
	char * volatile name;
	
	TRY
		suppressError = true;
		name = parseSymbol(&s);
	ONERR
		name = NULL;
	ENDTRY
	
	suppressError = false;
	
	if (name == NULL)
		return NULL;
	
	skipSpace(&s);
	if (startsWithSymbol(s, "\x15"))
		s++;
	else if (startsWithSymbol(s, "<-"))
		s += 2;
	else
		return NULL;
	
	Value *a = evaluate(parseExpression(s), env);
	
	return addEnv(env, name, perform(a));
}

static bool break_caught;

Environment runCommand(const char *cmd, Environment env)
{
	TRY
		const char *s = cmd;
		
		skipSpace(&s);
		
		if (*s == '\0' || (unsigned char)*s == 169) // 169 is the comment (i.e. ) character
			goto done;
		
		const char *orig_s;
		orig_s = s = gc_strdup(s);
		
		const char *w = s;
		while (*w != '\0' && !is_space(*w))
			w++;
		
		if (*w == '\0')
			w = s;
		
		if (w - s == 3 && strncmp(s, "let", 3) == 0) {
			s = w;
			skipSpace(&s);
			
			if (!is_name_start(*s))
				error("Syntax error: expected variable name after let");
			
			char *var = parseSymbol(&s);
			
			skipSpace(&s);
			if (*s != '=')
				error("Syntax error: expected =");
			s++;
			
			env = addLet(env, var, s);
			
		} else if (w - s == 6 && strncmp(s, "infixl", 6) == 0) {
			runFixity(INFIXL, w);
		} else if (w - s == 6 && strncmp(s, "infixr", 6) == 0) {
			runFixity(INFIXR, w);
		} else if (w - s == 5 && strncmp(s, "infix", 5) == 0) {
			runFixity(INFIX, w);
		} else if (w - s == 6 && strncmp(s, "prefix", 6) == 0) {
			runFixity(PREFIX, w);
		} else if (w - s == 6 && strncmp(s, "import", 6) == 0) {
			s = w;
			skipSpace(&s);
			env = import(s, env, true);
		} else {
			TRY
				enable_break();
				
				Environment tmp = runBind(s, env);
				
				if (tmp) {
					env = tmp;
				} else {
					Value *v = evaluate(parseExpression(orig_s), env);
					
					if (getType(v) == T_IO) {
						v = perform(v);
						
						if (force(v)->tag != V_UNIT)
							printLn(v);
					} else if (v->tag == V_RETURN && force(v->return_)->tag == V_UNIT) {
						// do nothing
					} else {
						printLn(v);
					}
					
					env = addEnv(env, "it", v);
				}
			
			FINALLY
				disable_break();
			ENDFINAL
		}
		
	done: ;
		
	ONERR
		
		if (errCode == ER_BREAK) {
			printf("\nBreak.\n");
			GKeyFlush();
			break_caught = true;
		} else if (errCode == ER_NO_MSG) {
			/* nothing */
		} else {
			PASS;
		}
		
	ENDTRY
	
	return env;
}

static bool fgets_indent(char *buffer, short n, FILE *f)
{
	bool hasContent = false;
	
	while (n > 1) {
		if (!fgets(buffer, n, f))
			break;
		hasContent = true;
		
		// Make buffer point to end, and subtract from n accordingly
		{
			char *p = buffer;
			while (*p != '\0') p++;
			n -= p - buffer;
			buffer = p;
		}
		
		// Peek at the next character
		char c = getc(f);
		if (c == EOF)
			break;
		ungetc(c, f);
		
		// If it's whitespace, keep on reading
		if (!is_space(c) || c == '\r')
			break;
	}
	
	return hasContent;
}

Environment import(const char *filename, Environment env, bool printError)
{
	FILE *f = fopen(filename, "rt");
	char buffer[1024];
	
	if (f == NULL) {
		if (printError)
			printf("Could not open '%s'\n", filename);
		return env;
	}
	
	break_caught = false;
	
	while (fgets_indent(buffer, sizeof(buffer), f)) {
		chomp(buffer);
		env = runCommand(buffer, env);
		if (break_pressed)
			break_caught = true;
		if (break_caught)
			break;
	}
	
	fclose(f);
	return env;
}