/*
   	EightPuzzle v1.0
    Copyright (C) 2005 Jonas Gehring
        
    Eight Puzzle is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    Eight Puzzle is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Eight Puzzle; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

// C Source File
// Created 15.12.04; 00:50:22


#include <alloc.h>															// Routines for dynamic memory allocation
#include <asmtypes.h>														// Common type definitions known to ASM programmers
#include <compat.h>															// Various calc-dependent and AMS-dependent pseudo-constants
#include <graph.h>															// Common (non-windowed) graphic routines
#include <intr.h>																// Routines for creating interrupt handlers
#include <kbd.h>																// Keyboard handling routines
#include <statline.h>														// Routines for manipulation of the status line
#include <stdlib.h>															// Some useful routines proposed by ANSI (sorting, searching, etc.)
#include <system.h>															// System routines (for accessing system timers, queues, etc.)

#include "extgraph.h"														// Include ExtGraph v2.00beta4 by TI-Chess Team


enum directions {LEFT, RIGHT, UP, DOWN};				// Different directions to move

// These pseudo constants make sure that the screen is centered on a TI92+ and V200
#define CALCY ((LCD_HEIGHT-100)/2)
#define CALCX ((LCD_WIDTH-160)/2)

#define TILEHEIGHT 16
#define LOGOHEIGHT 32

// Macro to get the address of a tile in the array tile_array
#define TILEADDR(nr, p) &(tile_array[(nr*TILEHEIGHT*2)+(p*TILEHEIGHT)])
// Macro to get the address of a tile in the array logo
#define LOGOADDR(nr,p) &(logo[(nr*LOGOHEIGHT*2)+(p*LOGOHEIGHT)])

// Tile array
unsigned short tile_array[2*TILEHEIGHT*10] = {
	// Cursor
	0x7FFF,0x4001,0x4001,0x4001,0x4001,0x4001,0x4001,0x4001,
	0x4001,0x4001,0x4001,0x4001,0x4001,0x4001,0x4001,0x7FFF,
	0x7FFF,0x7FFF,0x6003,0x6003,0x6003,0x6003,0x6003,0x6003,
	0x6003,0x6003,0x6003,0x6003,0x6003,0x6003,0x7FFF,0x7FFF,	
	// "1"
	0x0000,0x0000,0x0020,0x00E0,0x03E0,0x03C0,0x00C0,0x01C0,
	0x01C0,0x0180,0x0180,0x0380,0x0700,0x1FE0,0x0000,0x0000,
	0x0000,0x0000,0x0050,0x00E0,0x01C0,0x04E0,0x01C0,0x01C0,
	0x0180,0x03C0,0x0380,0x0380,0x0380,0x1FE0,0x0000,0x0000,
	// "2"
	0x0000,0x0000,0x01E0,0x0360,0x0270,0x0070,0x0060,0x00E0,
	0x0180,0x0100,0x0200,0x0420,0x0FC0,0x1FC0,0x0000,0x0000,
	0x0000,0x0000,0x01C0,0x00F0,0x0470,0x0070,0x00F0,0x00C0,
	0x0040,0x0280,0x0400,0x0840,0x0FE0,0x1FC0,0x0000,0x0000,
	// "3"
	0x0000,0x0000,0x01E0,0x0270,0x0070,0x0060,0x00C0,0x01E0,
	0x0070,0x0070,0x0060,0x00E0,0x19C0,0x0F00,0x0000,0x0000,
	0x0000,0x0000,0x00F0,0x0570,0x0070,0x00E0,0x00C0,0x03E0,
	0x00F0,0x0070,0x0070,0x0060,0x1CC0,0x1E80,0x0000,0x0000,
	// "4"
	0x0000,0x0000,0x0030,0x00F0,0x00F0,0x01E0,0x03E0,0x0660,
	0x04E0,0x0CC0,0x1FE0,0x00C0,0x01C0,0x01C0,0x0000,0x0000,
	0x0000,0x0000,0x0070,0x0070,0x01F0,0x03F0,0x0370,0x06E0,
	0x0AE0,0x1CE0,0x1FF0,0x01C0,0x01C0,0x01C0,0x0000,0x0000,
	// "5"
	0x0000,0x0010,0x03F0,0x03F0,0x0200,0x0600,0x0380,0x0180,
	0x01C0,0x01C0,0x0180,0x0180,0x3700,0x1E00,0x0000,0x0000,
	0x0000,0x0008,0x03F8,0x07F0,0x0400,0x0700,0x0780,0x03C0,
	0x01C0,0x01C0,0x01C0,0x0380,0x3380,0x3C00,0x0000,0x0000,
	// "6"
	0x0000,0x0000,0x0070,0x01C0,0x0300,0x0600,0x0FC0,0x0CC0,
	0x0CE0,0x1CE0,0x1CE0,0x1CC0,0x0CC0,0x0780,0x0000,0x0000,
	0x0000,0x0000,0x00B8,0x0380,0x0700,0x0E00,0x0D80,0x1FE0,
	0x1CE0,0x1CE0,0x1CE0,0x1CE0,0x1FC0,0x0F00,0x0000,0x0000,
	// "7"
	0x0000,0x0000,0x03F8,0x07E0,0x0400,0x0000,0x0000,0x0100,
	0x0100,0x0200,0x0600,0x0600,0x0C00,0x0C00,0x0000,0x0000,
	0x0000,0x0000,0x07F0,0x07F0,0x0A00,0x0000,0x00C0,0x0080,
	0x0200,0x0100,0x0600,0x0800,0x0E00,0x0E00,0x0000,0x0000,
	// "8"
	0x0000,0x0000,0x01E0,0x0330,0x0730,0x0760,0x03C0,0x0380,
	0x07C0,0x0CE0,0x0CE0,0x1CC0,0x1DC0,0x0F80,0x0000,0x0000,
	0x0000,0x0000,0x03E0,0x07F0,0x0730,0x0710,0x07A0,0x03C0,
	0x00E0,0x0EE0,0x1CE0,0x1CE0,0x0EC0,0x0700,0x0000,0x0000,
	// Arrow
	0x0000,0x0000,0x0000,0x00C0,0x00E0,0xFFB0,0x8018,0x800C,
	0x8006,0x800C,0x8018,0xFFB0,0x00E0,0x00C0,0x0000,0x0000,
	0x0000,0x0000,0x0000,0x00C0,0x00E0,0xFFF0,0xFFF8,0xFFFC,
	0xFFFE,0xFFFC,0xFFF8,0xFFF0,0x00E0,0x00C0,0x0000,0x0000
};

// Data for the logo
unsigned long logo[2*3*LOGOHEIGHT] = {
	// tile 0
	0x00000000,0x00000000,0x00000000,0x38000000,
	0x3E000000,0x3F800000,0x30E00000,0x30200000,
	0x30300000,0x18300000,0x1C380000,0x2E780000,
	0x3FE00000,0x23C3F000,0x21E0FF80,0x20F0FEC1,
	0x3078F070,0x3038F038,0x1838F038,0x0838F03C,
	0x0678F03C,0x03F8F03C,0x0180F038,0x0000F838,
	0x0000FCF0,0x0000F7E0,0x0000F000,0x0003F000,
	0x0001FC00,0x00000C00,0x00000000,0x00000000,
	0x00000000,0x00000000,0x00000000,0x44000000,
	0x7C000000,0x73000000,0x71C00000,0x70700000,
	0x18200000,0x18380000,0x08380000,0x5F380000,
	0x7FF3E000,0x61E3E800,0x60C07F80,0x70E079E1,
	0x70387021,0x30387070,0x303C7078,0x1C3C7078,
	0x0E387078,0x01F0707C,0x0060707C,0x00007078,
	0x00007E78,0x000073E0,0x00027000,0x0002F000,
	0x0003FC00,0x00001000,0x00000000,0x00000000,
	// tile 1
	0x00000000,0x00000000,0x00000000,0x00000000,
	0x00000000,0x00000000,0x00000000,0x00000000,
	0x00000000,0x00000000,0x00000000,0x00000000,
	0x00000000,0x00000000,0x00000000,0xF1F0FFA3,
	0xF1F0FFE3,0x7070E1C3,0x7070C382,0x70708700,
	0x70700F00,0x70701E00,0x70703C41,0x60707861,
	0x70F0F0E3,0x7FF8E3E7,0x1F7FFFE7,0x00000000,
	0x00000000,0x00000000,0x00000000,0x00000000,
	0x00000000,0x00000000,0x00000000,0x00000000,
	0x00000000,0x00000000,0x00000000,0x00000000,
	0x00000000,0x00000000,0x00000000,0x00000000,
	0x00000000,0x00000000,0x00000000,0xE2000047,
	0xE3F9FFF7,0xE071E3E7,0xE0F1E3C6,0xE0F00380,
	0xE0F00600,0xE0F00C00,0xE0F01C21,0xF0703863,
	0xF1F07847,0x31F0F0EF,0x1E3DFFE8,0x00000000,
	0x00000000,0x00000000,0x00000000,0x00000000,
	// tile 2
	0x00000000,0x00000000,0x00000000,0x00000000,
	0x00000000,0x00000000,0x00000000,0x00020000,
	0x00360000,0x003E00F8,0x000E0330,0x000E0630,
	0x000E0C30,0x000E0C60,0x0F8E1998,0xFF0E3E10,
	0x8F0E3C10,0x0E0E3810,0x1C0E3C30,0x380E1E60,
	0x700E0FC0,0xF10E0F80,0xE10F8000,0xC31F8000,
	0x873F0000,0xFF000000,0xE0000000,0x00000000,
	0x00000000,0x00000000,0x00000000,0x00000000,
	0x00000000,0x00000000,0x00000000,0x00000000,
	0x00000000,0x00000000,0x00000000,0x00000000,
	0x000E0070,0x003E01F0,0x001E03F8,0x001E0738,
	0x001E0620,0x031E08C0,0x079E0CC0,0xFF9E1F38,
	0x0E1E3818,0x1E1E3C30,0x1C1E3820,0x381E3E00,
	0x781E1F80,0xE01E0700,0xC11E4700,0xC51FC000,
	0x8F1F8000,0xBF000000,0x10000000,0x00000000,
	0x00000000,0x00000000,0x00000000,0x00000000
};

short number[9] = {0,1,2,3,4,5,6,7,8};					// Number array (in fact, the board to play)
short pos = 0;																	// Position of the cursor


// Randomizes the number array
void RandomizeNumbers(void)
{
	short i, r1, r2, buffer;
  
	// Initialize random number generator
	randomize();	

	// '50' is just a value I've choosen because after
	// 50 loops I think the field has been randomized enough
	for (i = 0; i < 50; i++) {
		r1 = random(9);
		r2 = random(9);
		buffer = number[r1];
		number[r1] = number[r2];
		number[r2] = buffer;
	}
}


// Initializes the board
void InitBoard(BOOL flag)
{
	short x, y;
	 
	// Draw tiles onto the screen
	for (y = 0; y < 3; y++) {
		for (x = 0; x < 3; x++) {
			GraySprite16_RPLC_R((x*16)+8+CALCX, (y*16)+44+CALCY, TILEHEIGHT, TILEADDR(number[(y*3)+x], LIGHT_PLANE),
				TILEADDR(number[(y*3)+x], DARK_PLANE), GrayGetPlane(DARK_PLANE), 
				GrayGetPlane(LIGHT_PLANE));
			if (flag)
				GraySprite16_XOR_R((x*16)+104+CALCX, (y*16)+44+CALCY, TILEHEIGHT, TILEADDR(((y*3)+x), LIGHT_PLANE),
					TILEADDR(((y*3)+x), DARK_PLANE), GrayGetPlane(DARK_PLANE), 
					GrayGetPlane(LIGHT_PLANE));
			else
				continue;
			
		}
	}
	
	// Draw the logo and these other things
	if (flag) {
		GrayDrawRect(0+CALCX, 0+CALCY, 159+CALCX, 99+CALCY, COLOR_BLACK, RECT_EMPTY);
		
		GraySprite16_XOR_R(72+CALCX, 60+CALCY, TILEHEIGHT, TILEADDR(9, LIGHT_PLANE),
			TILEADDR(9, DARK_PLANE), GrayGetPlane(DARK_PLANE), 
			GrayGetPlane(LIGHT_PLANE));
		
		for (x = 0; x < 3; x++)
			GraySprite32_XOR_R((x*32)+32+CALCX, 0+CALCY, LOGOHEIGHT, LOGOADDR(x, LIGHT_PLANE),
				LOGOADDR(x, DARK_PLANE), GrayGetPlane(DARK_PLANE), 
				GrayGetPlane(LIGHT_PLANE));
				
		GrayDrawRect(0+CALCX, 34+CALCY, 160+CALCX, 38+CALCY, COLOR_BLACK, RECT_FILLED);
		GrayFastDrawHLine(1+CALCX, 159+CALCX, 36+CALCY, COLOR_LIGHTGRAY);
		
		GrayDrawStrExt(66+CALCX, 7+CALCY, "saubue's", A_SHADOWED | A_XOR, F_4x6);	
	}
}


// Checks if a game has been won
BOOL CheckGameOver(void)
{
	short i;
	
	for (i = 0; i < 9; i++) {
		if (number[i] != i)
			return FALSE;
	}
	
	return TRUE;
}


// Lets the cursor blink
CALLBACK void CursorBlink(void)
{
	GraySprite16_XOR_R(((pos-(pos/3)*3)*16)+8+CALCX, ((pos/3)*16)+44+CALCY, TILEHEIGHT, TILEADDR(0, LIGHT_PLANE),
						TILEADDR(0, DARK_PLANE), GrayGetPlane(DARK_PLANE), 
						GrayGetPlane(LIGHT_PLANE));	
}


// Clears the keyboard queue
void ClearQueue(void)
{
	while (_rowread(0)) continue;	
}


// Moves the cursor
BOOL MoveBoard(short direction)
{
	ClearQueue();

	// Change the values and update the position
	if (direction == LEFT) {
		number[pos] = number[pos-1];
		number[pos-1] = 0;
		pos--;
	} else if (direction == RIGHT) {
		number[pos] = number[pos+1];
		number[pos+1] = 0;
		pos++;
	} else if (direction == UP) {
		number[pos] = number[pos-3];
		number[pos-3] = 0;
		pos-=3;
	} else if (direction == DOWN) {
		number[pos] = number[pos+3];
		number[pos+3] = 0;
		pos+=3;
	}

	InitBoard(FALSE);
	return CheckGameOver();
}


// Main game routine
void GameStart(void)
{
	short loop;
	
	// Game loop
	do {
		ClearGrayScreen();
		RandomizeNumbers();
	
		// Find the '0'-position in the array
		for (loop = 0; loop < 9; loop++) {
			if (!number[loop]) {
				pos = loop;
					break;
			} else continue;
		}

		// Register Vectored timer for blinking of the cursor
		OSVRegisterTimer(1, 10, CursorBlink); 

		// Draw the board onto the screen
		InitBoard(TRUE);

		// Playing loop
		do {
			if (_keytest(RR_RIGHT) && (pos-(pos/3)*3) < 2) {
				if (MoveBoard(RIGHT))
					break;
			} else if (_keytest(RR_LEFT) && (pos-(pos/3)*3)) {
				if (MoveBoard(LEFT))
					break;
			} else if (_keytest(RR_UP) && (pos/3)) {
				if (MoveBoard(UP))
					break;
			} else if (_keytest(RR_DOWN) && (pos/3) < 2) {
				if (MoveBoard(DOWN))
					break;

			} else if (_keytest(RR_ESC)) {
				OSVFreeTimer(1);
				return;
			}
		}	while (TRUE);

		OSVFreeTimer(1);

		// Display a message box
		GrayDrawRect(31+CALCX, 31+CALCY, 129+CALCX, 69+CALCY, COLOR_LIGHTGRAY, RECT_FILLED);
		GrayDrawRect(30+CALCX, 30+CALCY, 130+CALCX, 70+CALCY, COLOR_BLACK, RECT_EMPTY);
		GrayDrawStrExt(0, 32+CALCY, "Congratulations!", A_CENTERED | A_NORMAL, F_6x8);
		GrayDrawStrExt(0, 42+CALCY, "You have won!", A_CENTERED | A_NORMAL, F_6x8);
		GrayDrawStrExt(0, 60+CALCY, "Want to play again?", A_CENTERED | A_NORMAL, F_4x6);

		do {
			if (_keytest(RR_ESC)) {
				ClearQueue();
				return;
			}
		} while (!_keytest(RR_ENTER) && !_keytest(RR_ENTER2));

		ClearQueue();
		
	} while (TRUE);
}


// Program entry point
void _main(void)
{
	LCD_BUFFER *buffer;
	INT_HANDLER savint1 = GetIntVec(AUTO_INT_1);
	INT_HANDLER savint5 = GetIntVec(AUTO_INT_5);
 
	if ((buffer = malloc(sizeof(LCD_BUFFER))) == NULL) {
		ST_helpMsg("memory allocation failed");
		return;
	}
	
	// Save LCD contents
	LCD_save(buffer);
 
	// Redirect Auto-Int 1 and 5 to nothing
	SetIntVec(AUTO_INT_1, DUMMY_HANDLER);
	SetIntVec(AUTO_INT_5, DUMMY_HANDLER);
 
	if (!GrayOn())
		return;
 
 	// Start the game
	GameStart();
	
	ClearGrayScreen();
	GrayOff();
	 
	// Restor Auto-Ints
	SetIntVec(AUTO_INT_1, savint1);
	SetIntVec(AUTO_INT_5, savint5);

	// Restore LCD contents
	LCD_restore(buffer);
	HeapFreePtr(buffer);
	ST_helpMsg("EightPuzzle by saubue");
}