------------------------------------------------------
-- Reads temperature values from a temperture probe --
-- connected to an Arduino Leonardo                 --
-- (C) 2016 Jim Bauwens                             --
-- tiplanet.org                                     --
------------------------------------------------------

--  This program is free software: you can redistribute it and/or modify
--  it under the terms of the GNU General Public License as published by
--  the Free Software Foundation, either version 3 of the License, or
--  (at your option) any later version.

--  This program is distributed in the hope that it will be useful,
--  but WITHOUT ANY WARRANTY; without even the implied warranty of
--  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--  GNU General Public License for more details.

--  You should have received a copy of the GNU General Public License
--  along with this program.  If not, see <http://www.gnu.org/licenses/>.


platform.apiLevel = '2.7'

require 'asi'

local connected = false
local scanning = true
local global_port
local temp = "-.--C"


---------------------------
-- Set temperature value --
---------------------------

function setTemp(t)
	temp = t .. "C"
	
	platform.window:invalidate()
end

---------------------------------
-- Request temperature value   --
-- Will send 'T' and read port --
---------------------------------

function requestTemp()
	if global_port then
		global_port:write('T')
		global_port:read()
	end
end

--------------------
-- Scan for ports --
--------------------

function startScanning()
	asi.startScanning(portFoundCallback)
	scanning = true

	platform.window:invalidate()
end

----------------------------------
-- ASI state change callback    --
-- If ASI is on, start scanning --
----------------------------------

function asiStateCallback(state)
	if state == asi.ON then
		startScanning()
	end
end

--------------------------------
-- Port found callback        --
-- Will try to connect to the --
-- discovered port            --
--------------------------------

function portFoundCallback(port)
	port:connect(connectionCallback)
end

------------------------------------------
-- Port connection state callback       --
--                                      --
-- On connection it will stop scanning, --
-- set the read listeneren and send an  --
-- initial temperature probe            --
--                                      --
-- On disconnection it will start to    --
-- scan again for ports                 --
------------------------------------------

function connectionCallback(port, event)
	if event == asi.CONNECTED then
		connected = true
		
		asi.stopScanning()
		scanning = false

		port:setReadListener(readCallback)
		global_port = port

		requestTemp()

	elseif event == asi.DISCONNECTED then
		connected = false
		temp = "-.--C"

		startScanning()
	end

	platform.window:invalidate()
end

-------------------------------------------------
-- Called after data is read from the device   --
-- Data is send with the format 'XDATA;' where --
-- X is used as data type identifier           --
-------------------------------------------------

function readCallback(port)
	local data = port:getValue()

	if data then
		local response = data:split(';') 
		-- there could have been multiple packets send
		-- we need to handle them individually
		for k,v in ipairs(response) do
			doResponse(v:sub(1,1), v:sub(2))
		end
	end
end

----------------------------
-- Handle device response --
----------------------------

function doResponse(rtype, data)
	if rtype == 'T' then
		setTemp(data)
		-- After receiveing temperature, probe for an updated temperature
		requestTemp()
	end 
end

-------------------------------------------
-- Start listening for ASI state changes --
-------------------------------------------

function on.construction()
	timer.start(5)
	asi.addStateListener(asiStateCallback)
end

------------------------------------
-- Force temperature probe update --
------------------------------------

function on.enterKey()
	requestTemp()
end

function on.tabKey()
	-- fake temperature
	temp = "13.42C"
	platform.window:invalidate()
end

function on.timer()
	-- request at least every 5s an update
	requestTemp()
end

---------------------
-- Draw everything --
---------------------

function on.paint(gc, x, y, width, height)
	gc:setFont("sansserif", "r", 7)
	gc:drawString("Connected: " .. tostring(connected) .. ", scanning: " .. tostring(scanning), 2, 2, "top")

	gc:setFont("serif", "b", 24)
	gc:setColorRGB(0xFF0000)

	local w,h = gc:getStringWidth(temp), gc:getStringHeight(temp)
	gc:drawString(temp, (width-w)/2, (height-h)/2, "top")
end
