#include <stdio.h>
#include <stdlib.h>

enum header_type {
    header_raw,
    header_zshell4,
    header_86asm,
    header_ace,
    header_usgard
};

typedef struct {
    char extension[4];
    char id_bytes[8];
    char pad_variable_name;
    char store_extra_bytes;
    char store_name_length;
    char force_upper_case;
    char extra_on_calc_bytes;
    char variable_type;    
    enum header_type default_header;
} file_type;

file_type file_types[] = {
    { ".82P", "**TI82**", 1, 0, 0, 0, 4, 6, header_ace},
    { ".83P", "**TI83**", 1, 0, 0, 1, 4, 6, header_raw},
    { ".8XP", "**TI83F*", 1, 1, 0, 1, 6, 6, header_raw},
    { ".85S", "**TI85**", 0, 0, 1, 0, 4, 12, header_zshell4},
    { ".86P", "**TI86**", 1, 0, 1, 0, 5, 18, header_86asm}};

unsigned char input_data[29000];
unsigned int input_len;
unsigned char reloc_table[4804];
unsigned short reloc_size = 0;
unsigned short checksum = 0;
FILE *infile, *outfile, *relocs = NULL;

/* Write data to file, updating checksum */

void checksummed_write(unsigned char *data, int size)
{
    while (size--)
    {
        checksum += *data;
        fputc(*data++, outfile);
    }
}

void checksummed_write_word(unsigned short data)
{
    unsigned char buffer[2];
    buffer[0] = data & 0xFF;
    buffer[1] = data >> 8;
    checksummed_write(buffer, 2);
}

/* Calculate size of the data section of the variable */

unsigned short calculate_data_size(enum header_type header)
{
    int data_size = input_len + 2;
    if (header == header_zshell4) data_size += 4;
    if (header == header_usgard) data_size += 3 + reloc_size;
    if (header == header_ace) data_size += 3;
    if (header == header_86asm) data_size += 2;
    return data_size;
}

/* write the on-calc header of a variable */

unsigned char write_calc_header(enum header_type header)
{
    /* search for end of title (only used by ZShell) */
    unsigned char title_len = 0;
    do {
        if (input_data[title_len] == 0) break;
    } while (title_len++ < 100);

    if (header == header_ace)
    {
        char aceheader[] = {0xd5, 0, 0xa0};
        checksummed_write(aceheader, sizeof(aceheader));
    }
    if (header == header_86asm)
    {
        char headerbytes[] = {0x8e, 0x28};
        checksummed_write(headerbytes, sizeof(headerbytes));
    }
    title_len++;
    if (header == header_zshell4)
    {
        char headerbytes[] = {0, 0xfd};
        checksummed_write(headerbytes, sizeof(headerbytes));
        checksummed_write(&title_len, 1);
    }
    if (header == header_usgard)
    {
        char headerbytes[] = {0, 0xf8};
        checksummed_write(headerbytes, sizeof(headerbytes));
        checksummed_write(&title_len, 1);
    }
    
    return title_len;
}

int uicompare(const void *a, const void *b)
{
    unsigned short c = *((unsigned short *)a);
    unsigned short d = *((unsigned short *)b);
    if (c > d) return 1;
    else if (c < d) return -1;
    else return 0;
}

void build_reloc_table()
{
    unsigned short reloc_offsets[1600];
    unsigned short previous_offset = 0;
    unsigned short reloc_advance;
    int relative_stored = 0, absolute_stored = 0, reloc_count = 0, i; 
    char line[1600];

    if (relocs)
    {
        fgets(line, sizeof(line), relocs);
        if (line[0] == '0')
        {
            /* TASM mode, search forward for ----- to find table at the end */
            while (!feof(relocs))
            {
                fgets(line, sizeof(line), relocs);
                if (line[0] == '-') break;
            }     
            while (!feof(relocs))
            {
                fgets(line, sizeof(line), relocs);
                /* TASM entries are of the format "addr     N      label" */
                if (strlen(line) >= 18 && line[16] == 'R' && line[17] == '_')
                {
                    unsigned int reloc_target;
                    sscanf(line, "%x", &reloc_target);
                    if (reloc_count < 1600) reloc_offsets[reloc_count++] = reloc_target;
                }
            }
        }
        else if (line[0] == '<')
        {
            /* Brass mode, search for headers for each label */
            char match_string[] = "<tr><td class=\"l\" colspan=\"4\">R_";
            char match_second[] = "<tr class=\"r0\"><td class=\"c\">";
            char match_third[] = "<tr class=\"r1\"><td class=\"c\">";
            while (!feof(relocs))
            {
                fgets(line, sizeof(line), relocs);
                if (0 == strncmp(match_string, line, strlen(match_string)))
                {
                    fgets(line, sizeof(line), relocs);
                    if (0 == strncmp(match_second, line, strlen(match_second)) ||
                        0 == strncmp(match_third, line, strlen(match_third)))
                    {
                        unsigned int reloc_target;
                        sscanf(&line[strlen(match_second)], "%x", &reloc_target);
                        if (reloc_count < 1600) reloc_offsets[reloc_count++] = reloc_target;
                    }
                }
            }                        
        }
        else
        {
            do
            {
                /* SPASM entries are of the format "label = $addr" */
                if (strlen(line) > 4 && line[0] == 'R' && line[1] == '_')
                {
                    for (i = 0; i < strlen(line); i++)
                    {
                        if (line[i] == '$')
                        {
                            unsigned int reloc_target;
                            sscanf(&line[i + 1], "%x", &reloc_target);
                            if (reloc_count < 1600) reloc_offsets[reloc_count++] = reloc_target;
                        }
                    }
                }
                fgets(line, sizeof(line), relocs);
            }
            while (!feof(relocs));
        }

        qsort(reloc_offsets, reloc_count, sizeof(unsigned short), uicompare);
        for (i = 0; i < reloc_count; i++)
        {
            unsigned short reloc_offset = reloc_offsets[i];
            if (reloc_offset >= input_len)
            {
                printf("Var8x warning - ignoring out of range reloc label\n");
                break;
            }
            /* Advance 1 byte to reach address field, 2 bytes for IX/IY */
            if (input_data[reloc_offset] == 0xDD || input_data[reloc_offset] == 0xFD)
                reloc_offset++;
            reloc_offset++;
            if (reloc_offset + 1 >= input_len)
            {
                printf("Var8x warning - ignoring out of range reloc label\n");
                break;
            }
            reloc_advance = reloc_offset - previous_offset;
            if (reloc_advance == 0)
            {
                printf("Var8x warning -- duplicate reloc label\n");
                continue;
            }
            
            if (reloc_advance < 256)
            {
                reloc_table[reloc_size++] = reloc_advance;
                relative_stored++;
            }
            else
            {
                reloc_table[reloc_size++] = 0;
                reloc_table[reloc_size++] = reloc_offset & 255;
                reloc_table[reloc_size++] = reloc_offset >> 8;
                absolute_stored++;
            }
            
            previous_offset = reloc_offset;
        }
    }
    
    reloc_count = -reloc_size;
    reloc_table[reloc_size++] = reloc_count & 255;
    reloc_table[reloc_size++] = reloc_count >> 8;
    reloc_count = relative_stored + absolute_stored;
    reloc_table[reloc_size++] = reloc_count & 255;
    reloc_table[reloc_size++] = reloc_count >> 8;

    printf("reloc table is %d bytes with %d relative addrs and %d absolute adrs\n",
        reloc_size, relative_stored, absolute_stored);
        
}

void show_usage(char *filename)
{
    printf("var8x - Variable packager for TI-82 / 83 / 83+ / 84+ / 85 / 86 calculators\n");
    printf("by Patrick Davidson pad@ocf.berkeley.edu http://www.ocf.berkeley.edu/~pad\n");
    printf("Version 2 released August 18, 2011\n\n");
    printf("Usage: %s infile outfile [header type] [listing]\n", filename);
    printf("Listing file should only be specified for Usgard programs\n");
    printf("Variable type is determined by outfile extension which must be one of these\n");
    printf(".82P, .83P, .8XP, .85S, .86P\n\n");
    printf("The header type value decides the on-calc header and must be one of these\n");
    printf("raw - No additional header (default for 83P and 8XP)\n");
    printf("86asm - TI-86 assembly program (default for 86P)\n");
    printf("ace - TI-82 Ace beta 4 (default for 82P)\n");
    printf("usgard - TI-85 Usgard 1.5\n");
    printf("zshell - TI-85 ZShell 4.0 (default for 85S)\n");
}

int main(int argc, char *argv[])
{
    char *extension;
    char *header_bytes;
    enum header_type header;
    file_type *output_file_type = NULL;
    int var_name_len;
    unsigned short data_len, var_header_len, combined_len;
    char variable_name[9];
    char *name_pointer;
    int file_name_index, var_name_index, i;
    unsigned char zschecksum;
    
    if (argc < 3 || argc > 5)
    {
        show_usage(argv[0]);
        return 5;
    } 
    
    /* Open and read input file */
    
    infile = fopen(argv[1], "rb");
    if (infile == NULL)
    {
        printf("Could not open input file %s\n", argv[1]);
        return 20;
    }
    input_len = fread(input_data, 1, sizeof(input_data), infile);
    if (!(feof(infile)))
    {
        printf("Input file %s is too large\n", argv[1]);
        return 20;
    }
    if (ferror(infile))
    {
        printf("Error reading input file %s\n", argv[1]);
        return 20;
    }
    fclose(infile);
    
    /* Decode output file name to determine variable type */
    
    if (strlen(argv[2]) < 5)
    {
        printf("Invalid output file name %s\n", argv[2]);
        show_usage(argv[0]);
        return 20;
    }
    for (i = 0; i < 5; i++)
    {
        if (0 == strnicmp(file_types[i].extension, argv[2] + strlen(argv[2]) - 4, 4))
        {
            output_file_type = &file_types[i];
        }
    }
    if (output_file_type == NULL)
    {
        printf("Invalid outfile extension on %s\n", argv[2]);
        show_usage(argv[0]);
        return 20;
    }
    
    /* Open output file */
    
    outfile = fopen(argv[2], "wb");
    if (outfile == NULL)
    {
        printf("Unable to open output file %s\n", argv[2]);
        return 20;
    }
    
    /* Decode header type */

    header = output_file_type->default_header;
    if (argc >= 4)
    {
        if (0 == strnicmp(argv[3], "raw")) header = header_raw;
        else if (0 == strnicmp(argv[3], "ace")) header = header_ace;
        else if (0 == strnicmp(argv[3], "zshell")) header = header_zshell4;
        else if (0 == strnicmp(argv[3], "usgard")) header = header_usgard;
        else if (0 == strnicmp(argv[3], "86asm")) header = header_86asm;
        else {
            printf("Invalid header type %s\n", argv[3]);
            show_usage(argv[0]);
            return 20;
        }
    }
    
    if (header == header_usgard) 
    {
        if (argc == 5)
        {
            relocs = fopen(argv[4], "r");
            if (relocs == NULL)
            {
                printf("Unable to open listing file %s\n", argv[4]);
                return 20;
            }
        }
         
        build_reloc_table();
    }
    
    /* Expand variable name */
    
    file_name_index = 0;
    var_name_index = 0;
    for (i = 0; i < strlen(argv[2]); i++)
        if (argv[2][i] == '/' || argv[2][i] == '\\' || argv[2][i] == '"')
            file_name_index = i + 1;       
    if (header == header_ace) variable_name[var_name_index++] = '\xdc';
    while (var_name_index < 8)
    {
        if (argv[2][file_name_index] == '.') break;
        variable_name[var_name_index++] = argv[2][file_name_index++];
        if (output_file_type->force_upper_case)
            variable_name[var_name_index - 1] = toupper(variable_name[var_name_index - 1]);
    }
    var_name_len = var_name_index;
    if (output_file_type->pad_variable_name)
        while (var_name_index < 8)
            variable_name[var_name_index++] = 0;
    
    variable_name[var_name_index] = 0;
    
    /* Write output file header */
   
    fwrite(output_file_type->id_bytes, 1, 8, outfile);
    fwrite("\x1a\x0a\0", 1, 3, outfile);
    fputs("TI calculator file generated by Var8x v2  ", outfile);
    data_len = calculate_data_size(header);
    
    var_header_len = 3 + var_name_index;
    if (output_file_type->store_extra_bytes) var_header_len += 2;
    if (output_file_type->store_name_length) var_header_len++;

    combined_len = data_len + var_header_len + 4;
    fputc(combined_len & 0xFF, outfile);
    fputc(combined_len >> 8, outfile);
    
    /* Write variable header */
    
    checksummed_write_word(var_header_len);
    checksummed_write_word(data_len);
    checksummed_write(&output_file_type->variable_type, 1);
    if (output_file_type->store_name_length)
        checksummed_write((unsigned char *)&var_name_len, 1);
    checksummed_write(variable_name, var_name_index);
    if (output_file_type->store_extra_bytes) checksummed_write_word(0);
    
    /* Write data and checksum */
    
    checksummed_write_word(data_len);
    checksummed_write_word(data_len - 2);    
    zschecksum = write_calc_header(header);
    checksummed_write(input_data, input_len);
    checksummed_write(reloc_table, reloc_size);
    if (header == header_zshell4)
    {
        for (i = 0; i < input_len; zschecksum += input_data[i++]);
        checksummed_write(&zschecksum, 1);
    }
    fwrite(&checksum, 2, 1, outfile);
    fclose(outfile);
    
    printf("Var8x created %s, total size on calculator is %d bytes\n", 
        argv[2], data_len + var_name_index + output_file_type->extra_on_calc_bytes);
    
    return 0;
}
