//#define TI89_VERSION /*This controls whether the program is compiled for an 89 or a 92+.*/
/*Comment out the above line to compile for a 92+/V200*/
//#define AUTO_PLAY

/*This construct selects the proper compiler directives and calculator-dependent constants to compile on
whicever calculator is indicated by the previous line*/
#ifdef TI89_VERSION
#define USE_TI89
#define calc(x, y) (x)
#else
#define USE_TI92PLUS
#define USE_V200
#define calc(x, y) (y)
#endif

/*these are a bunch of non-calculator dependent constants that either make it easier for me to remember
which value does what, or control gameplay*/
#define CURSOR_BLINK_DELAY 5 /*in 20ths of a second*/
#define BONUS_BLINK_DELAY 9 /*in 20ths of a second*/
#define SCORE_DELAY 100 /*in milliseconds*/
#define DROP_DELAY 4 /*in milliseconds, repeated twelve times*/
#define NUM_TILES 7
#define CURSOR 8
#define QUEUE_SIZE 16
#define DATA_NAME SYMSTR("swapdata")
#define RANDOM_STRENGTH 4
#define VERSION_MAJOR 1
#define VERSION_MINOR 2
enum {NO_EMPTY = -1, NO_MOVE = -1, DEFAULT, ESC, HINT, NORMAL, PLAY_GAME, QUIT, SAVE, TIMED};

/*Boring.  A bunch of header files.*/
#include <tigcclib.h>
#include "extgraph.h"
#include "extrand.h"
#include "sprites.h" /*the sprite data*/

typedef struct
{
	short x1, y1, x2, y2;
} COMBO;

typedef struct
{
	COMBO data[QUEUE_SIZE];
	short front;
	short back;
} QUEUE;

typedef struct
{
	char initials[3];
	unsigned char random_strength : 4, padding : 4;
} FILE_VERSION;

typedef struct
{
	struct
	{
		unsigned short size;
		FILE_VERSION version;
	} header;
	struct
	{
		char mode;
		char board[8][8];
		char save;
		unsigned char bonusLevel;
		unsigned short bonus;
		unsigned long score;
		unsigned long highScores[7][2];
		RANDOM_BUFFER buffer;
	} data;
	struct
	{
		char hash[17];
		char type[7];
	} footer;
} DATA_FILE;

/*The function prototypes*/
static char MainMenu(unsigned long[][2], char *);
static void GenerateBoard(char[][8]);
static void DisplayBoard(char[][8]);
static void DrawSprite(short, short, short);
static char PlayGame(char[][8], unsigned long *, char, LCD_BUFFER[]);
static void CheckHighScores(char, unsigned long, unsigned long[][2]);
static char FindMove(char[][8], COMBO *);
static char CheckCombo(char[][8], COMBO);
static char GetMove(short *, short *, COMBO *, char);
static short Wrap(short, short, short);
static void DrawBonus(char);
static void ShowHint(COMBO, char, char, unsigned long *);
static char HandleMove(char[][8], COMBO *, QUEUE *, char, unsigned long *, char, LCD_BUFFER[]);
static void SwapPieces(char[][8], COMBO, char);
static char CheckRow(char[][8], short, QUEUE *, char *);
static char CheckCol(char[][8], short, QUEUE *, char *);
static void Pause(void);
static unsigned long DrawPoints(COMBO, short, char, char, LCD_BUFFER[]);
static void DrawScore(unsigned long);
static void RemoveCombo(char[][8], COMBO);
static void Gravity(char[][8]);
static void HandleBonus(char[][8]);
void SleepMS_HW1(unsigned short);
void SleepMS_HW2(unsigned short);

unsigned char FreezeBonus = 1; /*This is set to one to allow time to display animations by stopping the
timer*/
unsigned char BonusLevel = 2; /*which bonus level you are currently on*/
volatile unsigned char Time = 0; /*used to control cursor blink rate*/
volatile unsigned char Time2 = 0; /*used to control bonus meter blink rate*/
volatile short Bonus = 0; /*This is used for both timed and untimed modes.  To get untimed mode,
use Int5Timer instead of Int5Bonus.*/

/*If you are playing timed mode, install this handler.*/
DEFINE_INT_HANDLER(Int5Bonus)
{
	if(*(unsigned char *)&Time > 0)
	{
		(*(unsigned char *)&Time)--;
	}
	if(*(unsigned char *)&Time2 > 0)
	{
		(*(unsigned char *)&Time2)--;
	}
	if(*(unsigned short *)&Bonus > 0 && FreezeBonus == 0)
	{
		*(unsigned short *)&Bonus -= BonusLevel - 1;
	}
}

/*If you are not playing timed mode, install this handler.*/
DEFINE_INT_HANDLER(Int5Timer)
{
	if(*(unsigned char *)&Time > 0)
	{
		(*(unsigned char *)&Time)--;
	}
}

void _main(void)
{
	char goOn = PLAY_GAME; /*used to tell the main loop to exit when esc pressed in the menu*/
	char hash[16];
	short oldFont; /*used to save and restore the font that the AMS has set*/
	DATA_FILE *filep = NULL;
	static const FILE_VERSION Check = {{'P', 'J', 'R'}, RANDOM_STRENGTH, 0};
	INT_HANDLER oldInt1; /*This is used to save and restore the Int 1 timer so the AMS does not freak
	out*/
	INT_HANDLER oldInt5; /*This is used to save and restore the Int 5 timer so the AMS does not freak
	out*/
	LCD_BUFFER *lcd_buffers; /*pointer to memory for screen save/restore*/
	MD5_CTX ctx;
	SYM_ENTRY *file_vat;

	lcd_buffers = malloc(7 * sizeof(LCD_BUFFER)); /*allocate memory for saving the screen*/
	if(lcd_buffers == NULL) /*if there was not enough memory*/
	{
		ST_helpMsg("ERROR: memory low");
		return;
	}

	file_vat = SymFindPtr(DATA_NAME, 0); /*try to find the data file*/
	if(file_vat != NULL)
	{
		if(file_vat->flags.bits.archived)
		{
			EM_moveSymFromExtMem(DATA_NAME, HS_NULL);
		}
		filep = HeapDeref(file_vat->handle);

		MD5Init(&ctx);
		MD5Update(&ctx, (char *)&filep->data, sizeof(filep->data));
		MD5Final(hash, &ctx);
	}

	if(file_vat == NULL || memcmp(&filep->header.version, &Check, sizeof(FILE_VERSION)) ||
	memcmp(filep->footer.hash, hash, 16))
	{
		RANDOM_BUFFER buffer;

		randomize_long();

		SaveRandState(&buffer);
		/*make the file*/
		SymDel(DATA_NAME);
		file_vat = DerefSym(SymAdd(DATA_NAME));
		file_vat->handle = HeapAlloc(sizeof(DATA_FILE));
		if(file_vat->handle == H_NULL)
		{
			ST_helpMsg("ERROR: memory low");
			free(lcd_buffers);
			return;
		}
		filep = HeapDeref(file_vat->handle);
		memset(filep, 0, sizeof(DATA_FILE));

		filep->header.size = sizeof(DATA_FILE) - 2;
		memcpy(&filep->header.version, &Check, sizeof(FILE_VERSION));
		filep->data.mode = NORMAL;
		memcpy(&filep->data.buffer, &buffer, sizeof(RANDOM_BUFFER));
		memcpy(filep->footer.type, &(char []){0, 'D', 'a', 't', 'a', 0, OTH_TAG}, 7);
	}

	LoadRandState(&filep->data.buffer);

	FastCopyScreen_R(LCD_MEM, lcd_buffers[0]); /*save the screen*/

	oldInt1 = GetIntVec(AUTO_INT(1)); /*save the Int 1 handler*/
	oldInt5 = GetIntVec(AUTO_INT(5)); /*save the Int 5 handler*/
	SetIntVec(AUTO_INT(1), DUMMY_HANDLER); /*turn off the AMS's keyboard handler*/
	SetIntVec(AUTO_INT(5), Int5Timer); /*turn off the AMS's time handler*/
	oldFont = FontSetSys(F_4x6); /*set the font to 4x6 and save the old one*/
	GrayOn();

	do
	{
		if(!filep->data.save)
		{
			goOn = MainMenu(filep->data.highScores, &filep->data.mode);
		}
		if(goOn == PLAY_GAME)
		{
			GrayClearScreen();
			if(!filep->data.save)
			{
				BonusLevel = 2;
				filep->data.score = 0;

				GenerateBoard(filep->data.board);

				if(filep->data.mode == TIMED)
				{
					Bonus = 3000;
					SetIntVec(AUTO_INT(5), Int5Bonus); /*install my time handler*/
				}
				else
				{
					Bonus = 0;
				}
			}
			else
			{
				memcpy(&BonusLevel, &filep->data.bonusLevel, sizeof(BonusLevel));
				memcpy((unsigned short *)&Bonus, &filep->data.bonus, sizeof(Bonus));
				filep->data.save = 0;
			}

			if(filep->data.mode == TIMED)
			{
				SetIntVec(AUTO_INT(5), Int5Bonus); /*install my time handler*/
			}

			GrayDrawRect(0, calc(0, 14), 98, calc(98, 112), COLOR_BLACK, RECT_EMPTY);

			DisplayBoard(filep->data.board);

			goOn = PlayGame(filep->data.board, &filep->data.score, filep->data.mode, lcd_buffers);
#ifndef AUTO_PLAY
			if(goOn == PLAY_GAME)
			{
				CheckHighScores(filep->data.mode, filep->data.score, filep->data.highScores);
			}
#endif
		}
	} while(goOn == PLAY_GAME || goOn == ESC);

	if(goOn == SAVE)
	{
		filep->data.save = 1;
		memcpy(&filep->data.bonusLevel, &BonusLevel, sizeof(BonusLevel));
		memcpy(&filep->data.bonus, (unsigned short *)&Bonus, sizeof(Bonus));
	}

	GrayOff();
	FontSetSys(oldFont); /*set the font back to the old font*/
	SetIntVec(AUTO_INT(5), oldInt5); /*restore the Int 5 handler*/
	SetIntVec(AUTO_INT(1), oldInt1); /*restore the Int 1 handler*/
	FastCopyScreen_R(lcd_buffers[0], LCD_MEM); /*restore the screen*/
	SaveRandState(&filep->data.buffer);

	MD5Init(&ctx);
	MD5Update(&ctx, (char *)&filep->data, sizeof(filep->data));
	MD5Final(hash, &ctx);

	EM_moveSymToExtMem(DATA_NAME, HS_NULL);

	free(lcd_buffers); /*free the memory for the screen buffer*/
	ST_helpMsg(EXTGRAPH_VERSION_PWDSTR);
}

/*precond: highScores is a 7x2 matrix.  postcond: *modep is the mode that the plaer wants to play.
returns 2 for [2nd] pressed and 1 for [ESC] pressed*/
char MainMenu(unsigned long highScores[][2], char *modep)
{
	char buffer[13]; /*a buffer for sprintf*/
	char goOn = DEFAULT; /*used to inform _main if [2nd] or [ESC] was pressed*/
	unsigned char key; /*used to store rowread values*/
	short i;
	static unsigned char circle[5] = {0x70, 0x88, 0x88, 0x88, 0x70}; /*a 5x5 circle sprite*/
	static unsigned char circleX[2] = {9, calc(115, 195)}; /*the x coordinates to output the circle at*/
	static unsigned char dot[3] = {0x70, 0x70, 0x70}; /*a 3x3 dot sprite to fill in the	circle with*/

	GrayClearScreen();
	sprintf(buffer, "Swapper v%d.%d", VERSION_MAJOR, VERSION_MINOR);
	GrayDrawStr((calc(160, 240) - DrawStrWidth(buffer, F_4x6)) / 2, 0, buffer, A_NORMAL);
	GrayDrawStr(calc(51, 91), 6, (char *)"By Peter J. Rowe", A_NORMAL);
	GrayDrawStr(calc(71, 111), 24, (char *)"Mode", A_NORMAL);
	GrayDrawStr(14, 30, (char *)"Normal", A_NORMAL);
	GrayDrawStr(calc(120, 200), 30, (char *)"Timed", A_NORMAL);
	GrayDrawStr(calc(60, 100), 42, (char *)"High Scores", A_NORMAL);
	GrayDrawStr(2, 48, (char *)"1st", A_NORMAL);
	GrayDrawStr(0, 54, (char *)"2nd", A_NORMAL);
	GrayDrawStr(0, 60, (char *)"3rd", A_NORMAL);
	GrayDrawStr(1, 66, (char *)"4th", A_NORMAL);
	GrayDrawStr(1, 72, (char *)"5th", A_NORMAL);
	GrayDrawStr(1, 78, (char *)"6th", A_NORMAL);
	GrayDrawStr(1, 84, (char *)"7th", A_NORMAL);
	GrayDrawStr(calc(38, 78), calc(95, 122), (char *)"Use \x15/\x16 + [2nd] or [ESC]", A_NORMAL);

	/*output the high scores and the circles for mode selection*/
	for(i = 0; i < 7; i++)
	{
		if(i < 2)
		{
			Sprite8_OR_R(circleX[i], 30, 5, circle, GrayGetPlane(LIGHT_PLANE));
			Sprite8_OR_R(circleX[i], 30, 5, circle, GrayGetPlane(DARK_PLANE));
		}
		sprintf(buffer, "%lu", highScores[i][0]);
		GrayDrawStr(14, 48 + 6 * i, buffer, A_NORMAL);
		sprintf(buffer, "%lu", highScores[i][1]);
		GrayDrawStr(calc(120, 200), 48 + 6 * i, buffer, A_NORMAL);
	}

	Sprite8_OR_R(circleX[*modep == TIMED], 31, 3, dot, GrayGetPlane(LIGHT_PLANE));
	Sprite8_OR_R(circleX[*modep == TIMED], 31, 3, dot, GrayGetPlane(DARK_PLANE));

	do
	{
		key = _rowread(0xFFFE);

		if(key & calc(10, 80)) /*if left or right pressed*/
		{
			while(_rowread(0xFFFE) & calc(10, 80)); /*wait until it is let up*/
			Sprite8_XOR_R(circleX[*modep == TIMED], 31, 3, dot, GrayGetPlane(LIGHT_PLANE));
			Sprite8_XOR_R(circleX[*modep == TIMED], 31, 3, dot, GrayGetPlane(DARK_PLANE));
			if(*modep == NORMAL)
			{
				*modep = TIMED;
			}
			else
			{
				*modep = NORMAL;
			}
			Sprite8_OR_R(circleX[*modep == TIMED], 31, 3, dot, GrayGetPlane(LIGHT_PLANE));
			Sprite8_OR_R(circleX[*modep == TIMED], 31, 3, dot, GrayGetPlane(DARK_PLANE));
		}
		if(key & calc(16, 1)) /*if [2nd] pressed*/
		{
			while(_rowread(0xFFFE) & calc(16, 1)); /*wait until [2nd] released*/
			goOn = PLAY_GAME; /*tell main that [2nd] was pressed*/
		}
		if(_rowread(calc(0xFFBF, 0xFEFF)) & calc(1, 64)) /*if [ESC] pressed*/
		{
			while(_rowread(calc(0xFFBF, 0xFEFF)) & calc(1, 64)); /*wait until [ESC]
			released*/
			goOn = QUIT;
		}
	} while(goOn == DEFAULT); /*loop until [2nd] or [ESC] pressed*/
	return(goOn); /*tell _main which key was pressed*/
}

/*postcond: board is an arrangement of pieces such that there are no combinations*/
void GenerateBoard(char board[][8])
{
	short i;
	short j;

	for(i = 0; i < 8; i++) /*columns*/
	{
		for(j = 0; j < 8; j++) /*rows*/
		{
			do
			{
				board[j][i] = random_long(NUM_TILES) + 1; /*place a piece between 1 and NUM_TILES*/
			} while((i >= 2 && board[j][i - 1] == board[j][i] && board[j][i - 2] == board[j][i]) ||
			(j >= 2 && board[j - 1][i] == board[j][i] && board[j - 2][i] == board[j][i])); /*loop until a
			combination is not formed by the last piece placed*/
		}
	}
}

/*postcond: the board is output to the screen*/
void DisplayBoard(char board[][8])
{
	short i;
	short j;

	for(i = 0; i < 8; i++)
	{
		for(j = 0; j < 8; j++)
		{
			DrawSprite(i, j, board[j][i]);
		}
	}
}

/*precond: 0 <= x <= 7, 0 <= y <= 7, 0 <= type <= 7.  postcond: the specififed sprite (for
type <= NUM_TILES) is output to the screen, or the cursor is output for type == 8.*/
void DrawSprite(short x, short y, short type)
{
	switch(type)
	{
		case 0:
		GrayDrawRect(12 * x + 2, 12 * y + calc(2, 16), 12 * x + 13, 12 * y + calc(13, 27), COLOR_WHITE,
		RECT_FILLED);
		break;

		case 1 ... NUM_TILES:
		GraySprite16_XOR_R(12 * x + 2, 12 * y + calc(2, 16), 11, sprites_light[type - 1],
		sprites_dark[type - 1], GrayGetPlane(LIGHT_PLANE), GrayGetPlane(DARK_PLANE));
		break;

		case 8:
		GrayInvertRect(12 * x + 1, 12 * y + calc(1, 15), 12 * x + 13, 12 * y + calc(13, 27));
	}
}

/*postcond: you have finished, quit, or saved the game*/
char PlayGame(char board[][8], unsigned long *scorep, char mode, LCD_BUFFER lcd_buffers[])
{
	char buffer[18];
	char end; /*used to select which end of the hint to show*/
	char goOn = DEFAULT;
#ifndef AUTO_PLAY
	short x = 0;
	short y = 0;
#endif
	COMBO firstCombo = {NO_MOVE, 0, 0, 0}; /*When checking to see if any moves are left,
	this is set to the coordinates of the first move found.  This is used to provide a
	hint if the player gets stuck.*/
	COMBO lastMove = {NO_MOVE, 0, 0, 0}; /*This is used to store the coordinates of the
	last move so that it can be undone if it did not create a combination*/
	QUEUE q = {{}, 0, 0};

	sprintf(buffer, "Score: %lu", *scorep);
	GrayDrawStr(calc(160, 240) - DrawStrWidth(buffer, F_4x6), 0, buffer, A_NORMAL);

	DrawBonus(mode);

	do
	{
		if(FindMove(board, &firstCombo)) /*if a possible move was found*/
		{
			end = random_long(2);
			FreezeBonus = 0; /*start the clock*/
			do
			{
#ifdef AUTO_PLAY
				memcpy(&lastMove, &firstCombo, sizeof(COMBO));
#else
				goOn = GetMove(&x, &y, &lastMove, mode);
#endif

				if(goOn == HINT)
				{
					ShowHint(firstCombo, end, mode, scorep);
				}
			} while(goOn == HINT);
			FreezeBonus = 1; /*stop the clock*/
			if(lastMove.x1 != NO_MOVE)
			{
				goOn = HandleMove(board, &lastMove, &q, goOn, scorep,
				mode, lcd_buffers);
			}
		}
		else
		{
			FastCopyScreen_R(GrayGetPlane(LIGHT_PLANE), lcd_buffers[1]);
			FastCopyScreen_R(GrayGetPlane(DARK_PLANE), lcd_buffers[2]);

			FontSetSys(F_8x10);
			GrayDrawRect(21, calc(39, 53), 77, calc(59, 63), COLOR_WHITE, RECT_FILLED);
			GrayDrawStr(21, calc(40, 54), (char *)"NO MORE", A_NORMAL);
			GrayDrawStr(29, calc(50, 64), (char *)"MOVES", A_NORMAL);
			FontSetSys(F_4x6);

			if(mode == NORMAL)
			{
				GrayDrawRect(9, calc(93, 121), 89, calc(99, 127), COLOR_WHITE, RECT_FILLED);
				GrayDrawStr(10, calc(94, 122), (char *)"Press [2nd] to continue", A_NORMAL);
				Pause();
				FastCopyScreen_R(lcd_buffers[1], GrayGetPlane(LIGHT_PLANE));
				FastCopyScreen_R(lcd_buffers[2], GrayGetPlane(DARK_PLANE));
				goOn = PLAY_GAME;
			}
			else
			{
#ifndef AUTO_PLAY
				HW_VERSION == 1 ? SleepMS_HW1(SCORE_DELAY) : SleepMS_HW2(SCORE_DELAY);
#endif
				FastCopyScreen_R(lcd_buffers[1], GrayGetPlane(LIGHT_PLANE));
				FastCopyScreen_R(lcd_buffers[2], GrayGetPlane(DARK_PLANE));
				DisplayBoard(board);
				GenerateBoard(board);
				DisplayBoard(board);
			}
		}

		if(mode == TIMED && Bonus >= 6000)
		{
			HandleBonus(board);
			Gravity(board);
			BonusLevel++;
			Bonus -= 3000;
			DrawBonus(mode);
			lastMove.x1 = NO_MOVE;
			HandleMove(board, &lastMove, &q, goOn, scorep, mode, lcd_buffers);
		}

		if(mode == NORMAL && Bonus >= 100 * (BonusLevel - 1))
		{
			HandleBonus(board);
			Gravity(board);
			Bonus -= 100 * (BonusLevel - 1);
			BonusLevel++;
			DrawBonus(mode);
			lastMove.x1 = NO_MOVE;
			HandleMove(board, &lastMove, &q, goOn, scorep, mode, lcd_buffers);
		}
	} while(goOn == DEFAULT); /*loop until [ESC] or [CLEAR] pressed*/

	return(goOn);
}

void CheckHighScores(char mode, unsigned long score, unsigned long highScores[][2])
{
	char buffer[12];
	short col = (mode == NORMAL ? 0 : 1);
	short i;
	short j;
	static const char *ordinal[] = {"1ST", "2ND", "3RD", "4TH", "5TH", "6TH", "7TH"};

	for(i = 0; i < 7; i++)
	{
		if(score > highScores[i][col])
		{
			FontSetSys(F_8x10);
			GrayDrawRect(13, calc(39, 53), 85, calc(59, 73), COLOR_WHITE, RECT_FILLED);
			GrayDrawStr(21, calc(40, 54), (char *)"YOU GOT", A_NORMAL);
			sprintf(buffer, "%s PLACE", ordinal[i]);
			GrayDrawStr(13, calc(50, 64), buffer, A_NORMAL);
			FontSetSys(F_4x6);
			GrayDrawRect(9, calc(93, 121), 89, calc(99, 127), COLOR_WHITE, RECT_FILLED);
			GrayDrawStr(10, calc(94, 122), (char *)"Press [2nd] to continue", A_NORMAL);
			Pause();
			for(j = 6; j > i; j--)
			{
				highScores[j][col] = highScores[j - 1][col];
			}
			highScores[i][col] = score;
			break;
		}
	}
}

char FindMove(char board[][8], COMBO *firstCombop)
{
	char moveFound = 0;
	short i;
	short j;
	COMBO tempCombo;

	for(i = 0; i < 8; i++)
	{
		for(j = 0; j < 7; j++)
		{
			tempCombo.x1 = tempCombo.x2 = i;
			tempCombo.y2 = (tempCombo.y1 = j) + 1;
			SwapPieces(board, tempCombo, 0);
			moveFound = CheckCombo(board, tempCombo);
			SwapPieces(board, tempCombo, 0);

			if(moveFound)
			{
				memcpy(firstCombop, &tempCombo, sizeof(COMBO));
				break;
			}

			tempCombo.x2 = (tempCombo.x1 = j) + 1;
			tempCombo.y1 = tempCombo.y2 = i;
			SwapPieces(board, tempCombo, 0);
			moveFound = CheckCombo(board, tempCombo);
			SwapPieces(board, tempCombo, 0);

			if(moveFound)
			{
				memcpy(firstCombop, &tempCombo, sizeof(COMBO));
				break;
			}
		}

		if(moveFound)
		{
			break;
		}
	}

	return(moveFound);
}

/*returns 1 if there was a combo created at the spot indicated by the argument*/
char CheckCombo(char board[][8], COMBO c)
{
	short x;
	short y;

	if(c.x1 == c.x2) /*if it is a vertical move*/
	{
		/*check the possible vertical combinations*/
		for(x = c.x1, y = max(min(c.y1, c.y2) - 2, 0); y <= min(max(c.y1, c.y2), 5); y++)
		{
			if(board[y][x] == board[y + 1][x] && board[y][x] == board[y + 2][x])
			{
				return(1);
			}
		}
		/*check the possible horizontal combinations*/
		for(y = min(c.y1, c.y2); y <= max(c.y1, c.y2); y++)
		{
			for(x = max(c.x1 - 2, 0); x <= min(c.x1, 5); x++)
			{
				if(board[y][x] == board[y][x + 1] && board[y][x] == board[y][x + 2])
				{
					return(1);
				}
			}
		}
	}
	else /*it is a horizontal move*/
	{
		/*check the possible horizontal combinations*/
		for(y = c.y1, x = max(min(c.x1, c.x2) - 2, 0); x <= min(max(c.x1, c.x2), 5); x++)
		{
			if(board[y][x] == board[y][x + 1] && board[y][x] == board[y][x + 2])
			{
				return(1);
			}
		}
		/*check the possible vertical combinations*/
		for(x = min(c.x1, c.x2); x <= max(c.x1, c.x2); x++)
		{
			for(y = max(c.y1 - 2, 0); y <= min(c.y1, 5); y++)
			{
				if(board[y][x] == board[y + 1][x] && board[y][x] == board[y + 2][x])
				{
					return(1);
				}
			}
		}
	}
	return(0);
}

/*gets user input*/
char GetMove(short *xp, short *yp, COMBO *lastMovep, char mode)
{
	char blink = 1; /*used to tell if the cursor is on or off in the blinking routine*/
	char goOn = DEFAULT; /*used to tell main if [ESC] or [CLEAR] was pressed*/
	char second = 0; /*indicates piece movement (1) versus cursor movement (0) mode*/
	unsigned char key; /*used to store rowread values*/

	DrawSprite(*xp, *yp, CURSOR); /*draw the cursor*/

	do
	{
		key = _rowread(0xFFFE); /*get the arrow row*/

		if(key & calc(15, 240)) /*if any arrow key pressed*/
		{
			DrawSprite(*xp, *yp, CURSOR); /*erase the cursor*/
		}
		if(key & calc(1, 32)) /*if [Up] pressed*/
		{
			while(_rowread(0xFFFE) & calc(1, 32)); /*wait until [Up] released*/

			if(second)
			{
				if(*yp > 0)
				{
					if(blink == 1)
					{
						DrawSprite(*xp, *yp, CURSOR); /*draw the cursor*/
					}

					/*fill in the move struct*/
					lastMovep->x1 = *xp;
					lastMovep->y1 = *yp;
					lastMovep->x2 = *xp;
					lastMovep->y2 = *yp - 1;

					second = 0;
				}
			}
			else
			{
				*yp = Wrap(*yp - 1, 0, 7);
			}
		}
		if(key & calc(2, 16)) /*if [Left] pressed*/
		{
			while(_rowread(0xFFFE) & calc(2, 16)); /*wait until [Left] released*/

			if(second)
			{
				if(*xp > 0)
				{
					if(blink == 1)
					{
						DrawSprite(*xp, *yp, CURSOR); /*draw the cursor*/
					}

					/*fill in the move struct*/
					lastMovep->x1 = *xp;
					lastMovep->y1 = *yp;
					lastMovep->x2 = *xp - 1;
					lastMovep->y2 = *yp;

					second = 0;
				}
			}
			else
			{
				*xp = Wrap(*xp - 1, 0, 7);
			}
		}
		if(key & calc(4, 128)) /*if [Down] pressed*/
		{
			while(_rowread(0xFFFE) & calc(4, 128)); /*wait until [Down] released*/

			if(second)
			{
				if(*yp < 7)
				{
					if(blink == 1)
					{
						DrawSprite(*xp, *yp, CURSOR); /*draw the cursor*/
					}

					/*fill in the move struct*/
					lastMovep->x1 = *xp;
					lastMovep->y1 = *yp;
					lastMovep->x2 = *xp;
					lastMovep->y2 = *yp + 1;

					second = 0;
				}
			}
			else
			{
				*yp = Wrap(*yp + 1, 0, 7);
			}
		}
		if(key & calc(8, 64)) /*if [Right] pressed*/
		{
			while(_rowread(0xFFFE) & calc(8, 64)); /*wait until [Right] released*/

			if(second)
			{
				if(*xp < 7)
				{
					if(blink == 1)
					{
						DrawSprite(*xp, *yp, CURSOR); /*draw the cursor*/
					}

					/*fill in the move struct*/
					lastMovep->x1 = *xp;
					lastMovep->y1 = *yp;
					lastMovep->x2 = *xp + 1;
					lastMovep->y2 = *yp;

					second = 0;
				}
			}
			else
			{
				*xp = Wrap(*xp + 1, 0, 7);
			}
		}
		if(key & calc(15, 240)) /*if any arrow key pressed*/
		{
			DrawSprite(*xp, *yp, CURSOR); /*put the cursor in its new location*/
		}
		if(key & calc(16, 1)) /*if [2nd] pressed*/
		{
			while(_rowread(0xFFFE) & calc(16, 1)); /*wait until [2nd] is released*/

			second ^= 1; /*switch between cursor movement and piece swapping*/
			Time += CURSOR_BLINK_DELAY;
			if(blink == 0)
			{
				DrawSprite(*xp, *yp, CURSOR); /*draw the cursor*/
			}
		}
		if(_rowread(calc(0xFFBF, 0xFEFF)) & calc(1, 64)) /*if [ESC] pressed*/
		{
			while(_rowread(calc(0xFFBF, 0xFEFF)) & calc(1, 64)); /*wait until [ESC] let
			up*/
			goOn = ESC; /*tell main that [ESC] was pressed*/
		}
		if(_rowread(calc(0xFFFD, 0xFF7F)) & calc(64, 32)) /*if [CLEAR] pressed*/
		{
			while(_rowread(calc(0xFFFD, 0xFF7F)) & calc(64, 32)); /*wait until [CLEAR] let
			up*/
			goOn = SAVE; /*tell main that [CLEAR] was pressed*/
		}
		if(_rowread(calc(0xFFDF, 0xFF7F)) & calc(1, 64)) /*if [APPS] pressed*/
		{
			while(_rowread(calc(0xFFDF, 0xFF7F)) & calc(1, 64));
			DrawSprite(*xp, *yp, CURSOR); /*draw the cursor*/
			goOn = HINT;
		}
		if(second == 1 && Time == 0)
		{
			blink ^= 1;
			DrawSprite(*xp, *yp, CURSOR); /*draw the cursor*/
			Time += CURSOR_BLINK_DELAY;
		}

		if(mode == TIMED)
		{
			DrawBonus(mode);

			if(Bonus <= 0)
			{
				FontSetSys(F_8x10);
				GrayDrawRect(21, calc(44, 58), 77, calc(54, 68), COLOR_WHITE, RECT_FILLED);
				GrayDrawStr(21, calc(45, 59), (char *)"TIME UP", A_NORMAL);
				FontSetSys(F_4x6);
				GrayDrawRect(9, calc(93, 121), 89, calc(99, 127), COLOR_WHITE, RECT_FILLED);
				GrayDrawStr(10, calc(94, 122), (char *)"Press [2nd] to continue", A_NORMAL);
				Pause();
				goOn = PLAY_GAME;
			}
		}
	} while(lastMovep->x1 == NO_MOVE && goOn == DEFAULT);
	return(goOn);
}

short Wrap(short input, short lower, short upper) /*take the input value and put it
between the lower and upper bound with wrapping around the ends*/
{
	return(input >= lower ? (input <= upper ? input : lower) : upper);
}

void DrawBonus(char mode)
{
	char buffer[13];
	unsigned short tempBonus = Bonus;

	if(mode == NORMAL)
	{
		GrayDrawRect(100, 0, 102, max(calc(100, 128) * (100 * (BonusLevel - 1) - (long)tempBonus) / (100 *
		(BonusLevel - 1)), 0), COLOR_WHITE, RECT_FILLED);
		GrayDrawRect(100, max(calc(100, 128) * (100 * (BonusLevel - 1) - (long)tempBonus) / (100 *
		(BonusLevel - 1)), 0), 102, calc(99, 127), COLOR_BLACK, RECT_FILLED);
	}
	else
	{
		GrayDrawRect(100, 0, 102, max((calc(6000L, 762000L) - (long)tempBonus) / calc(60, 6000), 0),
		COLOR_WHITE, RECT_FILLED);
		GrayDrawRect(100, max((calc(6000L, 762000L) - (long)tempBonus) / calc(60, 6000), 0), 102, calc(99,
		127), COLOR_BLACK, RECT_FILLED);
	}

	sprintf(buffer, "Bonus x%d", BonusLevel - 1);
	if(Bonus > 600 || mode == NORMAL)
	{
		GrayDrawStr(103, calc(95, 122), buffer, A_REPLACE);
	}
	else
	{
		if(Time2 == 0)
		{
			GrayDrawStr(103, calc(95, 122), buffer, A_XOR);
			Time2 += BONUS_BLINK_DELAY;
		}
	}

}

void ShowHint(COMBO hint, char end, char mode, unsigned long *scorep)
{
	short i;

	if(mode == NORMAL)
	{
		(*scorep) -= min((unsigned)(5 * BonusLevel), *scorep);
		DrawScore(*scorep);
	}

	for(i = 0; i < 4; i++)
	{
		Time += 5;
		if(end)
		{
			DrawSprite(hint.x1, hint.y1, CURSOR);
		}
		else
		{
			DrawSprite(hint.x2, hint.y2, CURSOR);
		}
		while(Time);
	}
}

char HandleMove(char board[][8], COMBO *lastMovep, QUEUE *q, char goOn, unsigned long *scorep, char mode,
LCD_BUFFER lcd_buffers[])
{
	char checkForCombos = 0; /*set to 1 to tell this routine that it needs to check for
	new combos*/
	char multiplier; /*used to tell if you got two combos at once or not*/
	short comboString = 0; /*used to keep track of how many combos have happened in a
	row*/
	short i;
	unsigned long tempScore; /*a temporary storage area to catch the return from DrawPoints*/
	COMBO temp;

	if(lastMovep->x1 != NO_MOVE)
	{
		SwapPieces(board, *lastMovep, 1);

		if(!CheckCombo(board, *lastMovep)) /*if this move did not create any combos*/
		{
			SwapPieces(board, *lastMovep, 1); /*undo the move*/
			FastCopyScreen_R(GrayGetPlane(LIGHT_PLANE), lcd_buffers[3]);
			FastCopyScreen_R(GrayGetPlane(DARK_PLANE), lcd_buffers[4]);
			GrayDrawRect(26, calc(45, 59), 71, calc(53, 67), COLOR_WHITE, RECT_FILLED);
			GrayDrawStr(27, calc(47, 61), (char *)"Invalid Move", A_NORMAL);
			GrayDrawRect(9, calc(93, 121), 89, calc(99, 127), COLOR_WHITE, RECT_FILLED);
			GrayDrawStr(10, calc(94, 122), (char *)"Press [2nd] to continue", A_NORMAL);
			Pause();
			FastCopyScreen_R(lcd_buffers[3], GrayGetPlane(LIGHT_PLANE));
			FastCopyScreen_R(lcd_buffers[4], GrayGetPlane(DARK_PLANE));
		}
		else
		{
			checkForCombos = 2;
		}
	}
	else
	{
		checkForCombos = 1;
	}

	if(checkForCombos)
	{
		multiplier = 0;

		for(i = 0; i < 8; i++) /*put all the currently existing combinations on the
		stack*/
		{
			CheckRow(board, i, q, &multiplier);
			CheckCol(board, i, q, &multiplier);
		}

		if(checkForCombos != 2) /*if you did not just do a move*/
		{
			multiplier = 1;
		}

		if(q->front != q->back)
		{
			do
			{
				do
				{
					temp = q->data[q->front++];
					if(q->front >= QUEUE_SIZE)
					{
						q->front = 0;
					}

					RemoveCombo(board, temp);

					tempScore = DrawPoints(temp, comboString, mode,
					multiplier, lcd_buffers);

					*scorep += tempScore; /*add some points to the score*/
					DrawScore(*scorep);
					comboString++;
					DrawBonus(mode);
				} while(q->front != q->back);

				multiplier = 1;

				Gravity(board); /*Shift down any "hanging" pieces*/

				for(i = 0; i < 8; i++) /*check to see if the pieces falling down created
				any new combinations*/
				{
					CheckRow(board, i, q, NULL);
					CheckCol(board, i, q, NULL);
				}
			} while(q->front != q->back);
		}
	}

	lastMovep->x1 = NO_MOVE;

	return(goOn);
}

void SwapPieces(char board[][8], COMBO lastMove, char display)
{
	if(display)
	{
		DrawSprite(lastMove.x1, lastMove.y1, 0);
		DrawSprite(lastMove.x2, lastMove.y2, 0);
	}

	board[lastMove.y1][lastMove.x1] ^= board[lastMove.y2][lastMove.x2] ^=
	board[lastMove.y1][lastMove.x1] ^= board[lastMove.y2][lastMove.x2];

	if(display)
	{
		DrawSprite(lastMove.x1, lastMove.y1, board[lastMove.y1][lastMove.x1]);
		DrawSprite(lastMove.x2, lastMove.y2, board[lastMove.y2][lastMove.x2]);
	}
}

/*precond: 0 <= row <= 7. postcond: any combinations found are pushed onto the stack.  returns if a
combintaion was found*/
char CheckRow(char board[][8], short row, QUEUE *q, char *combop)
{
	char comboFound = 0;
	short i;
	short j;
	COMBO temp;

	for(i = 0; i <= 5; i++) /*combinations have to be at least three pieces long, so no
	combos start after five*/
	{
		for(j = 1; j <= NUM_TILES; j++) /*try all the piece types*/
		{
			if(board[row][i] == j && board[row][i + 1] == j && board[row][i + 2] == j)
			/*if the first three squares including the current one match the current
			type*/
			{
				comboFound = 1;
				if(combop != NULL)
				{
					(*combop)++;
				}
				temp.x1 = i;
				temp.y1 = temp.y2 = row;
				for(temp.x2 = i + 2; temp.x2 < 7 && board[row][temp.x2 + 1] == j; temp.x2++); /*if any
				more congruent sqares on the same row after the three match the current type, add them to
				the combination also*/
				i = temp.x2; /*skip to the end of the combo*/
				q->data[q->back++] = temp;
				if(q->back >= QUEUE_SIZE)
				{
					q->back = 0;
				}
				break;
			}
		}
	}
	return(comboFound);
}

/*precond: 0 <= col <= 7. postcond: if comboList != NULL and lastCombop != NULL, any
combinations found are pushed onto comboList.  returns if a combintaion was found*/
char CheckCol(char board[][8], short col, QUEUE *q, char *combop)
{
	char comboFound = 0;
	short i;
	short j;
	COMBO temp;

	for(i = 0; i <= 5; i++) /*combinations have to be at least three pieces long, so no
	combos start after five*/
	{
		for(j = 1; j <= NUM_TILES; j++) /*try all the piece types*/
		{
			if(board[i][col] == j && board[i + 1][col] == j && board[i + 2][col] == j)
			/*if the first three squares including the current one match the current
			type*/
			{
				comboFound = 1;
				if(combop != NULL)
				{
					(*combop)++;
				}
				temp.y1 = i;
				temp.x1 = temp.x2 = col;
				for(temp.y2 = i + 2; temp.y2 < 7 && board[temp.y2 + 1][col] == j; temp.y2++); /*if any
				more congruent sqares on the same column after the three match the current type, add them
				to the combination also*/
				i = temp.y2; /*skip to the end of the combo*/
				q->data[q->back++] = temp;
				if(q->back >= QUEUE_SIZE)
				{
					q->back = 0;
				}
				break;
			}
		}
	}
	return(comboFound);
}

void Pause(void)
{
	while(_rowread(0xFFFE) & calc(16, 1)); /*wait until [2nd] is released*/
	while(!(_rowread(0xFFFE) & calc(16, 1))); /*wait until [2nd] is pressed*/
	while(_rowread(0xFFFE) & calc(16, 1)); /*wait until [2nd] is released*/
}

unsigned long DrawPoints(COMBO combo, short comboString, char mode, char multiplier, LCD_BUFFER
lcd_buffers[])
{
	char buffer[11];
	unsigned char x, y;
	unsigned long score;

	FastCopyScreen_R(GrayGetPlane(LIGHT_PLANE), lcd_buffers[5]);
	FastCopyScreen_R(GrayGetPlane(DARK_PLANE), lcd_buffers[6]);

	if(combo.x1 == combo.x2) /*if it was a vertical combo*/
	{
		score = 2 * (abs(combo.y1 - combo.y2) - 1) * (comboString + 1) * (mode == NORMAL ? 1 : 2) *
		BonusLevel * multiplier / 2;
		Bonus += (mode == NORMAL ? score : (unsigned)(100 * (abs(combo.y1 - combo.y2) - 1) *
		(comboString + 1) * multiplier));
		sprintf(buffer, "%lu", score);
		x = min(max(12 * combo.x1 - DrawStrWidth(buffer, F_4x6) / 2 + 7, 2), 98 - DrawStrWidth(buffer,
		F_4x6));
		y = 6 * (combo.y1 + combo.y2) + calc(5, 19);
	}
	else
	{
		score = 2 * (abs(combo.x1 - combo.x2) - 1) * (comboString  + 1) * (mode == NORMAL ? 1 : 2) *
		BonusLevel * multiplier / 2;
		Bonus += (mode == NORMAL ? score : (unsigned)(100 * (abs(combo.x1 - combo.x2) - 1) *
		(comboString + 1) * multiplier));
		sprintf(buffer, "%lu", score);
		x = min(max(6 * (combo.x1 + combo.x2) - DrawStrWidth(buffer, F_4x6) / 2 + 7, 2), 98 -
		DrawStrWidth(buffer, F_4x6));
		y = 12 * combo.y1 + calc(5, 19);
	}

	GrayDrawStr(x, y, buffer, A_REPLACE);

#ifndef AUTO_PLAY
	HW_VERSION == 1 ? SleepMS_HW1(SCORE_DELAY) : SleepMS_HW2(SCORE_DELAY);
#endif

	FastCopyScreen_R(lcd_buffers[5], GrayGetPlane(LIGHT_PLANE));
	FastCopyScreen_R(lcd_buffers[6], GrayGetPlane(DARK_PLANE));

	return(score);
}

void DrawScore(unsigned long score)
{
	char buffer[18];

	sprintf(buffer, "Score: %lu", score);
	GrayDrawStr(calc(160, 240) - DrawStrWidth(buffer, F_4x6), 0, buffer, A_REPLACE);
}

void RemoveCombo(char board[][8], COMBO combo)
{
	short i;
	short j;

	for(i = min(combo.x1, combo.x2); i <= max(combo.x1, combo.x2); i++)
	{
		for(j = min(combo.y1, combo.y2); j <= max(combo.y1, combo.y2); j++)
		{
			board[j][i] = 0;
			DrawSprite(i, j, 0);
		}
	}
}

/*postcond: any pieces "hanging" above empty space are shifted down and the reulting empty
space is filled in with new random pieces*/
void Gravity(char board[][8])
{
	short sprite_buffer[2][8][8 * 12]; /*two planes, 8 sprites wide, 8 sprites tall, and 12 pixels per
	sprite*/
	short empty[8]; /*stores the lowest empty square.  is -1 if no empty squares in that column*/
	short full[8]; /*stores the lowest full square above the empty square.*/
	short i;
	short j;

	memset(sprite_buffer, 0, sizeof(sprite_buffer));
	for(;;)
	{
		for(i = 0; i < 8; i++) /*iterate through the columns*/
		{
			for(empty[i] = 7; empty[i] >= 0; empty[i]--) /*start at the bottom and look for empty
			squares*/
			{
				if(board[empty[i]][i] == 0)
				{
					break;
				}
			}

			for(full[i] = empty[i]; full[i] >= 0; full[i]--) /*start at the empty square and see if there
			is anything hanging above it*/
			{
				if(board[full[i]][i] != 0)
				{
					break;
				}
			}
		}

		if(!memcmp(empty, full, 8 * sizeof(short)))
		{
			break;
		}

		for(i = 0; i < 8; i++) /*iterate through the columns*/
		{
			if(full[i] != -1 || empty[i] != -1) /*if there are hanging pieces in this column or the top
			needs to be filled in*/
			{
				/*the most that will ever be gotten is seven squares, but I need the extra square for
				having the new piece fall in*/
				Sprite16Get(12 * i + 2, calc(2, 16), 12 * empty[i], GrayGetPlane(LIGHT_PLANE),
				sprite_buffer[0][i] + 12);
				Sprite16Get(12 * i + 2, calc(2, 16), 12 * empty[i], GrayGetPlane(DARK_PLANE),
				sprite_buffer[1][i] + 12);

				for(j = 0; j < 12 * empty[i]; j++)
				{
					/*we only want the first eleven pixels*/
					sprite_buffer[0][i][j + 12] &= 0xFFE0;
					sprite_buffer[1][i][j + 12] &= 0xFFE0;
				}

				for(j = empty[i]; j > 0; j--) /*shift the board data down*/
				{
					board[j][i] = board[j - 1][i];
				}

				board[0][i] = random_long(NUM_TILES) + 1; /*put a new random square at the top of the
				column*/

				/*put the graphics for the new square into the buffer*/
				memcpy(sprite_buffer[0][i], sprites_light[(short)board[0][i] - 1], 11 * sizeof(short));
				memcpy(sprite_buffer[1][i], sprites_dark[(short)board[0][i] - 1], 11 * sizeof(short));
			}
		}

		for(i = 0; i < 12; i++) /*shift the sprites down*/
		{
			for(j = 0; j < 8; j++) /*iterate through the columns*/
			{
				if(full[j] != -1 || empty[j] != -1) /*if there are hanging pieces in this column or the
				top needs to be filled in*/
				{
					GraySprite16_XOR_R(12 * j + 2, calc(2, 16), 12 * empty[j] + i,
					&sprite_buffer[0][j][12 - i], &sprite_buffer[1][j][12 - i], GrayGetPlane(LIGHT_PLANE),
					GrayGetPlane(DARK_PLANE)); /*erase the column above the empty space*/
					GraySprite16_XOR_R(12 * j + 2, calc(2, 16), 12 * empty[j] + i + 1,
					&sprite_buffer[0][j][11 - i], &sprite_buffer[1][j][11 - i], GrayGetPlane(LIGHT_PLANE),
					GrayGetPlane(DARK_PLANE)); /*move the erased squares one down*/
				}
			}

#ifndef AUTO_PLAY
			HW_VERSION == 1 ? SleepMS_HW1(DROP_DELAY) : SleepMS_HW2(DROP_DELAY); /*slow the animation down
			to visible speed*/
#endif
		}
	}
}

void HandleBonus(char board[][8])
{
	short i;
	short x;
	short y;

	for(i = 0; i < 16; i++)
	{
		do
		{
			x = random_long(8);
			y = random_long(8);
		} while(board[y][x] == 0);

		DrawSprite(x, y, board[y][x]);
		board[y][x] = 0;
	}
}