/*
Copyright (C) 1998, 1999, 2000 Wabasoft

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free Software
Foundation; either version 2 of the License, or (at your option) any later version. 

This program is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
PARTICULAR PURPOSE. See the GNU General Public License for more details. 

You should have received a copy of the GNU General Public License along with
this program; if not, write to the Free Software Foundation, Inc., 675 Mass Ave,
Cambridge, MA 02139, USA. 
*/

static WObject globalMainWin = 0;	// for WabaVM
static WINDOW mainWindow;	// for TIOS
static char cmdLine[80];

/* Calls the _doPaint method of the main window.
 */
static void drawMainWin()
{
    WObject mainWinObj;
    WClass *vclass;
    WClassMethod *method;
    Var params[7];

    mainWinObj = globalMainWin;
    if (!mainWinObj)
	return;
    vclass = WOBJ_class(mainWinObj);	// get runtime class
    method = getMethod(vclass, createUtfString("_doPaint"),
		       createUtfString("(IIII)V"), &vclass);
    if (method == NULL)
	return;
    params[0].obj = mainWinObj;
    params[1].intValue = 0;	// x
    params[2].intValue = 0;	// y
    params[3].intValue = 160;	// width
    params[4].intValue = 100;	// height
    executeMethod(vclass, method, params, 5);
}

static int pointerX = 0, pointerY = 0;
static unsigned char *pointerBufAddress;
static unsigned long pointerBuf[8];
static unsigned char pointer[] =
    { 0x80, 0xc0, 0xa0, 0x90, 0x88, 0xbc, 0xc0, 0x80 };

static inline void unbufferPointer()
{
    pointerBufAddress = NULL;
}

static void bufferPointer()
{
    int n;
    pointerBufAddress = (unsigned char *) LCD_MEM;
    pointerBufAddress += 30 * pointerY;
    pointerBufAddress += (pointerX >> 3) & 0xfe;
    for (n = 0; n < 8; ++n) {
	pointerBuf[n] = *(unsigned long *) (pointerBufAddress + 30 * n);
    }
}

static void undrawPointer()
{
    int i;
    if (pointerBufAddress != NULL) {
	unsigned char *dest = pointerBufAddress;
	for (i = 0; i < 8; ++i) {
	    *(unsigned long *) dest = pointerBuf[i];
	    dest += 30;
	}
    }
}

static void drawPointer()
{
    undrawPointer();
    bufferPointer();
    Sprite8(pointerX, pointerY, 8, pointer, LCD_MEM, SPRT_OR);
}

/* An error window.
 */
static void drawErrorWin()
{
    char *msg;
    WINDOW w;

    WinOpen(&w, MakeWinRect(20, 10, 140, 90), WF_SAVE_SCR | WF_TTY);
    WinActivate(&w);
    WinFont(&w, F_6x8);

    WinStr(&w, "WabaVM Error!\n");
    msg = errorMessages[vmStatus.errNum - 1];
    WinStr(&w, msg);
    WinChar(&w, '\n');
    msg = vmStatus.arg1;
    if (msg[0]) {
	WinStr(&w, msg);
	WinChar(&w, '\n');
    }
    msg = vmStatus.arg2;
    if (msg[0]) {
	WinStr(&w, msg);
	WinChar(&w, '\n');
    }
    if (vmStatus.className[0] || vmStatus.methodName[0]) {
	WinStr(&w, "In: ");
	msg = vmStatus.className;
	WinStr(&w, msg);
	WinChar(&w, '\n');
	msg = vmStatus.methodName;
	WinStr(&w, msg);
	WinChar(&w, '\n');
    }
    if (vmStatus.errNum == ERR_CantAccessCoreClasses) {
	WinStr(&w, "Can't access core classes.\n");
    }

    ngetchx();
    WinClose(&w);
}

/* Search the folder 'wabacore' for classes and return the pointer to
 * the beginning of the byte code or NULL if not found.
 *
 * Class strings have this format:
 *   2 bytes	string length - 2 (TIOS specific)
 *   1 byte	0 (TIOS specific)
 *   1 byte	l: length of class name
 *   l bytes	class name (without .class)
 *   c bytes	the actual class (the rest of the string)
 *   1 byte	ESI-tag, 0x2d for string (TIOS specific)
 *
 * The name of the string is of no interest, it's way too short to hold
 * the class name.
 */
static uchar *lockClassMem(char *className, uint16 nameLen, uint32 * size)
{
    SYM_ENTRY *sym;
    int first;
    uchar *p;
    uchar type;

    first = 1;
    while (1) {
	if (first) {
	    sym = SymFindFirst($(wabacore), 1);
	    first = 0;
	} else {
	    sym = SymFindNext();
	}
	if (sym == NULL) {
	    return NULL;
	}
	type = *(uchar *) HToESI(sym->handle);
	if (type != STR_TAG) {
	    continue;
	}
	p = (uchar *) HeapDeref(sym->handle);
	if ((*(uchar *) (p + 3) == nameLen)
	    && (xstrncmp(p + 4, className, nameLen) == 0)) {
	    HeapLock(sym->handle);
	    if (size != NULL) {
		*size = *(unsigned short *) p + 2 - nameLen - 1;	// size of the class
	    }
	    return (p + 3 + nameLen + 1);	// where the class starts
	}
    }
}

/* Unlock the heap for a class.
 * ptr points to the beginning of the byte code representaion of the class,
 * backwards calculation gives us the pointer to the symbol on the heap.
 */
static void unlockClassMem(uchar * ptr, uint16 nameLen)
{
    HANDLE h;

    h = HeapPtrToHandle(ptr - nameLen - 1 - 3);
    HeapUnlock(h);
}

/* Starts the Application.
 * cmdline is prepared, but not fully implemented.
 * Starts waba/ui/Welcome by default.
 */
static WObject startApp(int runApp)
{
    char className[40];
    unsigned long vmStackSize, nmStackSize, classHeapSize, objectHeapSize;
    int i, j, n;

    // defaults
    vmStackSize = 1500;
    nmStackSize = 300;
    classHeapSize = 14000;
    objectHeapSize = 8000;

    if (!runApp) {
	// run "welcome" app (no app class specified)
	xstrncpy(cmdLine, "waba/ui/Welcome", 15);
	cmdLine[15] = 0;
    } else {
    }

    // parse options
    i = 0;
    while (cmdLine[i] == '/') {
	char c;
	unsigned long value;

	c = cmdLine[++i];
	if (!c)
	    break;
	if (cmdLine[++i] != ' ')
	    break;
	i++;
	value = 0;
	while (cmdLine[i] >= '0' && cmdLine[i] <= '9') {
	    value = (value * 10) + (cmdLine[i] - '0');
	    i++;
	}
	if (c == 'l')
	    classHeapSize = value;
	else if (c == 'm')
	    objectHeapSize = value;
	else if (c == 's')
	    vmStackSize = value;
	else if (c == 't')
	    nmStackSize = value;
	while (cmdLine[i] == ' ')
	    i++;
    }

    // parse class name
    j = 0;
    while (cmdLine[i] && cmdLine[i] != ' ')
	className[j++] = cmdLine[i++];
    className[j] = 0;

    VmInit(vmStackSize, nmStackSize, classHeapSize, objectHeapSize);
    return VmStartApp(className);
}

/* Stop the application and unlock heap.
 */
static void stopApp(WObject mainWinObj)
{
    WClass *wclass, *nextClass;
    uint32 i;

    VmStopApp(mainWinObj);
    if (vmInitialized)		// set by VmInit()
    {
	// unlock all record pointers
	for (i = 0; i < CLASS_HASH_SIZE; i++) {
	    wclass = classHashList[i];
	    while (wclass != NULL) {
		UtfString className;
		className = getUtfString(wclass, wclass->classNameIndex);
		nextClass = wclass->nextClass;
		unlockClassMem(wclass->byteRep, className.len);
		wclass = nextClass;
	    }
	}
	VmFree();
    }
}

/* Called by the WabaVM to load a class.
 * The actual search is performed in lockClassMem().
 */
static uchar *nativeLoadClass(UtfString className, uint32 * size)
{
    uchar *p;

    p = lockClassMem(className.str, className.len, size);
    if (p == NULL) {
	return NULL;
    }
    if (getUInt32(p) != (uint32) 0xCAFEBABE) {	// magic
	VmError(ERR_BadClass, NULL, &className, NULL);
	unlockClassMem(p, className.len);
	return NULL;
    }
    return p;
}

static short penDown;

void handlePenEvent(int);

static int handleKeyEvent()
{
    unsigned short key;
    void *kbd = kbd_queue();
    unsigned char statKeys;
    unsigned char keyUp, keyDown, keyLeft, keyRight;
    unsigned char key2nd, keyDiamond, keyShift;

    int updated = 0;

    if (!OSdequeue(&key, kbd)) {
	if (key == KEY_ESC) {
	    return 1;
	}
    }

    OSSetSR(0x0700);		// disable ints
    statKeys = _rowread(0);
    OSSetSR(0x0000);		// enable ints
    if (TI89) {
	keyUp = statKeys & 0x01;
	keyDown = statKeys & 0x04;
	keyLeft = statKeys & 0x02;
	keyRight = statKeys & 0x08;
	key2nd = statKeys & 0x10;
	keyDiamond = statKeys & 0x40;
	keyShift = statKeys & 0x20;
    } else {
	keyUp = statKeys & 0x20;
	keyDown = statKeys & 0x80;
	keyLeft = statKeys & 0x10;
	keyRight = statKeys & 0x40;
	key2nd = statKeys & 0x01;
	keyDiamond = statKeys & 0x02;
	keyShift = statKeys & 0x04;
    }
    undrawPointer();
    unbufferPointer();
    if (keyLeft && pointerX > 0) {
	--pointerX;
	handlePenEvent(201);	// pen move
	updated = 1;
    }
    if (keyRight && pointerX < 160) {	// should be LCD_WIDHT
	++pointerX;
	handlePenEvent(201);	// pen move
	updated = 1;
    }
    if (keyUp && pointerY > 0) {
	--pointerY;
	handlePenEvent(201);	// pen move
	updated = 1;
    }
    if (keyDown && pointerY < 100) {	// should be LCD_HEIGHT
	++pointerY;
	handlePenEvent(201);	// pen move
	updated = 1;
    }
    if (keyDiamond && !penDown) {
	penDown = 1;
	handlePenEvent(200);	// pen down
	//unbufferPointer();
	//drawMainWin();
	updated = 1;
    }
    if (!keyDiamond && penDown) {
	penDown = 0;
	handlePenEvent(202);	// pen up
	//unbufferPointer();
	//drawMainWin();
	updated = 1;
    }
    drawPointer();
    return 0;
}

void handlePenEvent(int type)
{
    WClass *vclass;
    WClassMethod *method;
    Var params[7];


    vclass = WOBJ_class(globalMainWin);	// get runtime class
    method = getMethod(vclass, createUtfString("_postEvent"),
		       createUtfString("(IIIIII)V"), &vclass);
    if (method != NULL) {
	params[0].obj = globalMainWin;
	params[1].intValue = type;	// type
	params[2].intValue = 0;	// key
	params[3].intValue = pointerX;	// x
	params[4].intValue = pointerY;	// y
	params[5].intValue = 0;	// modifiers
	params[6].intValue = 0;	// timeStamp
	executeMethod(vclass, method, params, 7);
    }
}

// put ESC into the keyboard queue to stop the app
static void postStopEvent()
{
    void *kbd = kbd_queue();
    OSenqueue(KEY_ESC, kbd);
}

void _main()
{
    WObject mainWinObj;
    int runApp;
    WClass *vclass;
    WClassMethod *method;
    Var params[5];

    unsigned short key;
    void *kbd = kbd_queue();
    unsigned short escape;

    unsigned long oldTime, newTime;

    ESI argptr;
    int argtype;

    pointerBufAddress = NULL;
    penDown = 0;
    runApp = 0;

    clrscr();

    if (ArgCount() != 0) {
	InitArgPtr(argptr);
	argtype = GetArgType(argptr);
	if (argtype == STR_TAG) {
	    strcpy(cmdLine, GetStrnArg(argptr));
	    runApp = 1;
	}
    }

    mainWinObj = startApp(runApp);

    if (mainWinObj == 0) {
	VmQuickError(ERR_CantAccessAppClasses);
    }

    if (vmStatus.errNum == 0) {
	drawMainWin();
	drawPointer();
    } else {
	drawErrorWin();
	goto leadout;
    }

    // rudimentary timer, not really needed here
    // loop lasts for 100 seconds
    OSFreeTimer(USER_TIMER);
    OSRegisterTimer(USER_TIMER, 2000);

    while (1) {
	if (OSTimerExpired(USER_TIMER)) {
	    break;
	}
	newTime = OSTimerCurVal(USER_TIMER);
	if (newTime == oldTime) {
	    continue;
	}
	oldTime = newTime;
	escape = handleKeyEvent();
	if (escape) {
	    break;
	}
    }

  leadout:
    stopApp(mainWinObj);
    if (globalMainWin != 0) {
	WinClose(&mainWindow);
    }
}
