/*
** $Id: lua.h,v 1.175b 2003/03/18 12:31:39 roberto Exp $
** Lua - An Extensible Extension Language
** Tecgraf: Computer Graphics Technology Group, PUC-Rio, Brazil
** http://www.lua.org	mailto:info@lua.org
** See Copyright Notice at the end of this file
*/


#ifndef lua_h
#define lua_h

#include <stdarg.h>
#include <stddef.h>


#define LUA_VERSION	"Lua 5.0.2"
#define LUA_COPYRIGHT	"Copyright (C) 1994-2004 Tecgraf, PUC-Rio"
#define LUA_AUTHORS 	"R. Ierusalimschy, L. H. de Figueiredo & W. Celes"



/* option for multiple returns in `lua_pcall' and `lua_call' */
#define LUA_MULTRET	(-1)


/*
** pseudo-indices
*/
#define LUA_REGISTRYINDEX	(-10000)
#define LUA_GLOBALSINDEX	(-10001)
#define lua_upvalueindex(i)	(LUA_GLOBALSINDEX-(i))


/* error codes for `lua_load' and `lua_pcall' */
#define LUA_ERRRUN	1
#define LUA_ERRFILE	2
#define LUA_ERRSYNTAX	3
#define LUA_ERRMEM	4
#define LUA_ERRERR	5


typedef struct lua_State lua_State;

typedef int (*lua_CFunction) (lua_State *L);


/*
** functions that read/write blocks when loading/dumping Lua chunks
*/
typedef const char * (*lua_Chunkreader) (lua_State *L, void *ud, size_t *sz);

typedef int (*lua_Chunkwriter) (lua_State *L, const void* p,
                                size_t sz, void* ud);


/*
** basic types
*/
#define LUA_TNONE	(-1)

#define LUA_TNIL	0
#define LUA_TBOOLEAN	1
#define LUA_TLIGHTUSERDATA	2
#define LUA_TNUMBER	3
#define LUA_TSTRING	4
#define LUA_TTABLE	5
#define LUA_TFUNCTION	6
#define LUA_TUSERDATA	7
#define LUA_TTHREAD	8


/* minimum Lua stack available to a C function */
#define LUA_MINSTACK	20


/*
** generic extra include file
*/
#ifdef LUA_USER_H
#include LUA_USER_H
#endif


/* type of numbers in Lua */
#ifndef LUA_NUMBER
typedef double lua_Number;
#else
typedef LUA_NUMBER lua_Number;
#endif


/* mark for all API functions */
#ifndef LUA_API
#define LUA_API		extern
#endif

/*
** {======================================================================
** Debug API
** =======================================================================
*/


/*
** Event codes
*/
#define LUA_HOOKCALL	0
#define LUA_HOOKRET	1
#define LUA_HOOKLINE	2
#define LUA_HOOKCOUNT	3
#define LUA_HOOKTAILRET 4


/*
** Event masks
*/
#define LUA_MASKCALL	(1 << LUA_HOOKCALL)
#define LUA_MASKRET	(1 << LUA_HOOKRET)
#define LUA_MASKLINE	(1 << LUA_HOOKLINE)
#define LUA_MASKCOUNT	(1 << LUA_HOOKCOUNT)

typedef struct lua_Debug lua_Debug;  /* activation record */

typedef void (*lua_Hook) (lua_State *L, lua_Debug *ar);


#define LUA_IDSIZE	60

struct lua_Debug {
  int event;
  const char *name;	/* (n) */
  const char *namewhat;	/* (n) `global', `local', `field', `method' */
  const char *what;	/* (S) `Lua', `C', `main', `tail' */
  const char *source;	/* (S) */
  int currentline;	/* (l) */
  int nups;		/* (u) number of upvalues */
  int linedefined;	/* (S) */
  char short_src[LUA_IDSIZE]; /* (S) */
  /* private part */
  int i_ci;  /* active function */
};

#define lua_checkstack liblua__0001
LUA_API int liblua__0001 (lua_State *L, int size);

#define lua_xmove liblua__0002
LUA_API void liblua__0002 (lua_State *from, lua_State *to, int n);

#define lua_atpanic liblua__0003
LUA_API lua_CFunction liblua__0003 (lua_State *L, lua_CFunction panicf);

#define lua_newthread liblua__0004
LUA_API lua_State *liblua__0004 (lua_State *L);

#define lua_gettop liblua__0005
LUA_API int liblua__0005 (lua_State *L);

#define lua_settop liblua__0006
LUA_API void liblua__0006 (lua_State *L, int idx);

#define lua_remove liblua__0007
LUA_API void liblua__0007 (lua_State *L, int idx);

#define lua_insert liblua__0008
LUA_API void liblua__0008 (lua_State *L, int idx);

#define lua_replace liblua__0009
LUA_API void liblua__0009 (lua_State *L, int idx);

#define lua_pushvalue liblua__0010
LUA_API void liblua__0010 (lua_State *L, int idx);

#define lua_type liblua__0011
LUA_API int liblua__0011 (lua_State *L, int idx);

#define lua_typename liblua__0012
LUA_API const char *liblua__0012 (lua_State *L, int t);

#define lua_iscfunction liblua__0013
LUA_API int liblua__0013 (lua_State *L, int idx);

#define lua_isnumber liblua__0014
LUA_API int liblua__0014 (lua_State *L, int idx);

#define lua_isstring liblua__0015
LUA_API int liblua__0015 (lua_State *L, int idx);

#define lua_isuserdata liblua__0016
LUA_API int liblua__0016 (lua_State *L, int idx);

#define lua_rawequal liblua__0017
LUA_API int liblua__0017 (lua_State *L, int index1, int index2);

#define lua_equal liblua__0018
LUA_API int liblua__0018 (lua_State *L, int index1, int index2);

#define lua_lessthan liblua__0019
LUA_API int liblua__0019 (lua_State *L, int index1, int index2);

#define lua_tonumber liblua__0020
LUA_API lua_Number liblua__0020 (lua_State *L, int idx);

#define lua_toboolean liblua__0021
LUA_API int liblua__0021 (lua_State *L, int idx);

#define lua_tostring liblua__0022
LUA_API const char *liblua__0022 (lua_State *L, int idx);

#define lua_strlen liblua__0023
LUA_API size_t liblua__0023 (lua_State *L, int idx);

#define lua_tocfunction liblua__0024
LUA_API lua_CFunction liblua__0024 (lua_State *L, int idx);

#define lua_touserdata liblua__0025
LUA_API void *liblua__0025 (lua_State *L, int idx);

#define lua_tothread liblua__0026
LUA_API lua_State *liblua__0026 (lua_State *L, int idx);

#define lua_topointer liblua__0027
LUA_API const void *liblua__0027 (lua_State *L, int idx);

#define lua_pushnil liblua__0028
LUA_API void liblua__0028 (lua_State *L);

#define lua_pushnumber liblua__0029
LUA_API void liblua__0029 (lua_State *L, lua_Number n);

#define lua_pushlstring liblua__0030
LUA_API void liblua__0030 (lua_State *L, const char *s, size_t len);

#define lua_pushstring liblua__0031
LUA_API void liblua__0031 (lua_State *L, const char *s);

#define lua_pushvfstring liblua__0032
LUA_API const char *liblua__0032 (lua_State *L, const char *fmt, va_list argp);

#define lua_pushfstring liblua__0033
LUA_API const char *liblua__0033 (lua_State *L, const char *fmt, ...);

#define lua_pushcclosure liblua__0034
LUA_API void liblua__0034 (lua_State *L, lua_CFunction fn, int n);

#define lua_pushboolean liblua__0035
LUA_API void liblua__0035 (lua_State *L, int b);

#define lua_pushlightuserdata liblua__0036
LUA_API void liblua__0036 (lua_State *L, void *p);

#define lua_gettable liblua__0037
LUA_API void liblua__0037 (lua_State *L, int idx);

#define lua_rawget liblua__0038
LUA_API void liblua__0038 (lua_State *L, int idx);

#define lua_rawgeti liblua__0039
LUA_API void liblua__0039 (lua_State *L, int idx, int n);

#define lua_newtable liblua__0040
LUA_API void liblua__0040 (lua_State *L);

#define lua_getmetatable liblua__0041
LUA_API int liblua__0041 (lua_State *L, int objindex);

#define lua_getfenv liblua__0042
LUA_API void liblua__0042 (lua_State *L, int idx);

#define lua_settable liblua__0043
LUA_API void liblua__0043 (lua_State *L, int idx);

#define lua_rawset liblua__0044
LUA_API void liblua__0044 (lua_State *L, int idx);

#define lua_rawseti liblua__0045
LUA_API void liblua__0045 (lua_State *L, int idx, int n);

#define lua_setmetatable liblua__0046
LUA_API int liblua__0046 (lua_State *L, int objindex);

#define lua_setfenv liblua__0047
LUA_API int liblua__0047 (lua_State *L, int idx);

#define lua_call liblua__0048
LUA_API void liblua__0048 (lua_State *L, int nargs, int nresults);

#define lua_pcall liblua__0049
LUA_API int liblua__0049 (lua_State *L, int nargs, int nresults, int errfunc);

#define lua_cpcall liblua__0050
LUA_API int liblua__0050 (lua_State *L, lua_CFunction func, void *ud);

#define lua_load liblua__0051
LUA_API int liblua__0051 (lua_State *L, lua_Chunkreader reader, void *data, const char *chunkname);

#define lua_dump liblua__0052
LUA_API int liblua__0052 (lua_State *L, lua_Chunkwriter writer, void *data);

#define lua_getgcthreshold liblua__0053
LUA_API int liblua__0053 (lua_State *L);

#define lua_getgccount liblua__0054
LUA_API int liblua__0054 (lua_State *L);

#define lua_setgcthreshold liblua__0055
LUA_API void liblua__0055 (lua_State *L, int newthreshold);

#define lua_version liblua__0056
LUA_API const char *liblua__0056 (void);

#define lua_error liblua__0057
LUA_API int liblua__0057 (lua_State *L);

#define lua_next liblua__0058
LUA_API int liblua__0058 (lua_State *L, int idx);

#define lua_concat liblua__0059
LUA_API void liblua__0059 (lua_State *L, int n);

#define lua_newuserdata liblua__0060
LUA_API void *liblua__0060 (lua_State *L, size_t size);

#define lua_pushupvalues liblua__0061
LUA_API int liblua__0061 (lua_State *L);

#define lua_getupvalue liblua__0062
LUA_API const char *liblua__0062 (lua_State *L, int funcindex, int n);

#define lua_setupvalue liblua__0063
LUA_API const char *liblua__0063 (lua_State *L, int funcindex, int n);

#define lua_sethook liblua__0066
LUA_API int liblua__0066 (lua_State *L, lua_Hook func, int mask, int count);

#define lua_gethook liblua__0067
LUA_API lua_Hook liblua__0067 (lua_State *L);

#define lua_gethookmask liblua__0068
LUA_API int liblua__0068 (lua_State *L);

#define lua_gethookcount liblua__0069
LUA_API int liblua__0069 (lua_State *L);

#define lua_getstack liblua__0070
LUA_API int liblua__0070 (lua_State *L, int level, lua_Debug *ar);

#define lua_getlocal liblua__0071
LUA_API const char *liblua__0071 (lua_State *L, const lua_Debug *ar, int n);

#define lua_setlocal liblua__0072
LUA_API const char *liblua__0072 (lua_State *L, const lua_Debug *ar, int n);

#define lua_getinfo liblua__0073
LUA_API int liblua__0073 (lua_State *L, const char *what, lua_Debug *ar);

#define lua_resume liblua__0074
LUA_API int liblua__0074 (lua_State *L, int nargs);

#define lua_yield liblua__0075
LUA_API int liblua__0075 (lua_State *L, int nresults);

#define lua_open liblua__0064
LUA_API lua_State *liblua__0064 (void);

#define lua_close liblua__0065
LUA_API void liblua__0065 (lua_State *L);


/* 
** ===============================================================
** some useful macros
** ===============================================================
*/

#define lua_boxpointer(L,u) \
	(*(void **)(lua_newuserdata(L, sizeof(void *))) = (u))

#define lua_unboxpointer(L,i)	(*(void **)(lua_touserdata(L, i)))

#define lua_pop(L,n)		lua_settop(L, -(n)-1)

#define lua_register(L,n,f) \
	(lua_pushstring(L, n), \
	 lua_pushcfunction(L, f), \
	 lua_settable(L, LUA_GLOBALSINDEX))

#define lua_pushcfunction(L,f)	lua_pushcclosure(L, f, 0)

#define lua_isfunction(L,n)	(lua_type(L,n) == LUA_TFUNCTION)
#define lua_istable(L,n)	(lua_type(L,n) == LUA_TTABLE)
#define lua_islightuserdata(L,n)	(lua_type(L,n) == LUA_TLIGHTUSERDATA)
#define lua_isnil(L,n)		(lua_type(L,n) == LUA_TNIL)
#define lua_isboolean(L,n)	(lua_type(L,n) == LUA_TBOOLEAN)
#define lua_isnone(L,n)		(lua_type(L,n) == LUA_TNONE)
#define lua_isnoneornil(L, n)	(lua_type(L,n) <= 0)

#define lua_pushliteral(L, s)	\
	lua_pushlstring(L, "" s, (sizeof(s)/sizeof(char))-1)



/*
** compatibility macros and functions
*/


LUA_API int lua_pushupvalues (lua_State *L);

#define lua_getregistry(L)	lua_pushvalue(L, LUA_REGISTRYINDEX)
#define lua_setglobal(L,s)	\
   (lua_pushstring(L, s), lua_insert(L, -2), lua_settable(L, LUA_GLOBALSINDEX))

#define lua_getglobal(L,s)	\
		(lua_pushstring(L, s), lua_gettable(L, LUA_GLOBALSINDEX))


/* compatibility with ref system */

/* pre-defined references */
#define LUA_NOREF	(-2)
#define LUA_REFNIL	(-1)

#define lua_ref(L,lock)	((lock) ? luaL_ref(L, LUA_REGISTRYINDEX) : \
      (lua_pushstring(L, "unlocked references are obsolete"), lua_error(L), 0))

#define lua_unref(L,ref)	luaL_unref(L, LUA_REGISTRYINDEX, (ref))

#define lua_getref(L,ref)	lua_rawgeti(L, LUA_REGISTRYINDEX, ref)



/*
** {======================================================================
** useful definitions for Lua kernel and libraries
** =======================================================================
*/

/* formats for Lua numbers */
#ifndef LUA_NUMBER_SCAN
#define LUA_NUMBER_SCAN		"%lf"
#endif

#ifndef LUA_NUMBER_FMT
#define LUA_NUMBER_FMT		"%.14g"
#endif

/******************************************************************************
* Copyright (C) 1994-2004 Tecgraf, PUC-Rio.  All rights reserved.
*
* Permission is hereby granted, free of charge, to any person obtaining
* a copy of this software and associated documentation files (the
* "Software"), to deal in the Software without restriction, including
* without limitation the rights to use, copy, modify, merge, publish,
* distribute, sublicense, and/or sell copies of the Software, and to
* permit persons to whom the Software is furnished to do so, subject to
* the following conditions:
*
* The above copyright notice and this permission notice shall be
* included in all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
* EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
* IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
* CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
* TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
* SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
******************************************************************************/


#endif
