;   bfi.asm - Brainfuck interpreter for the TI-86 (ROM v1.6)
;   
;   Copyright (C) Dean Scarff, 2003.
;   Licensed under the Academic Free License version 1.2
;
;   Revision 0.9, 04 September 2003.
;

;   System equates
#include "ti86asm.inc"          ;   ROM, RAM, character and key equates
_ldHLind        equ     $4010h  ;   ld a,(hl) / ld hl,(hl[0-1])
_cphlde         equ     $403Ch  ;   compare hl and de, set c and z flags
_keytab         equ     $07ECh  ;   keycode to character table (ROM v1.6)

;   Constants
TAPE_LENGTH     equ     $00FFh  ;   length of tape in bytes (255=conservative)
                                ;   length of tape endian-reversed:
TAPE_LENGTHLE   equ     (TAPE_LENGTH<<8) + (TAPE_LENGTH>>8)

.org        _asm_exec_ram
    nop
    jp      ProgStart
    .dw     0
    .dw     ShellTitle
ProgStart:
    call    _cursorOn
    ld      ix,$0000
    add     ix,sp               ;   (ix) = return address
    ld      hl,BF_VARNAME-1
    rst     20h                 ;   _MOV10TOOP1: ld (_op1),(hl[0-9])
    call    _findsym_error      ;   bde = absolute address of var, with check
    ex      de,hl
    call    _get_var_size       ;   de = data length, ahl = abs address of var
    call    _load_ram_ahl       ;   hl = paged address of bf var
    push    hl
    add     hl,de
    ex      de,hl               ;   de = first address beyond variable
    pop     hl
    inc     hl
    inc     hl                  ;   hl = token byte of variable
    xor     a
    cp      (hl)
    jr      z,cleartape
    call    _errDataType        ;   BF not basic and unlocked
cleartape:
    push    hl                  ;   hl becomes the entry point of the bf source    
    ld      bc,TAPE_LENGTHLE
    ld      hl,TAPE-$1
doclear:
    inc     hl
    ld      (hl),$00            ;   clear tape
    djnz    doclear             ;   while (--bc != 0)
    ld      bc,TAPE
fetch:
    pop     hl
    inc     hl                  ;   Increment the source pointer
    call    _cphlde
    jr      nz,interpret
    call    checkframe
    call    nz,_errIllegalNest
    ld      a,Lenter
    call    newline
    ret                         ;   Successful exit point
    
interpret:
    push    hl                  ;   save the source pointer
    ld      a,(hl)
    sub     LplusSign           ;   '+'
    jr      z,plus
    dec     a                   ;   ','
    jr      z,comma
    dec     a                   ;   '-'
    jr      z,minus
    dec     a                   ;   Lperiod '.'
    jr      z,dot
    cp      LLT-Lperiod         ;   '<'
    jr      z,leftangle
    cp      LGT-Lperiod         ;   '>'
    jr      z,rightangle
    cp      LrBrack-Lperiod     ;   ']'
    jr      z,rightbracket
    sub     LlBrack-Lperiod     ;   '['
    jp      z,leftbracket
    jr      fetch               ;   (comment character)

plus:
    ld      h,b
    ld      l,c                 ;   hl = bc
    inc     (hl)
    jr      fetch

minus:
    ld      h,b
    ld      l,c                 ;   hl = bc
    dec     (hl)
    jr      fetch
    
dot:
    ld      a,(bc)
    call    newline
    jr      z,fetch
    call    _putc
    jr      fetch
    
leftangle:
    dec     bc
    ld      hl,TAPE             ;   Check to see if we've exhausted the tape
    sbc     hl,bc
    jr      z,fetch
    jr      c,fetch
    call    _errBound

rightangle:
    inc     bc
    ld      hl,TAPE+TAPE_LENGTH ;   Check to see if we've exhausted the tape
    sbc     hl,bc
    call    c,_errBound
    jr      fetch

rightbracket:
    pop     hl
    call    checkframe
    call    z,_errIllegalNest
    jr      nz,fetch            ;   fetch with the matching '[' - 1

comma:
    push    de                  ;   _getkey clobbers registers
    push    bc
    call    _getkey
    ld      h,$00
    ld      l,a
    add     hl,hl
    ld      bc,_keytab-$2
    add     hl,bc
    call    _ldHLind
    and     a
    jr      z,save
    ld      a,h
    inc     a
    jr      c,save
    ld      a,l
    call    newline
    jr      z,save
    call    _putc
save:
    pop     bc
    pop     de
    ld      (bc),a
    jp      fetch

leftbracket:
    ld      a,(bc)
    and     a                   ;   set flags based on a
    jr      z,skip              ;   if the tape cell is zero, skip to the ']'
    dec     hl
    ex      (sp),hl
    push    hl                  ;   (SP) = this '[', (SP + 2) = this '[' - 1
    jp      fetch
skip:
    push    bc                  ;   save bc
    ld      bc,$01              ;   keep track of the level of nesting
skipdo:
    call    _cphlde
    jr      c,matchl            ;   hl < de
    call    _errIllegalNest
matchl:                         ;   match a '['
    inc     hl
    ld      a,(hl)
    cp      LlBrack
    jr      nz,matchr
    inc     bc                  ;   the nesting has increased
    jr      skipwhile
matchr:                         ;   match a ']'
    cp      LrBrack
    jr      nz,skipwhile
    dec     bc                  ;   the nesting has decreased
skipwhile:
    push    hl
    sbc     hl,hl               ;   carry is reset
    adc     hl,bc               ;   set flags on bc
    pop     hl
    jr      nz,skipdo           ;   repeat until the matching ']' is found
    pop     bc
    ex      (sp),hl
    jp      fetch               ;   fetch with this ']' on the stack

checkframe:                     ;   z set if frame is empty
    ld      hl,$2
    add     hl,sp
    call    _ldHLind
    cp      (ix+$0)
    ret     nz
    ld      a,h
    cp      (ix+$1)
    ret

newline:                        ;   if a is lEnter, go to a new line and set z
    cp      Lenter              ;   (a gets clobbered)
    ret     nz
    call    _putc
    xor     a
    ld      (_curCol),a
    ld      a,Lspace
    call    _putmap
    ld      a,Lenter
    ret

BF_VARNAME:
    .db $02                     ;   name length
ShellTitle:
    .db     "BF"                ;   variable name
    .db     "I"                 ;   title for shells
TAPE:
    .db 0                       ;   also ShellTitle terminator
    
.end