#include "common.h"
#include "gui.h" 
#include <string.h>
#include <ctype.h>
#include <stdlib.h>
#include <stdbool.h>
#include <stdio.h> 

char op_stack[MAX_EXPR_LEN];
int op_top = -1;

void push_op(char op) { op_stack[++op_top] = op; }
char pop_op() { return (op_top >= 0) ? op_stack[op_top--] : 0; }
char peek_op() { return (op_top >= 0) ? op_stack[op_top] : 0; }

int precedence(char op) {
    if (op == '!') return 3; 
    if (op == '*') return 2; 
    if (op == '+') return 1; 
    return 0;
}

bool is_var(char c) { return (c >= 'A' && c <= 'J'); }
bool is_const(char c) { return (c == '0' || c == '1'); }

bool can_implicit_mul_left(char c) { return is_var(c) || is_const(c) || c == ')' || c == '!'; }

bool can_implicit_mul_right(char c) { return is_var(c) || is_const(c) || c == '('; }

void to_rpn(const char* input, char* output) {
    int k = 0; 
    op_top = -1; 
    draw_status("Converting to RPN...", 0); 

    for (int i = 0; input[i] != '\0'; i++) {
        char c = input[i];
        
        if (i > 0) {
            char prev = input[i-1];
            if (can_implicit_mul_left(prev) && can_implicit_mul_right(c)) {
                while (op_top >= 0 && precedence(peek_op()) >= precedence('*')) {
                    output[k++] = pop_op();
                }
                push_op('*');
            }
        }

        if (is_var(c) || is_const(c)) {
            output[k++] = c;
        }
        else if (c == '(') {
            push_op(c);
        }
        else if (c == ')') {
            while (op_top >= 0 && peek_op() != '(') {
                output[k++] = pop_op();
            }
            pop_op(); 
        }
        else if (c == '+' || c == '*' || c == '!') {
            while (op_top >= 0 && precedence(peek_op()) >= precedence(c)) {
                output[k++] = pop_op();
            }
            push_op(c);
        }
    }

    while (op_top >= 0) {
        output[k++] = pop_op();
    }
    output[k] = '\0';
}

bool evaluate_rpn(const char* rpn, int vars_mask, int num_vars) {
    bool stack[MAX_EXPR_LEN];
    int top = -1;

    for (int i = 0; rpn[i] != '\0'; i++) {
        char c = rpn[i];

        if (is_var(c)) {
            int char_idx = c - 'A'; 
            if (char_idx >= num_vars) char_idx = num_vars - 1;
            int bit_shift = (num_vars - 1) - char_idx;
            if (bit_shift < 0) bit_shift = 0; 
            bool val = (vars_mask >> bit_shift) & 1;
            stack[++top] = val;
        }
        else if (c == '0') { stack[++top] = false; }
        else if (c == '1') { stack[++top] = true;  }
        else if (c == '!') {
            bool val = stack[top--];
            stack[++top] = !val;
        }
        else if (c == '*') {
            bool v2 = stack[top--];
            bool v1 = stack[top--];
            stack[++top] = v1 && v2;
        }
        else if (c == '+') {
            bool v2 = stack[top--];
            bool v1 = stack[top--];
            stack[++top] = v1 || v2;
        }
    }
    return stack[0];
}

void parse_expression(const char* input, TruthTable* table) {
    memset(table, 0, sizeof(TruthTable));
    draw_status("Analyzing variables...", 0); 

    int max_char = 0;
    bool vars_present[MAX_VARS] = {false};
    
    for(int i=0; input[i]; i++) {
        if(is_var(input[i])) {
            if(input[i] > max_char) max_char = input[i];
            vars_present[input[i] - 'A'] = true;
        }
    }
    
    table->used_vars_count = 0;
    for(int i=0; i<MAX_VARS; i++) {
        if(vars_present[i]) {
            table->used_var_names[table->used_vars_count] = 'A' + i;
            table->used_var_indices[table->used_vars_count] = i;
            table->used_vars_count++;
        }
    }

    table->num_vars = (max_char == 0) ? 0 : (max_char - 'A' + 1);
    int effective_vars = (table->num_vars < 1) ? 1 : table->num_vars;

    to_rpn(input, table->rpn_expr);

    int limit = 1 << effective_vars; 
    char msg[32]; 

    for (int i = 0; i < limit; i++) {
        if ((i % 128) == 0) {
            sprintf(msg, "Table: %d", i);
            draw_status(msg, 0);
        }
        table->output[i] = evaluate_rpn(table->rpn_expr, i, effective_vars);
    }
    
    if (table->num_vars == 0) table->num_vars = 1;
    draw_status("Table done.", 0);
}