#include "gui.h"
#include <tice.h> 
#include <graphx.h>
#include <keypadc.h>
#include <fileioc.h> 
#include <stdio.h>
#include <string.h>

static const int GRAY_CODE[] = {0, 1, 3, 2};
static const char* GRAY_STRS[] = {"00", "01", "10", "11"}; 
static const char* GRAY_STRS_2[] = {"0", "1"}; 

void gui_init(void) {
    gfx_Begin();
    gfx_SetDrawBuffer();
    gfx_SetTextTransparentColor(255);
    gfx_SetTextBGColor(255);
}

void gui_close(void) {
    gfx_End();
}

void draw_menu_bar(const char* f1, const char* f2, const char* f3, const char* f4, const char* f5) {
    int y = 220;
    int w = 320 / 5;
    
    gfx_SetColor(0);
    gfx_Line(0, y, 320, y);
    
    const char* labels[] = {f1, f2, f3, f4, f5};
    for(int i=0; i<5; i++) {
        if(i>0) gfx_VertLine(i*w, y, 20);
        if(labels[i]) {
            int tw = gfx_GetStringWidth(labels[i]);
            gfx_PrintStringXY(labels[i], i*w + (w-tw)/2, y+6);
        }
    }
}

void draw_status(const char* msg, uint8_t color) {
    int y = 190;
    int h = 15;
    
    gfx_SetColor(255); 
    gfx_FillRectangle(0, y, 320, h);
    
    gfx_SetColor(224);
    gfx_HorizLine(0, y, 320);
    gfx_HorizLine(0, y+h, 320);

    gfx_SetTextFGColor(color);
    gfx_PrintStringXY(msg, 5, y + 4);
    gfx_SetTextFGColor(0);

    gfx_Blit(1); 
}

void export_to_y1(const char* expr) {
    ti_CloseAll();
    ti_var_t var = ti_OpenVar(ti_Y1, "w", TI_EQU_TYPE);
    if (!var) return;

    for(int i=0; expr[i] != '\0'; i++) {
        char c = expr[i];
        uint8_t token = 0;

        if (c >= '0' && c <= '9') token = 0x30 + (c - '0');
        else if (c >= 'A' && c <= 'Z') token = 0x41 + (c - 'A');
        else {
            switch(c) {
                case '+': token = 0x70; break; 
                case '*': token = 0x82; break; 
                case '(': token = 0x10; break; 
                case ')': token = 0x11; break; 
                case '!': token = 0x2D; break; 
                case '\'': token = 0x0E; break;
                default:  token = 0x00; break; 
            }
        }
        if (token != 0) ti_PutC((char)token, var);
    }
    ti_Close(var);
}

void draw_input_field(const char* text, int y, int height, int cursor_idx) {
    int x_start = 0;
    int width = 320;
    int padding = 10; 

    char temp_buf[512];
    strncpy(temp_buf, text, cursor_idx);
    temp_buf[cursor_idx] = '\0';
    int cursor_relative_x = gfx_GetStringWidth(temp_buf);
    
    int scroll_offset = 0;
    int visible_width = width - (2 * padding); 
    
    if (cursor_relative_x > visible_width) {
        scroll_offset = cursor_relative_x - visible_width;
    }
    
    gfx_SetClipRegion(x_start, y, x_start + width, y + height);
    
    int current_x = 0; 
    int text_y = y + (height - 8) / 2;
    int len = strlen(text);

    for (int i = 0; i <= len; i++) {
        int screen_obj_x = x_start + padding + current_x - scroll_offset;

        if (i == cursor_idx) {
            if (screen_obj_x >= 0 && screen_obj_x < 319) {
                gfx_SetColor(0); 
                gfx_VertLine(screen_obj_x, y + 4, height - 8);
                gfx_VertLine(screen_obj_x + 1, y + 4, height - 8);
            }
        }

        if (i == len) break; 

        char buf[2] = {text[i], '\0'};
        int char_width = gfx_GetStringWidth(buf);
        
        if (screen_obj_x < 0) {
            current_x += char_width;
            continue;
        }

        if (screen_obj_x + char_width > 320) {
            break;
        }

        gfx_PrintStringXY(buf, screen_obj_x, text_y);
        current_x += char_width;
    }
    
    gfx_SetClipRegion(0, 0, 320, 240);
}

bool draw_wrapped_text(const char* text, int x, int y, int max_width, int max_height_y, bool render) {
    int start_x = x;
    int cur_x = x;
    int cur_y = y;
    int line_height = 15; 
    
    for (int i = 0; text[i] != '\0'; i++) {
        char buf[2] = {text[i], '\0'};
        int char_width = gfx_GetStringWidth(buf);
        
        if (cur_x + char_width > start_x + max_width) {
            cur_x = start_x;
            cur_y += line_height;
            if (cur_y > max_height_y) return true; 
        }
        
        if (render) gfx_PrintStringXY(buf, cur_x, cur_y);
        cur_x += char_width;
    }
    return false; 
}

void draw_dashboard(const char* input, const char* result, bool is_solved, int num_vars, int cursor_idx) {
    gfx_FillScreen(255);
    gfx_SetTextFGColor(0);

    gfx_PrintStringXY("Logic Expression (A..J):", 5, 10);
    
    gfx_SetColor(224);
    gfx_FillRectangle(0, 25, 320, 30); 
    gfx_SetColor(0);
    gfx_HorizLine(0, 25, 320);
    gfx_HorizLine(0, 55, 320);
    
    gfx_SetTextScale(2, 2);
    draw_input_field(input, 27, 26, cursor_idx);
    gfx_SetTextScale(1, 1);

    if (is_solved) {
        gfx_PrintStringXY("Simplified Form:", 5, 80);
        
        gfx_SetTextScale(2, 2);
        gfx_SetTextFGColor(0x03); 
        
        int limit_y = (num_vars > 4) ? 145 : 135;

        bool overflow = false;
        
        int scale = (gfx_GetStringWidth(result) > 280) ? 1 : 2;
        gfx_SetTextScale(scale, scale);

        overflow = draw_wrapped_text(result, 15, 95, 290, limit_y, false);

        if (overflow) {
            gfx_SetTextScale(1, 1);
            gfx_SetTextFGColor(224); 
            gfx_PrintStringXY("Too long! Saved to Y1", 15, 110);
            export_to_y1(result);
        } 
        else {
            draw_wrapped_text(result, 15, 95, 290, limit_y, true);
        }
        
        gfx_SetTextFGColor(0);
        gfx_SetTextScale(1, 1);
        
        gfx_SetTextFGColor(0x03); 
        gfx_PrintStringXY("Done.", 5, 194);
        gfx_SetTextFGColor(0);

        if(num_vars > 4) {
            gfx_SetTextFGColor(192); 
            gfx_PrintStringXY("K-Map/Table only works with A-D variables", 15, 150);
            draw_menu_bar("OR", "AND", "NOT", "---", "---");
        } else {
            gfx_PrintStringXY("[F4] = K-Map   [F5] = Truth Table", 15, 140);
            draw_menu_bar("OR", "AND", "NOT", "K-MAP", "TABLE");
        }
    } else {
        gfx_SetTextFGColor(128);
        gfx_PrintStringXY("Enter expression and press [ENTER]", 5, 80);
        gfx_SetTextFGColor(0);
        
        gfx_PrintStringXY("[+] OR   [*] AND   [!] NOT", 5, 150);
        draw_menu_bar("OR", "AND", "NOT", "---", "CLEAR");
    }

    gfx_SwapDraw();
}

void show_kmap_window(TruthTable* table) {
    while(os_GetCSC()); 
    
    if (table->used_vars_count > 4) {
        draw_status("Too many vars (>4) for map!", 224);
        delay(1000); return;
    }
    if (table->used_vars_count == 0) {
        draw_status("No variables for map.", 0);
        delay(1000); return;
    }

    while(1) {
        gfx_FillScreen(255);
        int vars = table->used_vars_count;
        
        int start_x = 100, start_y = 80, cell = 30;
        
        int rows = (vars >= 3) ? 4 : 2;
        int cols = (vars == 2 || vars == 4) ? 4 : 2;
        if (vars == 1) { rows = 2; cols = 1; }
        else if (vars == 2) { rows = 2; cols = 2; } 
        else if (vars == 3) { rows = 2; cols = 4; }
        else { rows = 4; cols = 4; }

        gfx_SetColor(0);
        for(int r=0; r<=rows; r++) gfx_HorizLine(start_x, start_y+r*cell, cols*cell);
        for(int c=0; c<=cols; c++) gfx_VertLine(start_x+c*cell, start_y, rows*cell);

        int row_bits = (rows == 4) ? 2 : 1;
        int col_bits = (cols == 4) ? 2 : 1;
        
        for (int c = 0; c < cols; c++) {
            int g_code = (cols == 4) ? GRAY_CODE[c] : c;
            const char* label = (col_bits == 2) ? GRAY_STRS[g_code] : GRAY_STRS_2[g_code];
            int text_w = gfx_GetStringWidth(label);
            gfx_PrintStringXY(label, start_x + c*cell + (cell-text_w)/2, start_y - 12);
        }

        for (int r = 0; r < rows; r++) {
            int g_code = (rows == 4) ? GRAY_CODE[r] : r;
            const char* label = (row_bits == 2) ? GRAY_STRS[g_code] : GRAY_STRS_2[g_code];
            int text_w = gfx_GetStringWidth(label);
            gfx_PrintStringXY(label, start_x - text_w - 4, start_y + r*cell + (cell-8)/2);
        }

        gfx_Line(start_x, start_y, start_x - 30, start_y - 20);
        
        char top_vars[5] = {0};
        char left_vars[5] = {0};
        
        for(int i=0; i<col_bits; i++) {
            char s[2] = {table->used_var_names[i], 0};
            strcat(top_vars, s);
        }
        for(int i=0; i<row_bits; i++) {
            if (col_bits + i < vars) {
                char s[2] = {table->used_var_names[col_bits + i], 0};
                strcat(left_vars, s);
            }
        }
        
        gfx_PrintStringXY(top_vars, start_x - 10, start_y - 22);
        gfx_PrintStringXY(left_vars, start_x - 35, start_y - 8);

        for(int r=0; r<rows; r++) {
            for(int c=0; c<cols; c++) {
                int g_row = (rows == 4) ? GRAY_CODE[r] : r;
                int g_col = (cols == 4) ? GRAY_CODE[c] : c; 
                
                int master_index = 0;
                int total_vars = table->num_vars;
                
                int current_var_idx = 0;

                for(int i = col_bits - 1; i >= 0; i--) {
                     int bit_val = (g_col >> i) & 1;
                     int real_var_idx = table->used_var_indices[current_var_idx];
                     int shift = (total_vars - 1) - real_var_idx;
                     if (bit_val) master_index |= (1 << shift);
                     current_var_idx++;
                }

                for(int i = row_bits - 1; i >= 0; i--) {
                     int bit_val = (g_row >> i) & 1;
                     int real_var_idx = table->used_var_indices[current_var_idx];
                     int shift = (total_vars - 1) - real_var_idx;
                     if (bit_val) master_index |= (1 << shift);
                     current_var_idx++;
                }

                if(table->output[master_index]) {
                    gfx_SetTextFGColor(224);
                    gfx_PrintStringXY("1", start_x+c*cell+12, start_y+r*cell+10);
                } else {
                    gfx_SetTextFGColor(128);
                    gfx_PrintStringXY("0", start_x+c*cell+12, start_y+r*cell+10);
                }
            }
        }
        
        gfx_SetTextFGColor(0);
        gfx_PrintStringXY("K-Map", 140, 20);
        draw_menu_bar("BACK", "BACK", "BACK", "BACK", "BACK");
        gfx_SwapDraw();
        
        uint8_t key = os_GetCSC();
        if (key == sk_Clear || key == sk_Enter || 
            key == sk_Yequ || key == sk_Window || key == sk_Zoom || 
            key == sk_Trace || key == sk_Graph) return;
    }
}

void show_truthtable_window(TruthTable* table) {
    while(os_GetCSC());
    if (table->used_vars_count > 4) {
        draw_status("Too many vars (>4) for table!", 224);
        delay(1000); return;
    }
    while(1) {
        gfx_FillScreen(255);
        gfx_SetTextFGColor(0);
        gfx_PrintStringXY("Truth Table", 90, 10);
        int vars = table->used_vars_count;
        if (vars == 0) vars = 1; 
        int rows = 1 << vars;
        int start_y = 35;
        int col_width = 30;
        int start_x = (320 - ((vars + 1) * col_width)) / 2;
        gfx_SetColor(0);
        gfx_HorizLine(start_x, start_y + 12, (vars + 1) * col_width);
        for (int v = 0; v < vars; v++) {
            char var_name[2] = { table->used_var_names[v], '\0' };
            if (table->used_vars_count == 0) var_name[0] = '-'; 
            gfx_PrintStringXY(var_name, start_x + v * col_width + 10, start_y);
        }
        gfx_PrintStringXY("Y", start_x + vars * col_width + 10, start_y);
        for (int i = 0; i < rows; i++) {
            int y = start_y + 20 + (i * 10);
            int master_index = 0;
            int total_vars = table->num_vars;
            for (int v = 0; v < vars; v++) {
                int bit = (i >> (vars - 1 - v)) & 1;
                gfx_PrintStringXY(bit ? "1" : "0", start_x + v * col_width + 10, y);
                if (table->used_vars_count > 0) {
                    int real_var_idx = table->used_var_indices[v];
                    int shift = (total_vars - 1) - real_var_idx;
                    if(bit) master_index |= (1 << shift);
                }
            }
            if (table->output[master_index]) {
                gfx_SetTextFGColor(224); 
                gfx_PrintStringXY("1", start_x + vars * col_width + 10, y);
                gfx_SetTextFGColor(0);
            } else {
                gfx_SetTextFGColor(128); 
                gfx_PrintStringXY("0", start_x + vars * col_width + 10, y);
                gfx_SetTextFGColor(0);
            }
        }
        for (int i = 0; i <= vars + 1; i++) {
             gfx_VertLine(start_x + i * col_width, start_y, 20 + rows * 10);
        }
        draw_menu_bar("BACK", "BACK", "BACK", "BACK", "BACK");
        gfx_SwapDraw();
        
        uint8_t key = os_GetCSC();
        if (key == sk_Clear || key == sk_Enter || 
            key == sk_Yequ || key == sk_Window || key == sk_Zoom || 
            key == sk_Trace || key == sk_Graph) return;
    }
}