;*** timer.h
; This part not only handles timers, but also ON-key events.
; Link-cable-interrupts are ignores.

; Timer Table:
;
; Current Counter : Max Counter (or 0 for no repeat) : Function-call-address

MAX_TIMERS:       .equ 32
                                ; (one memory block)

MULTITASK_TIMER:  .equ  2

;** timer_init
;* Initialises the timer-table, sets up interrupts and enables multitasking. Is called by the OS at boot-time.

timer_init:
    LD HL, MAX_TIMERS * 4
    CALL save_calloc
    LD (timertab), HL

    LD A, MULTITASK_TIMER
    LD (HL), A
    INC HL
    LD (HL), A
    INC HL
    LD DE, nextproc_IR
    LD (HL), E
    INC HL
    LD (HL), D

    ; Setting interrupt mode:

    ; * Bit 0: Set 1 to enable the ON key. Set 0 acknowledge the interrupt request and/or to disable it.
    ; * Bit 1: Set 1 to enable the first hardware timer. Set 0 acknowledge the interrupt request and/or to disable it.
    ; * Bit 2: Set 1 to enable the second hardware timer. Set 0 acknowledge the interrupt request and/or to disable it.
    ; * Bit 3: Set 0 to put the calculator in low power mode if and only if the Z80 cpu enters the halt state
    ;   (during the executing of a halt instruction). Set 1 to keep the calculator powered.
    ; * Bit 4: Set 1 to enable the linkport interrupt. Set 0 acknowledge the interrupt request and/or to disable it.
    ;   (Does not completely disable the linkport.)

    IM 1              ; Interrupt Mode 1, Normal System Interrupts.
    LD A, %00001011   ; Enable Timer 1 + ON-key interrupts (+ HALT means low-power state).
    OUT (3), A

    LD (timer_on), A  ; from this moment interrupts are allowed.

    EI                ; Enable them.
    LD HL, timer_success_msg
    CALL puts
    RET


;** INTERRUPT
;* This function is called by the OS when an interrupt happends.
;* From here for example the timer-table is read to react on the interrupt.

INTERRUPT:
    DI

    PUSH AF           ; Protect EVERY register.
    PUSH BC
    PUSH DE
    PUSH HL
    PUSH IX
    PUSH IY

    ; For multitasking, save the flash page + our screen.
    IN A, (6)
    PUSH AF
    LD A, (myscrn)
    LD B, A
    LD A, (mydir)
    LD C, A
    PUSH BC

    ; First ACK all interrupts:

    IN A, (4)	     ; Save i'rupt status.
    LD B, A
    XOR A
    OUT (3), A
    LD A, %00001011   ; Enable Timer 1 + ON-key interrupts.
    OUT (3), A

    ; Are interrupts allowed?

    LD A, (timer_on)
    OR A
    JP Z, IR_common_end

    ; Since Tilem doesn't support port #4 (you get always an $01), we must do it in an other way

#include "tilem.1.patch"

    LD A, B
    AND 8
    JR Z, IR_handle_ON

#include "tilem.2.patch"

    ; Just processing the whole table, one by one.


    LD HL, (timertab)
    LD DE, MAX_TIMERS * 4
    ADD HL, DE
    EX DE, HL
    LD HL, (timertab)

IR_next_entry:
    LD A, (HL)
    OR A
    JR Z, IR_next
    DEC A
    JR Z, IR_call
    LD (HL), A

IR_next:
    INC HL
    INC HL
    INC HL
    INC HL
    LD A, H
    CP D
    JR NZ, IR_next_entry
    LD A, L
    CP E
    JR NZ, IR_next_entry

    ; Were done.

IR_common_end:
    POP BC
    LD A, B
    LD (myscrn), A
    LD A, C
    LD (mydir), A
    POP AF
    OUT (6), A
    POP IY
    POP IX
    POP HL
    POP DE
    POP BC
    POP AF
    EI
    RET

IR_call:
    INC HL
    LD A, (HL)
    DEC HL
    LD (HL), A
    PUSH HL
    PUSH DE
    INC HL
    INC HL
    LD E, (HL)
    INC HL
    LD D, (HL)
    CALL IR_do_call
    POP DE
    POP HL
    JR IR_next

IR_handle_ON:	; For now we only check for screenswitching or 2nd to turn of...
    CALL sysinkey
    OR A
    JR NZ, IR_handle_ON_Fkeys
    LD A, $BF
    OUT (1), A
    NOP
    NOP
    IN A, (1)
    CP $DF
    JR NZ, IR_common_end
    JP shutdown

IR_handle_ON_Fkeys:
    CP ASCF5 + 1
    JR NC, IR_common_end
    CP ASCF1
    JR C, IR_common_end
    SUB 127
    CALL switchscrn
    JR IR_common_end

IR_do_call:
    PUSH DE
    RET

;** sleep
;* IN B: Interrupts to sleep (see also <code>newtimer</code>).
;* OUT HL: Zero on success, $FFFF on failure.
;* If a program has to wait a certain amount of time, you can call this function.
;* It will set a special flag inside the process-table, which makes the mutitasking engine just skip this process.
;* After a specified amount of time, a timer will be triggered, which simply calls the wakeup function.
;* This routine resets the flag, after which the process will be acting like a normal one again.

sleep:

    ; We should not put ourself to sleep before setting up the timer !!.

    ; How it works, we allocate memory and put the following code in there:
    ;
    ;    LD A, <PID>
    ;    JP wakeup

    PUSH AF
    PUSH BC
    PUSH DE

    LD HL, 2 + 3
    PUSH BC
    CALL pgmalloc
    POP BC
    LD A, H
    OR L
    JR Z, sleep_error
    PUSH HL
    LD (HL), $3E
    INC HL
    LD A, (curproc)
    LD (HL), A
    INC HL
    LD (HL), %11000011
    INC HL
    LD DE, wakeup
    LD (HL), E
    INC HL
    LD (HL), D

    ; First already lookup our place in the process table, to save interrupt cycles.

    LD A, (curproc)

    LD HL, (proctab)
    LD DE, 16           ; Proctab length

    LD C, (HL)
    CP C
    JR Z, sleep_found1

sleep_loop1:
    ADD HL, DE
    LD C, (HL)
    CP C
    JR NZ, sleep_loop1

sleep_found1:
    INC HL

    ; Critical section

    DI
    LD (HL), 0
    POP DE
    LD C, 0
    CALL newtimer
    LD A, H
    OR L
    JR NZ, sleep_error2

    ; Just pray for good timing :p
    EI
    CALL nextproc
    PUSH DE
    POP HL
    CALL free
    LD HL, $0000
    POP DE
    POP BC
    POP AF
    RET

sleep_error2:
    EI
    PUSH DE
    POP HL
    CALL free
sleep_error:
    LD HL, $ffff
    POP DE
    POP BC
    POP AF
    RET


;** wakeup
;* IN A: PID to wake up.
;* OUT A: Garbage
;* OUT BC: Garbage
;* OUT DE: Garbage
;* OUT HL: Zero on success, $FFFF on failure.
;* This function wakes up a sleeping process.
;* Under normal circumstances this function will only be called from a timer, activated by the sleep function,
;* however it's theoretically possible to call it to wake-up a process from within another process.
;* <B>NOTE</B>: When called in a normal way (not triggered by a timer), <U>the calling process must disable
;* interrupts before calling (and offcourse enable them afterwards), to prevent a system crash</U>.

wakeup:   ; Args: A is PID to wake.
          ; Warning: should be called when interrupts are off (i.e. from an interrupt routine)

    LD HL, (proctab)
    LD DE, 16     ; Proctab lenght
    LD B, MAX_PROCS

    LD C, (HL)
    CP C
    JR Z, wakeup_found

wakeup_loop:
    ADD HL, DE
    DEC B
    JR Z, wakeup_error
    LD C, (HL)
    CP C
    JR NZ, wakeup_loop

wakeup_found:
    INC HL
    LD (HL), 1
    LD HL, $0000
    RET

wakeup_error:
    LD HL, $ffff
    RET

;** newtimer
;* IN B: Countdown time (measured in interrupts, 140 = ~ 1 sec).
;* IN C: Repeat-countdown, after the first time the timer is triggered, every C interrupts, the timer will be triggered again (use 0 ('zero') for no repeat).
;* IN DE: Pointer to the function that should be called by the timer.
;* OUT A: Garbage
;* OUT BC: Same as input
;* OUT DE: Same as input
;* OUT HL: Zero on success, $FFFF on failure.
;* This function sets up a timer for a scheduled execution of a function.
;* This is internally used for the <code>sleep</code> function, multitasking and cursor blinking.
;* It can be used for practicly anything that needs timer interrupts.

newtimer:         ; Creates a new interrupt-timer
                  ; Args: DE call destination location
                  ;       B countdown
                  ;       C repeat countdown (0 for none)
    PUSH DE
    PUSH BC
    LD HL, (timertab)
    LD DE, 4
    LD C, MAX_TIMERS

newtimer_loop:
    ADD HL, DE
    LD A, (HL)
    OR A
    JR Z, newtimer_found
    DEC C
    JR NZ, newtimer_loop

newtimer_found:
    POP BC
    POP DE
    OR A
    JR NZ, newtimer_error
    LD (HL), B
    INC HL
    LD (HL), C
    INC HL
    LD (HL), E
    INC HL
    LD (HL), D
    LD HL, $0000
    RET

newtimer_error:
    LD HL, $ffff
    RET
