;*** screen.h

;** screen_init
;* Called by the OS at boot time to initialize the screen manager.

screen_init:
    ; Initialise the master (physical) screen locks.
    XOR A
    LD (screen_lock), A
    LD (screen_master), A
    LD HL, 0
    LD (scrnloc), HL
    LD (scrnloc+2), HL
    LD (scrnloc+4), HL
    LD (scrnloc+6), HL
    LD (scrnloc+8), HL
    INC A
    LD (curscrn), A
    LD (myscrn), A
    LD (scrnblink), A


    ; Now init the 5 virtual screens ("consoles").

    ; We need to reserve memory for:
    ;
    ; * 4 x text screen (9 x 16 = 144) = 576
    ; * 1 x text screen, for what is currently displayed (physical) = 144
    ; * 1 x graph screen = 768

    LD HL, 576+144+768
    CALL save_malloc
    LD (screens), HL

    ; Now init all screens with spaces (except graphical screen)
    DI
    LD (temp), SP
    LD DE, 576+144
    ADD HL, DE
    LD (proctempSP), HL
    LD SP, (proctempSP)
    LD HL, (doublespace)
    LD B, 45    ; Should be (576+144)/16
screen_init_loop:
    PUSH HL
    PUSH HL
    PUSH HL
    PUSH HL
    PUSH HL
    PUSH HL
    PUSH HL
    PUSH HL
    DJNZ screen_init_loop

    LD SP, (temp)
    EI

    ; Now start the screen manager and return
    LD HL, screenmanager
    CALL newproc

    ; Set up cursor blinking (by adding a timer to the blink routine).

    LD DE, blink	; call vector
    LD B, 50		; Every 50 Interrupts.
    LD C, 50		; 140 = ~ 1 sec.
    JP newtimer		; call and exit.

;** release_screen
;* Is called by all functions that communicate with the LCD display, to release the lock on the screen.
;* After this functions is called, any process will be able to communicate with the LCD screen again.
;* Only needed by the LCD-driver functions.

release_screen:
    PUSH AF
    PUSH BC
    LD A, (curproc)
    LD B, A
    LD A, (screen_lock)
    CP B
    JR NZ, release_screen_cont
    XOR A
    LD (screen_lock), A
release_screen_cont:
    POP BC
    POP AF
    RET

;** lock_screen
;* This function is called by every function that communicated with the LCD display.
;* It sets a special byte to lock other communication to the LCD driver until the caller function is done.
;* This is automatically done, so you shouldn't need it.
;* To get a full lock on the LCD driver, you should use <code>become_screen_master</code>,
;* however you should only use this in for example games, since all normal communication is done by the screenmanager.

lock_screen:
    ; Important !!! Disable interrupts: we don't want to overwrite an existing lock !!
    PUSH AF
    PUSH BC
lock_screen_loop:
    DI
    LD A, (screen_lock)
    OR A
    JR Z, lock_screen_lock

    ; Someone is already writing to the screen, just wait a bit...
    ; Interrupts will be enabled by sub-routine...

lock_screen_wait:
    CALL nextproc
    JR lock_screen_loop

lock_screen_lock:
    LD A, (curproc)
    LD B, A
    LD A, (screen_master)
    OR A
    JR Z, lock_screen_cont
    CP B
    JR NZ, lock_screen_wait

lock_screen_cont:
    LD A, (curproc)
    LD (screen_lock), A
    EI
    POP BC
    POP AF
    RET

;** become_screen_master
;* OUT A: Garbage
;* OUT B: Garbage
;* OUT HL: 0 on succes, on failure the PID from the process that owns the screen.
;* Used to get full access over the LCD display.
;* When a process is the master of screen, there is no other process that may lock to screen to write to it.
;* This means no process can write to the LCD display until it is released by <code>share_screen</code>.
;* Since to OS should handle all data to the LCD display (using virtual screens), this function should only to be used by the OS.

become_screen_master:
    DI
    LD A, (screen_master)
    OR A
    JR Z, become_screen_master_go
    LD B, A
    LD A, (curproc)
    CP B
    JR Z, become_screen_master_go
    LD H, $00               ; Return value.
    LD L, B
    EI
    RET

become_screen_master_go:
    LD A, (curproc)
    LD (screen_master), A
    LD HL, $0000            ; Return value.
    EI
    RET

;** share_screen
;* OUT A: Garbage
;* OUT B: Garbage
;* When this function is called, the previous "master-of-screen"-lock (previous got by the <code>become_screen_master</code> function)
;* will be released so any process will be able to use the LCD display again.
;* Usefull when designing games so the screen-manager will not interfere.

share_screen:
    LD A, (curproc)
    LD B, A
    LD A, (screen_master)
    CP B
    RET NZ
    XOR A
    LD (screen_master), A
    RET

;** screenmanager
;* The actual screen manager that runs in it's own process.

screenmanager:    ; This is the screen manager process.
                  ; It is made to run always in background
                  ; while syncing the LCD display to the virtual screen.

    ; First: identify ourself.

    LD A, (curproc)
    LD HL, screenmngrname
    CALL setprocname

screenmanager_endlessloop:
    LD HL, (screens)
    LD DE, 144
    LD A, (curscrn)
    CP 5
    JR Z, screenmanager_graphscrn
    LD B, A
screenmanager_calcloop:
    ADD HL, DE
    DJNZ screenmanager_calcloop
    LD DE, (screens)
    LD B, 144
screenmanager_loop:
    LD A, (DE)
    LD C, (HL)
    CP C
    JR Z, screenmanager_cont
    LD A, C
    LD (DE), A
    LD (screenmanager_var), A

    ; We need to calculate the location first:
    PUSH DE     ; Save our pointers.
    PUSH HL
    PUSH BC

    ; DE is the destination byte of the screen plus (screen)
    EX DE, HL
    LD DE, (screens)
    SCF
    CCF
    SBC HL, DE

    ; Now HL is the actual destination byte.
    ; Since we are using 16 bytes / row, the ms nibble of L is what H should be.
    LD H, L
    SRL H
    SRL H
    SRL H
    SRL H     ; H is now the current row.
    LD A, L
    AND $0f
    LD L, A   ; L is now the current column.
    EX DE, HL

screenmanager_master__loop:
    CALL become_screen_master
    XOR A
    OR L
    JR NZ, screenmanager_master__loop

    LD (location), DE
    LD A, (screenmanager_var)
    CALL putc
    CALL share_screen

    POP BC
    POP HL
    POP DE

screenmanager_cont:
    INC HL
    INC DE
    DJNZ screenmanager_loop

    ; We don't really need to make multiple screen-cycles within 1 timeslice.
screenmanager_graphscrn:
    CALL nextproc
    JR screenmanager_endlessloop

;** scrnsetloc
;* IN D: Row (starting at 0)
;* IN E: Column (starting at 0)
;* Sets the location where the next character will be printed on the screen. Same as <code>locate</code>, but used with virtual screens.

scrnsetloc:       ; Arguments: D = row, E = column.
                  ; Destroys HL, A
    CALL findscrnloc
    LD (HL), E
    INC HL
    LD (HL), D
    RET



;** findscrnloc
;* OUT A: Garbage
;* OUT HL: Pointer the the current screen location.
;* This function checks the current virtual screen and calculates a pointer to the current cursor location on that screen.
;* It is internally called by <code>getscrnloc</code>.

findscrnloc:		; Find location of current scrnloc and put it in HL
    LD HL, scrnloc	; Destroys A.
    LD A, (myscrn)
    DEC A
    ADD A, A
    ADD A, L
    LD L, A
    RET C
    INC H
    RET


;** getscrnloc
;* OUT DE: Current Screen cursor location.
;* This function puts the location of the cursor into DE (D = row, E = column), the left-upper corner is DE = $0000.

getscrnloc:
    PUSH HL
    PUSH AF
    CALL findscrnloc
    LD E, (HL)
    INC HL
    LD D, (HL)
    POP AF
    POP HL
    RET

;** setscrnloc
;* IN DE: New cursor location on the current virtual screen.
;* This function sets the location of the cursor (D = row, E = column) on the current virtual screen,
;* without destroying registers.

setscrnloc:
    PUSH HL
    PUSH AF
    CALL scrnsetloc
    POP AF
    POP HL
    RET

;** scrngetramloc
;* OUT A: Garbage
;* OUT BC: Garbage
;* OUT DE: Garbage
;* OUT HL: Pointer to memory location of the character belonging to the cursor of this virtual screen.
;* This function returns the location in the memory where the next character printed on this virtual screen will be stored.

scrngetramloc:	; get the RAM location of the location DE of the current screen.
    PUSH DE
    LD HL, (screens)
    LD DE, 144
    LD A, (curscrn)
    LD B, A
scrngetramloc_loop:
    ADD HL, DE
    DJNZ scrngetramloc_loop
    POP DE
    LD B, D
    LD D, 0
    ADD HL, DE
    XOR A
    OR B
    RET Z
    PUSH HL
    LD HL, $0000
    LD DE, 16
scrngetramloc_loop2:
    ADD HL, DE
    DJNZ scrngetramloc_loop2
    EX DE, HL
    POP HL
    ADD HL, DE
    RET

;** blink
;* OUT A: Garbage
;* OUT BC: Garbage
;* OUT DE: Garbage
;* OUT HL: Garbage
;* This function is called by the timer-trigger-routine.
;* It simply show or hides the cursor, the create a blink effect. Only used by the system.

blink:
    LD A, (curscrn)
    CP 5
    RET Z
    LD (myscrn), A
    LD A, (scrnblink)
    OR A
    RET Z
    DEC A
    JR NZ, blink_restore

    LD A, 2
    LD (scrnblink), A
    CALL getscrnloc
    CALL scrngetramloc
    LD A, (HL)
    LD (blinkchar), A
    LD (blinkloc), HL
    LD A, blink_cursor		; see data.h
    JP blink_putc

blink_restore:
    LD (scrnblink), A
    CALL getscrnloc
    CALL scrngetramloc
    LD DE, (blinkloc)
    LD A, D
    CP H
    JR NZ, blink_newloc
    LD A, E
    CP L
    JR NZ, blink_newloc
    LD A, (HL)
    CP blink_cursor
    RET NZ
    LD A, (blinkchar)

blink_putc:
    LD (HL), A
    RET

blink_newloc:
    LD A, (DE)
    LD B, blink_cursor
    CP B
    JR NZ, blink	; Restart to let the cursor blink on the new place.
    LD A, (blinkchar)
    LD (DE), A
    JR blink		; Restart to let the cursor blink on the new place.


;** switchscrn
;* IN A: Screen to switch to
;* Switches the active virtual screen (the screen "connected" to the physical screen)
;* to the screen A (can be 1-5, where 5 is the graphical screen, which requires a third-party window manager).

switchscrn:   ; A = screen to switch to.

    ; Because of the blinking routine, we have to correct some things first.
    ; It seems not really necesarily.

    LD (curscrn), A
    RET

;** scrnrefresh
;* Refreshes the Physical screen from the buffer of the currently selected virtual screen.

scrnrefresh:
    PUSH HL
    PUSH BC
    PUSH AF
    LD HL, (screens)
    LD B, 144
    LD A, 8     ; Backspace, no printable character.
scrnrefresh_loop:
    LD (HL), A
    INC HL
    DJNZ scrnrefresh_loop
    POP AF
    POP BC
    POP HL
    RET

screenmngrname:
.db "Screen mngr", 0
