;*** procs.h
; Process Manager

MAX_PROCS:        .equ  32
PROC_TAB_ENTRY:   .equ  16

proc_sleep:  .equ 0
proc_run:    .equ 1


;** proc_init
;* Called at boot time to initialize the process manager.
proc_init:
    CALL proc_init2
    LD A, 1
    LD (multitasking), A
    LD HL, proc_init_success
    CALL puts
    RET

proc_init2:
    DI

    ; Init the proc-table:
    ; Nr (1) : State (1) : Stackptr (2) : Name (11) : \0

    ; WARNING: We need to identify ourselfs, for malloc (PID = 1)
    LD A, 1
    LD (curproc), A

    LD HL, PROC_TAB_ENTRY * MAX_PROCS
    CALL save_malloc
    LD (proctab), HL
    LD A, 1
    LD (HL), A  ; PID
    INC HL
    LD (HL), A  ; Running


                  ; Put our name in it.
    INC HL
    PUSH HL
    INC HL
    INC HL
    LD DE, ProcName
    CALL strcpy
    POP HL        ; Pointer at Stackptr.

    ; Prepare the system stack
    PUSH AF
    PUSH BC
    PUSH DE
    PUSH HL
    PUSH IX
    PUSH IY
    IN A, (6)
    PUSH AF
    LD A, (myscrn)
    LD B, A
    LD C, 0       ; Root directory
    PUSH BC

    ; We also need to put some data in place to get
    ; to the routines that bring us back where we came from.

    LD DE, nextproc_catchback
    LD (proc_timer_tempSP), DE

    ; Now we put the pointer into the table.

    LD (proctempSP), SP
    LD DE, (proctempSP)
    LD (HL), E
    INC HL
    LD (HL), D

    ; IMPORTANT !! Clear the rest of the table !!

    LD HL, (proctab)
    LD DE, PROC_TAB_ENTRY * MAX_PROCS
    ADD HL, DE
    LD (proctempSP), HL
    LD SP, (proctempSP)
    LD HL, $0000
    LD B, MAX_PROCS
    DEC B
proc_clear_loop:
    PUSH HL
    PUSH HL
    PUSH HL
    PUSH HL
    PUSH HL
    PUSH HL
    PUSH HL
    PUSH HL
    DJNZ proc_clear_loop

    ; Ok, now switch to the kernel-process...

    LD A, 0
    LD (proctabentry), A
    JP nextproc_next


;** nextproc
;* Let the process manager make a switch to the next process.
nextproc:
    ; This routine is only for the purpose to emulate the Timer-Handler.

    DI

    ; Prepare the system stack
    PUSH AF
    PUSH BC
    PUSH DE
    PUSH HL
    PUSH IX
    PUSH IY
    IN A, (6) ; Current flash Page
    PUSH AF
    LD A, (myscrn)
    LD B, A
    LD A, (mydir)
    LD C, A
    PUSH BC

nextproc_no_push_args:
    LD HL, $0000
    PUSH HL
    PUSH HL

    ; Call the real function
    CALL nextproc_IR

nextproc_catchback:

    ; First we need to get rid of the simulation crap.

    POP HL
    POP HL

    ; Now we can POP all the registers back.

    POP BC
    LD A, B
    LD (myscrn), A
    LD A, C
    LD (mydir), A
    POP AF
    OUT (6), A   ; Current Flash Page.
    POP IY
    POP IX
    POP HL
    POP DE
    POP BC
    POP AF

    EI      ; Enable Interrupts.
    RET

nextproc_IR:
    ; We need to get rid of the timer-interrupt-vars in the stack.
    POP HL
    LD (proc_timer_tempSP), HL
    POP HL
    LD (proc_timer_temp1), HL
    POP HL
    LD (proc_timer_temp2), HL

    ; Look up our entry.

    LD A, (proctabentry)
    LD E, A
    LD D, 0
    LD HL, $0000
    LD B, PROC_TAB_ENTRY
nextproc_loop:
    ADD HL, DE
    DJNZ nextproc_loop
    LD DE, (proctab)
    ADD HL, DE

    ; Put the stack-pointer into the table.

    INC HL
    INC HL
    LD (proctempSP), SP
    LD DE, (proctempSP)
    LD (HL), E
    INC HL
    LD (HL), D

    LD A, (proctabentry)
    LD C, A

nextproc_next:
    LD A, (proctabentry)
    INC A
    CP MAX_PROCS
    JR NZ, nextproc_next_cont
    XOR A
nextproc_next_cont:
    LD (proctabentry), A
    CP C
    ;JR Z, nextproc_sleep
    LD E, A
    LD D, 0
    LD HL, $0000
    LD B, PROC_TAB_ENTRY
nextproc_next_loop:
    ADD HL, DE
    DJNZ nextproc_next_loop
    LD DE, (proctab)
    ADD HL, DE
    LD A, (HL)
    OR A
    JR Z, nextproc_next
    LD (curproc), A       ; Save the new PID.
    INC HL
    LD A, (HL)
    OR A
    JR Z, nextproc_next

    ; Found a running process !

    INC HL        ; Get the stack-pointer.
    LD E, (HL)
    INC HL
    LD D, (HL)
    LD (proctempSP), DE
    LD SP, (proctempSP)

    ; Push back the timer variables.

    LD HL, (proc_timer_temp2)
    PUSH HL
    LD HL, (proc_timer_temp1)
    PUSH HL
    LD HL, (proc_timer_tempSP)
    PUSH HL

    RET     ; Run the process.


;** newproc
;* IN HL: Pointer to start address of process
;* OUT A: Garbage
;* OUT BC: Garbage
;* OUT DE: Garbage
;* OUT HL: Garbage
;* This function starts a new process
;* (which is very different from starting a program, see the OS Documentation page)
;* and returns the the place it was called from.

newproc:    ; Args: HL pointer to start address
    LD A, 1

newproc_start:
	    ;       A = 0 means started from newprocex

    ; When starting a process, we need to do a couple of things:
    ;
    ;  * Search for a unused PID
    ;  * Allocate memory for the stack.
    ;  * Set up the stack, with the endproc routine as last point.
    ;  * Put the process into the process table
    ;  * Return to the process which *made* the new one.

    DI      ; We don't want to be interrupted while doing process-table stuff.

    LD (newproc_mode), A

    LD (newproc_target), HL
    LD DE, PROC_TAB_ENTRY
    LD C, 0
    LD HL, $0000
    LD (newproc_empty), HL
newproc_restart:
    LD HL, (proctab)
    LD B, MAX_PROCS
    INC C
newproc_loop:
    LD A, (HL)
    CP C
    JR Z, newproc_restart
    OR A
    JR NZ, newproc_not_empty
    LD (newproc_empty), HL
newproc_not_empty:
    ADD HL, DE
    DJNZ newproc_loop

    ; We now have a new PID.

    LD A, (curproc)
    PUSH AF

    ; First: check if we found an empty spot in the table, saves trouble later on.
    LD HL, (newproc_empty)
    LD A, H
    OR L
    JR Z, newproc_error

    ; Now allocate memory for the stack, with the correct PID ofcourse...

    LD A, C
    LD (curproc), A
    LD HL, stack_size        ; see config.h
    CALL malloc
    LD A, H
    OR L
    JR Z, newproc_error
    LD DE, 128
    ADD HL, DE
    LD (proc_timer_tempSP), SP
    LD (proctempSP), HL
    LD SP, (proctempSP)

    ; Ok, init the stack.
    LD HL, endproc
    PUSH HL
    LD HL, (newproc_target)
    PUSH HL

    LD HL, $0100	; Push some registers into the stack (use $01 as flash page).
    PUSH HL
    PUSH HL
    PUSH HL
    PUSH HL
    PUSH HL
    PUSH HL

    LD A, (newproc_mode)	; Check if we need to use custom values for screen and flash page (by newprocex)
    OR A
    JP Z, newprocex_return

    PUSH HL			; If not, push flash page $01
    LD A, (myscrn)
    LD B, A
    LD A, (mydir)
    LD C, A
    PUSH BC			; And our parent's screen.

newproc_return:	    ; Newprocex returns here

    ; We already found an empty place in the proc. table.

    LD HL, (newproc_empty)
    LD A, (curproc)
    LD (HL), A
    LD A, (newproc_mode)                 ; Running
    INC HL
    LD (HL), A
    INC HL
    LD (proctempSP), SP
    LD DE, (proctempSP)
    LD (HL), E
    INC HL
    LD (HL), D

    ; Restore stack.

    LD SP, (proc_timer_tempSP)

    ; Restore PID and return:

newproc_error:
    POP AF
    LD (curproc), A
    EI
    RET


;** setprocname
;* IN A: The PID the name should be changed of.
;* IN HL: Pointer to string (new name of the process)
;* OUT A: Garbage.
;* OUT BC: Garbage.
;* OUT DE: Garbage.
;* OUT HL: Pointing to the end of the string, or $FFFF if the caller process doesn't exists (after which you better directly force a system reboot), which should never happen.
;* This function will name the current process, after which the given name can be found in the process-table.
;* If the given name is longer than the table can store, it will automatically be shortened.
;* It's called by the system when you start a program, but programs may rename themself too.

setprocname:
    PUSH HL
    LD HL, (proctab)
    LD DE, PROC_TAB_ENTRY
    LD B, MAX_PROCS
setprocname_loop:
    LD C, (HL)
    CP C
    JR NZ, setprocname_cont
    POP DE
    INC HL ; Proc State
    INC HL ; Stack ptr
    INC HL
    INC HL ; Proc name
    LD A, 11
    JP strncpy

setprocname_cont:
    ADD HL, DE
    DJNZ setprocname_loop
    POP HL
    LD HL, $ffff
    RET

;** endproc
;* This function terminates the process instandly and cleans up the system
;* (freeing memory, clean op process table). After this it switches to the next process.

endproc:    ; Only used by the system itself (or maybe too by the EXIT command)
    DI
    LD A, (proctabentry)
    LD E, A
    LD D, $00
    LD HL, (proctab)
    LD B, PROC_TAB_ENTRY
endproc_loop:
    ADD HL, DE
    DJNZ endproc_loop
    XOR A
    LD (HL), A            ; Clear table entry.

    ; We need to free all allocated memory of this process.
    LD A, (curproc)
    LD HL, $C000          ; MAT address.
    LD D, 0
    LD B, 0
endproc_loop2:
    LD C, (HL)
    CP C
    JR NZ, endproc_cont
    LD (HL), D		  ; Clear from index table
    INC H
    LD (HL), D		  ; Clear from usage index table.
    DEC H
endproc_cont:
    INC HL
    DJNZ endproc_loop2

    ; If we for whatever reason have the screen locked, remove it.
    LD B, A
    LD A, (screen_lock)
    CP B
    JP NZ, nextproc_no_push_args    ; Run the next process.
    XOR A
    LD (screen_lock), A
    JP nextproc_no_push_args    ; Run the next process.

;** newprocex
;* IN A: State of the new process (0 = sleeping, 1 = running)
;* IN B: Flash page
;* IN C: Virtual screen to start on.
;* IN DE: Pointer to the name of the new process (string)
;* IN HL: Pointer to start address of process
;* OUT A: Garbage.
;* OUT BC: Garbage.
;* OUT DE: Garbage.
;* OUT HL: Garbage.
;* Same as newproc, but with more options.

newprocex:
    PUSH AF
    PUSH BC
    PUSH DE
    XOR A
    JP newproc_start

newprocex_return:
    LD (proctempSP), SP
    LD SP, (proc_timer_tempSP)
    POP HL	; PID
    POP DE
    POP BC
    POP AF
    PUSH HL	; restore PID
    LD (newproc_mode), A	; Active or asleep
    LD (proc_timer_tempSP), SP
    LD SP, (proctempSP)
    XOR A			; Flash page
    PUSH AF
    PUSH BC                     ; Screen + Directory

    LD HL, (newproc_empty)	; Copy our name.
    INC HL
    INC HL
    INC HL
    INC HL
    LD A, 11
    CALL strncpy
    JP newproc_return
