;*** memmngt.h
; **** Memory manager ****

; We have $8000 - $ffff of memory.
; We need to save some for System stack.
; We use 128 bytes / block
; We use 1 block of system stack.
; Block index is 1 byte / block = process num (or 0)
; 2 blocks of index (2*128*128=$8000)

; New situation:
; We use 128 bytes / block
; We use 2 blocks of system stack.
; Block index is located at $C000 and 2 blocks in size (2*128*128=$8000) (PID)
; Block usage index is located at $C100 and also 2 blocks in size 	 (used blocks)


;** mem_init
;* Called by the OS at boot time to initialize the memory manager.
mem_init:
  ; Write Memory Allocation Table.
    DI
    LD (temp), SP
    LD SP, $C200
    LD B, 0
    LD HL, 0
mem_init_clear:
    PUSH HL
    DJNZ mem_init_clear
    LD SP, (temp)
    LD A, 1

    ; Reserve memory blocks:
    LD ($C080), A ; Index
    LD ($C081), A ; Index
    LD ($C082), A ; Usage Index
    LD ($C083), A ; Usage Index
    LD ($C084), A ; System vars
    LD ($C0fe), A ; System Stack
    LD ($C0ff), A ; System Stack


    INC A
    LD ($C1fe), A
    LD A, 5
    LD ($C180), A

    LD HL, MesMemMngt
    CALL puts
    RET

; Function for memory allocation, HL is needed bytes:

;** pgmalloc
;* IN HL: Bytes needed
;* OUT A: Garbage
;* OUT B: Garbage
;* OUT DE: Garbage
;* OUT HL: Pointer to memory at succes, zero at failure.
;* Same as <code>malloc</code>, but returns returns only pointers to memory that is executable.

pgmalloc:
    LD B, 3
    JP malloc_test

malloc_di:
    LD A, (mallocable)
    OR A
    JP NZ, malloc_deadlock
    LD A, B
    LD (mallocable), A
    JR malloc_able_di

;** malloc
;* IN HL: Bytes needed
;* OUT A: Garbage
;* OUT B: Garbage
;* OUT DE: Garbage
;* OUT HL: Pointer to memory at succes, zero at failure.
;* <code>malloc</code> can be called by any process that needs some (more) memory.
;* It returns a pointer to a bit of free memory that will only be used by the process that allocated it.
;* After the process is done using it, it should call <code>free</code> with the same pointer.

malloc:

    LD B, 1

malloc_test:
    ; First check if we are allowed to run here (we don't want different programs overwrite each memory).

    LD A, I
    JP PO, malloc_di

    DI
    LD A, (mallocable)
    OR A
    JR Z, malloc_able
    EI
    CALL nextproc
    JP malloc_test

malloc_able:
    LD A, B
    LD (mallocable), A
    EI

malloc_able_di:
    SLA L       ; HL * 2
    RL H
    LD B, H     ; H = needed blocks.
    XOR A
    ADD A, L
    JR Z, malloc_cont1    ; If there is a rest
    INC B                 ; add 1 to the needed blocks.
malloc_cont1:
    PUSH BC
    LD HL, $Bfff
    LD A, (mallocable)
    CP 1
    JR NZ, malloc_loop
    LD HL, $C07f

malloc_loop:
    INC HL

    LD A, (mallocable)
    CP 3
    JR NZ, malloc_loop_cont
    LD A, L
    CP $80
    JR Z, malloc_error

    LD A, H
    CP $C1
    JR Z, malloc_error

malloc_loop_cont:
    LD A, (HL)
    OR A
    JR NZ, malloc_reset
    DJNZ malloc_loop

    ; Found it !!
    LD A, L      ; Save block number
    POP BC
    SUB B        ; Minus needed blocks...
    INC A        ;                 ... - 1.
    LD L, A      ; Save it in HL
    LD A, (curproc)
    PUSH HL
    INC H
    LD (HL), B	; Writeout number of blocks
    DEC H
malloc_flag_loop:
    LD (HL), A   ; Flag it as used.
    INC HL
    DJNZ malloc_flag_loop
    POP HL
    LD H, $00
    LD DE, $8000
    ADD HL, HL   ; 2 bytes / block
    ADD HL, HL   ; 4 bytes / block
    ADD HL, HL   ; 8 bytes / block
    ADD HL, HL   ; 16 bytes / block
    ADD HL, HL   ; 32 bytes / block
    ADD HL, HL   ; 64 bytes / block
    ADD HL, HL   ; 128 bytes / block
    ADD HL, DE   ; + Base address.
    JR malloc_end
malloc_error:
    LD A, (mallocable)
    CP 1
    JP Z, fullmalloc
    LD HL, $0000
    POP BC
malloc_end:
    XOR A
    LD (mallocable), A
    RET
malloc_reset:
    POP BC
    PUSH BC
    JR malloc_loop

fullmalloc:
    POP BC
    LD A, 2
    LD (mallocable), A
    JP malloc_able		; restart for full search (including executable memory)

malloc_deadlock:
    ; To keep things stable, I have right now no other solution...
    LD HL, $0000
    RET

;** calloc
;* IN HL: Bytes needed
;* OUT A: Garbage
;* OUT B: Garbage
;* OUT DE: Garbage
;* OUT HL: Pointer to memory at succes, zero at failure.
;* Called does the same as <code>malloc</code>, but will initialize the memory with zeros.

calloc:
    PUSH HL
    CALL malloc
    POP BC
    LD A, H
    OR L
    RET Z       ; Return on Exit.
    LD D, B
    LD E, C

; It's better to not disable interrupts.

;    RR B
;    SRL C
;    JR NC, calloc_cont
;    INC BC
;calloc_cont:
;    PUSH HL
;    ADD HL, DE
;    DI
;    LD (callocSPbak), SP
;    LD (callocSPtemp), HL
;    LD SP, (callocSPtemp)
;    LD HL, $0000
;calloc_loop:
;    DEC BC
;    PUSH HL
;    LD A, B
;    OR C
;    JR NZ, calloc_loop
;    LD SP, (callocSPbak)
;    POP HL
;    RET

    ADD HL, DE
calloc_loop:
    DEC BC
    DEC HL
    LD (HL), 0
    LD A, B
    OR C
    JR NZ, calloc_loop
    RET


;** free
;* IN HL: Pointer to previous allocated memory
;* OUT A: Garbage
;* OUT B: Garbage
;* OUT DE: Always $8000
;* OUT HL: Zero on succes, $FFFF on failure.
;* Frees memory that was allocated by this process (using <code>malloc</code>, <code>calloc</code> or <code>pgmalloc</code>.
;* If it was not allocated by this process, HL will be $FFFF.

free:   ; Args: pointer to memory: HL
    LD DE, $8000
    ADD HL, DE      ; This subtracts the leading $8000 of the pointer.

    LD B, 7
free_loop:
    SRL H
    RR L
    DJNZ free_loop

    ; We now have the offset of the table entry.

    LD DE, $C000
    ADD HL, DE      ; Adding begin of the table.

    LD B, (HL)
    LD A, (curproc)
    CP B
    JR NZ, free_error

    INC H
    XOR A
    OR (HL)
    JR Z, free_error
    LD B, A
    XOR A
    LD (HL), A
    DEC H
free_freeloop:
    LD (HL), A
    INC HL
    DJNZ free_freeloop

    LD HL, $0000
    RET

free_error:
    LD HL, $ffff
    RET
