;*** fs_high.h
; Higher level of the filesystem driver (like support for paths.)

;** findfile
;* Tries to find a file in the filesystem.
;* IN HL: Filename (full or relative path is also possible).
;* OUT A: File's flags.
;* OUT BC: Length of the file.
;* OUT DE: Garbage.
;* OUT HL: Pointer to the file's data on success, $FFFF if the file couldn't be found, $FFFE if there was a problem resolving the path.
;* OUT IX: Pointer to the file-header.

findfile:       ; HL is path to file.
    PUSH HL
    POP IX
    LD A, (mydir)
    LD D, A
    LD A, '/'
    CP (HL)
    JR NZ, findfile_loop
    LD D, 0
    INC IX

findfile_loop:
     JP findfile_nopoint

;    XOR A
;    CP (IX+0)
;    JP Z, findfile_failed
;    LD A, '.'
;    CP (IX+0)
;    JR NZ, findfile_nopoint
;    LD A, '/'
;    CP (IX+1)
;    JR Z, findfile_inc2
;    LD A, '.'
;    CP (IX+1)
;    JR NZ, findfile_nopoint
;    LD A, '/'
;    CP (IX+2)
;    JR NZ, findfile_nopoint
;    XOR A
;    OR D
;    JR Z, findfile_inc3
;    CALL findparentdir

;findfile_inc3:
;    INC IX
;findfile_inc2:
;    INC IX

findfile_inc1:
    INC IX
    JR findfile_loop

findfile_nopoint:
    LD A, '/'
    CP (IX+0)
    JR Z, findfile_inc1

    ; Already done before.
    ;XOR A
    ;CP (IX+0)
    ;JR Z, findfile_failed

    PUSH IX
    POP HL
findfile_smallloop:
    XOR A
    CP (HL)
    JR Z, findfile_findfile
    LD A, '/'
    CP (HL)
    JR Z, findfile_finddir
    INC HL
    JR findfile_smallloop

findfile_finddir:

            ; Ok, we have to copy the dirname to a place in the RAM,
            ; otherwise we will lose information when writing a '\0'
            ; to the string (if it's in the RAM, otherwise we will
            ; just get a file-not-found error.

    ; First calculate the length.
    PUSH DE

    PUSH IX
    POP DE
    SCF
    CCF
    SBC HL, DE
    PUSH HL
    CALL malloc   ; Alloc memory
    LD A, H
    OR L
    JR Z, findfile_finddir_nomem

    ; HL = pointer to RAM
    ; Stack: [ Length ] | [ dir | garbage ]
    ; BC is "empty"

    POP BC
    LD B, C     ; A filename longer than 255 bytes? I don't think so.
    PUSH IX
    POP DE
    PUSH HL
findfile_finddir_loop:
    LD A, (DE)
    LD (HL), A
    INC HL
    INC DE
    DJNZ findfile_finddir_loop
    XOR A
    LD (HL), A
    POP HL

    ; Stack:  [ dir | garbage ]
    ; HL = ptr to str in RAM
    ; DE = End of string (SRC)
    ; IX = Start of string (SRC)

    POP BC
    INC DE
    PUSH DE
    PUSH HL  ; Stack  [ ptr to str in RAM ] | [ ptr to next "file" ]

    CALL findsym
    PUSH AF
    LD A, H
    AND L
    CP $ff
    JR Z, findfile_faildir
    POP AF
    AND %10000000
    JR NZ, findfile_faildir2
    LD D, (HL)
    POP HL
    PUSH DE
    CALL free
    POP DE
    POP IX
    JP findfile_loop

findfile_findfile:
    PUSH IX
    POP HL
    LD B, D
    JP findsym

findfile_faildir:
    POP HL
findfile_faildir2:
    POP HL
    CALL free
    POP HL

findfile_failed:
    LD HL, $fffe    ; Return $FFFE, because $FFFF means that the actual file doesn't exists (which is handy when creating one).
    RET

findfile_finddir_nomem:
    POP HL
    POP HL
    POP HL
    JP findfile_failed


;** writefile
;* Creates a new file.
;* If the file already exists, it is overwritten.
;* All files written by this function are not executable.
;* IN HL: The file's path (Must reside in the RAM!!)
;* IN IX: Pointer to the file's data (Must reside in the RAM!!)
;* IN BC: The length of the file (in bytes).
;* OUT A: Garbage.
;* OUT BC: Garbage.
;* OUT DE: Garbage.
;* OUT HL: $0000 on success, $FFFF on failure.
;* OUT IX: Garbage.
;* OUT IY: Garbage.

writefile:
    PUSH BC
    PUSH IX
    PUSH HL
    CALL findfile
    LD A, H
    CP $FF
    JR Z, writefile_nounlink
    POP HL
    PUSH HL
    CALL unlink
    LD A, H
    CP $FF
    JR Z, writefile_failed
    POP HL
    PUSH HL
    CALL findfile
writefile_nounlink:
    LD A, L
    AND H
    CP $FF
    JR NZ, writefile_failed
    POP HL
    CALL basename
    POP IX
    LD D, B
    LD E, %10000000   ; Normal file.
    POP BC
    JP mkfile

writefile_failed:
    POP HL
    POP HL
    POP HL
    LD HL, $FFFF
    RET

;** startfile
;* Loads a file from the filesystem into the memory and starts executing that file.
;* The program will be started in a seperate process, and the called will imidiatly resume operation.
;* See also <code>execfile</code>.
;* For more options see <code>startfileex</code>.
;* IN HL: Path to the program
;* OUT HL: $0000 on success, $FFFD if the file couldn't be found, $FFFE if the file isn't executable, $FFFF if an other error occurs.
;* OUT A: Garbage
;* OUT BC: Garbage
;* OUT DE: Garbage
;* OUT IX: Garbage


startfile:  ; Same as execfile, but "forks".
    LD A, 1
    JP execfile_start

;** execfile
;* Loads a file from the filesystem into the memory and starts executing that file.
;* The loaded program will take over control.
;* If the program exits in a normal way, execution will be resumed.
;* See also <code>startfile</code>.
;* IN HL: Path to the program.
;* OUT HL: $FFFD if the file couldn't be found, $FFFE if the file isn't executable, $FFFF if an other error occurs. Depends on the program on success.
;* OUT A: Depends on the program.
;* OUT BC: Depends on the program.
;* OUT DE: Depends on the program.
;* OUT IX: Depends on the program.
;* OUT IY: Depends on the program.

execfile:   ; Find a file and execute if that's possible.
    LD A, 0
execfile_start:
    PUSH AF
    CALL findfile
    PUSH AF
    LD A, H
    ;AND L    ; Error can also be $FFFE
    CP $ff
    JR Z, execfile_nofile
    POP AF
    CP %11000000
    JR NZ, execfile_noexec
    POP AF
    OR A
    JP Z, loadpgm
    CP 1
    JP Z, startpgm
    RET

execfile_nofile:
    POP AF
    POP AF
    LD HL, $FFFE
    RET
execfile_noexec:
    POP AF
    LD HL, $FFFD
    RET

;** startfileex
;* Loads a file from the filesystem into the memory and starts executing that file.
;* The program will be started in a seperate process, and the called will imidiatly resume operation.
;* Has more options than <code>startfile</code>.
;* See also <code>execfile</code> and <code>startfile</code>
;* IN B: Initial flash page that will be in use by the program.
;* IN C: Virtual screen to run on.
;* IN DE: Pointer to name of the program.
;* IN HL: Path to the program
;* OUT HL: $0000 on success, $FFFD if the file couldn't be found, $FFFE if the file isn't executable, $FFFF if an other error occurs.
;* OUT A: Garbage
;* OUT BC: Garbage
;* OUT DE: Garbage
;* OUT IX: Garbage

startfileex:  ; Same as startfile, but accepts all the startpgmex arguments.
    PUSH BC
    PUSH DE
    LD A, 2
    CALL execfile_start
    POP DE
    POP BC
    LD A, H
    CP $ff    ; This is an error since we lookup in flash.
    RET Z
    JP startpgmex

;** load
;* Loads a file from the flash to the RAM (for editing and reading purposes).
;* IN DE: Pointer to a place in the RAM to load the file to (or $0000 to let the function allocate memory).
;* IN HL: Path to the file to load.
;* OUT A: file flags
;* OUT BC: file length
;* OUT HL: Start of the file in the RAM on success, $FFFF or $FFFE on failure (see also <code>findfile</code>).
;* OUT DE: Pointer to the last byte of the file in the RAM + 1.
;* OUT IX: Pointer to the file-header in the flash.

load:
    PUSH DE
    CALL findfile
    POP DE

    ; Register summary:
    ;
    ; A = file flags
    ; BC = file length
    ; DE = RAM destination (or $0000, for malloc)
    ; HL = start of the file (of $FFFx if the file doesn't exists)

    LD A, H
    CP $FF
    RET Z

    LD A, D
    OR E
    CALL Z, load_malloc

    PUSH DE
    PUSH BC
    LDIR
    POP BC
    POP HL
    RET

load_malloc:
    PUSH BC
    PUSH HL
    LD H, B
    LD L, C
    CALL malloc
    LD A, H
    OR L
    JR Z, load_malloc_err
    LD D, H
    LD E, L
    POP HL
    POP BC
    RET

load_malloc_err:
    POP HL          ; HL
    POP HL          ; BC
    POP HL          ; RET
    LD HL, $FFFD    ; Error
    RET
