;*** fs.h
; The FileSystem driver, build upon the Flash-driver.

;** mkdir
;* Creates a new directory.
;* IN D: Directory IDentity of the parent directory (the directory the new one will reside in).
;* IN HL: The name of the new directory.
;* OUT A: Garbage
;* OUT BC: Garbage
;* OUT DE: Garbage
;* OUT HL: $0000 on success, $FFFF on failure.
;* OUT IY: Garbage

mkdir:
    PUSH HL
    PUSH DE
    LD HL, 16+1
    CALL malloc
    LD A, H
    OR L
    JR Z, mkdir_nomem
    PUSH HL
    POP IY
    POP DE
    LD (IY+0), $FF    ; We're alive
    LD (IY+1), $FE    ; One block in use
    LD (IY+2), D      ; Directory
    LD (IY+3), $0d    ; Directory-flag + size = 13

    LD DE, 04
    ADD HL, DE
    POP DE
    LD A, 12
    CALL strncpy
    CALL seeknewdirid
    XOR A
    OR L
    JR NZ, mkdir_noempty
    LD (IY+16), D
    CALL mkdir_wrtblk

    ; New way dirs are handled: also create ./ and ../

    LD D, (IY+16)
    LD E, (IY+2)
    LD (IY+2), D
    LD (IY+16), E
    LD (IY+4), '.'
    LD (IY+5), '.'
    LD (IY+6), 0
    CALL mkdir_wrtblk

    LD A, (IY+2)
    LD (IY+16), A
    LD (IY+5), 0
    CALL mkdir_wrtblk
    PUSH IY
    POP HL
    CALL free
    LD HL, $0000
    RET

mkdir_wrtfail:
    POP HL
mkdir_noempty:
    PUSH IY
    POP HL
    CALL free
    LD HL, $ffff
    RET

mkdir_nomem:
    POP BC
    POP BC
    LD HL, $FFFF
    RET

mkdir_wrtblk:
    XOR A
    LD B, 1
    CALL falloc
    LD A, L
    AND H
    CP $FF
    JR Z, mkdir_wrtfail
    LD D, H
    LD E, L
    PUSH IY
    POP HL
    CALL writeblk
    LD A, H
    AND L
    CP $FF
    JR Z, mkdir_wrtfail
    RET


;** basename
;* Strips a filename from a complete path (i.e. "file" from "/a/directory/with/a/file")
;* IN HL: A full path
;* OUT HL: Pointer to the start of the filename in the given string.

basename:
    PUSH AF
    PUSH BC
    PUSH HL
basename_cont:
    LD A, (HL)
    INC HL
    CP '\0'
    JR Z, basename_end
    CP '/'
    JR NZ, basename_cont
    POP BC
    PUSH HL
    JR basename_cont
basename_end:
    POP HL
    POP BC
    POP AF
    RET

;** unlink
;* Deletes ("kills") a file from the filesystem.
;* In case of a directory, all files and directories inside it are also deleted.
;* IN HL: Path to the file.
;* OUT A: Garbage
;* OUT BC: Garbage
;* OUT DE: Garbage
;* OUT HL: $FFFF on failure, something else on success.
;* OUT IX: Garbage
;* OUT IY: Garbage

unlink:
    CALL findfile
    LD A, H
    CP $FF
    RET Z
    LD DE, -16
    ADD HL, DE  ; HL is now the dead block field
unlink_recursive:
    PUSH HL
    POP IY

    ; First check if this is a special (i.e. "." or "..") directory
    LD A, '.'
    CP (IY+4)
    JR NZ, unlink_remove
    CP (IY+5)
    JR NZ, unlink_check2
    XOR A
    CP (IY+6)
    JR Z, unlink_fail
    JR unlink_remove

unlink_check2:
    XOR A
    CP (IY+5)
    JR NZ, unlink_remove

unlink_fail:
    LD HL, $FFFF
    RET

unlink_remove:
    LD B, (IY+3)
    LD C, (IY+16)   ; Directory ID (when it's a dir ofcourse).
    PUSH BC
    LD D, H
    LD E, L
    LD HL, Zero
    CALL writebyte
    LD A, H
    CP $FF
    JR Z, unlink_ret
    POP BC
    LD A, B
    AND %10000000

    RET NZ

    ; It is a directory, so we need to delete all files in it.
    ; We call unlink, because if it's a subdir, we need to del
    ; that's contains too.

    LD DE, 64
    LD A, 4
unlink_page_loop:
    OUT (6), A
    PUSH AF
    LD HL, $4000
unlink_loop:
    PUSH HL
    POP IX
    LD A, $FF
    CP (HL)
    JR NZ, unlink_next
    LD A, (IX+2)
    CP C
    JR NZ, unlink_next

    ; Special dirs (i.e. "." and "..") should be deleted manually.
    LD A, '.'
    CP (IX+4)
    JR NZ, unlink_dorecursive
    CP (IX+5)
    JR NZ, unlink_nextcheck
    XOR A
    CP (IX+6)
    JR Z, unlink_manual

unlink_dorecursive:
    PUSH BC
    PUSH DE
    PUSH HL
    PUSH IX
    CALL unlink_recursive
    POP IX
    POP HL
    POP DE
    POP BC
unlink_next:
    LD A, (IX+1)
    CPL
    OR A
    LD B, A
    JR NZ, unlink_calcloop
    INC B
unlink_calcloop:
    ADD HL, DE
    DJNZ unlink_calcloop
    LD A, H
    AND $80
    JR Z, unlink_loop
    POP AF
    INC A
    CP 20
    JR NZ, unlink_page_loop
    RET

unlink_ret2:
    POP AF
    POP AF
    POP AF
    POP AF
unlink_ret:
    POP AF
    RET

unlink_nextcheck:
    XOR A
    CP (IX+5)
    JR NZ, unlink_dorecursive

unlink_manual:
    PUSH BC
    PUSH DE
    PUSH HL
    PUSH IX
    LD D, H
    LD E, L
    LD HL, Zero
    CALL writebyte
    LD A, H
    CP $FF
    JR Z, unlink_ret2
    POP IX
    POP HL
    POP DE
    POP BC
    JR unlink_next


;** mkfile
;* Creates a new file.
;* IN D: Directory IDentity of the parent directory (directory the file should be placed in).
;* IN HL: The name of the file (Should reside in the RAM!!).
;* IN IX: A pointer to the file's data (Also somewhere in the RAM!!).
;* IN BC: The length of the file (in bytes).
;* IN E: The flags of the file.
;* OUT A: Garbage.
;* OUT BC: Garbage.
;* OUT DE: Garbage.
;* OUT HL: $0000 on success, $FFFF on failure.
;* OUT IX: Garbage.
;* OUT IY: Garbage.

mkfile:

    ; First add the file's header to BC:

    PUSH HL
    LD HL, 16
    ADD HL, BC
    LD B, H
    LD C, L

    ; Calculate the blocks needed (length / blocksize (= 64))
    SLA L
    RL H
    SLA L
    RL H
    XOR A
    OR L
    JR Z, mkfile_skipadd
    INC H
mkfile_skipadd:     ; H is now the amount of blocks needed.

    PUSH BC
    PUSH DE
    PUSH HL
    LD B, H
    XOR A           ; Disable sector skip.
    CALL falloc
    LD A, H
    AND L
    CP $ff
    JP Z, mkfile_noroom   ; HL is now the block in [page|blknr] format.

    ; Create first block in memory.

    PUSH HL   ; Stack now contains: [First Block] : [ Blocks needed | Garbage ] : [ Directory | Flags ] : [ Length of data ] : [ Pointer to filename ]
    LD HL, 64
    CALL malloc
    LD A, H
    OR L
    JP Z, mkfile_nomem
    PUSH HL
    POP IY
    LD (IY+0), $FF  ; Alife?
    POP HL
    POP AF
    LD B, A         ; Save the needed blocks
    CPL
    LD (IY+1), A    ; Blocks in use
    POP DE
    LD (IY+2), D    ; Directory
    LD A, %11000000
    AND E
    LD E, A
    LD D, B         ; Save the needed blocks
    POP BC
    LD A, %00111111
    AND C
    OR E
    LD (IY+3), A    ; Flags

    ; Now follows the 12 character filename

    LD B, D
    POP DE
    PUSH BC
    PUSH HL   ; Stack: [ First block ] : [ Needed blocks | Garbage ]
    PUSH IY
    POP BC    ; BC = first block (in memory)
    LD HL, 4
    ADD HL, BC
    LD A, 12
    CALL strncpy

    ; Now copy the first 64 - 16 = 48 bytes.

    LD BC, 48
    LD HL, 16
    PUSH IY
    POP DE
    ADD HL, DE
    EX DE, HL
    PUSH IX
    POP HL
    LDIR
    LD B, H
    LD C, L     ; Save data stream.

    ; First block is ready to be written.
    POP DE
    PUSH DE
    PUSH BC
    PUSH IY
    PUSH IY
    POP HL        ; Stack: [ first block (in mem) ] : [ data ] : [ First block (in flash) ] : [ blocks needed | garbage ]
    CALL writeblk
    LD A, H
    AND L
    CP $ff
    JR Z, mkfile_wrterr1
    POP HL
    CALL free

    POP HL
    POP DE
    POP BC
    INC DE
    DJNZ mkfile_wrtcont
    LD HL, $0000
    RET

mkfile_wrtcont:
    PUSH BC
    PUSH DE
    PUSH HL
    CALL writeblk
    LD A, H
    AND L
    CP $ff
    JR Z, mkfile_wrterr2
    POP DE
    LD HL, 64
    ADD HL, DE
    POP DE
    INC DE
    POP BC
    DJNZ mkfile_wrtcont
    LD HL, $0000
    RET

mkfile_wrterr1:
    CALL free
    LD HL, $FFFF
mkfile_wrterr2:
    POP BC
    POP BC
    POP BC
    RET

mkfile_nomem:
    LD HL, $ffff
    POP BC
mkfile_noroom:
    POP BC
    POP BC
    POP BC

    ; FIXME: In the future, we need to put sector cleaning in here.

    RET


;** findsym
;* This function will lookup a certain file in the filesystem. It's called by <code>findfile</code>.
;* IN HL: Name of the file.
;* IN B: Directory IDentifier of the file's directory.
;* OUT A: The file's flags.
;* OUT BC: The file's length (in bytes).
;* OUT DE: Garbage.
;* OUT HL: Pointer to the file's data (or $FFFF on failure).
;* OUT IX: Pointer to the file-header.

findsym:
    PUSH HL
    PUSH BC
    LD A, 3
    PUSH AF
findsym_outer_loop:
    POP AF
    INC A
    CP 20
    JR Z, findsym_not_found
    PUSH AF

    OUT (6), A  ; Switch to flash page.
    LD IX, $4000

findsym_middle_loop:
    XOR A
    OR (IX+0)
    JR Z, findsym_nextfile
    LD A, (IX+1)
    CP $FF
    JR Z, findsym_nextfile

    ; Now test if this file is what we where looking for.

    POP AF
    POP BC
    POP HL
    PUSH HL
    PUSH BC
    PUSH AF

    LD A, (IX+2)  ; Check directory.
    CP B
    JR NZ, findsym_nextfile

    PUSH IX
    POP DE
    INC DE
    INC DE
    INC DE
    INC DE

    LD A, 12
    CALL strncmp
    LD A, H
    OR A
    JR NZ, findsym_nextfile

    ; We found it !! Calculate size.

    LD A, (IX+1)
    CPL
    LD DE, 64
    LD HL, 0-80
    LD B, A

findsym_sizeloop:
    ADD HL, DE
    DJNZ findsym_sizeloop

    LD D, 0
    LD A, (IX+3)
    AND %00111111
    LD E, A
    ADD HL, DE

    LD B, H
    LD C, L       ; BC is now the size.

    LD DE, 16
    PUSH IX
    POP HL
    ADD HL, DE    ; HL is now the start of the file.

    LD A, (IX+3)
    AND %11000000 ; A is now the flags.

    POP DE
    POP DE        ; DE is now garbage ;-)
    POP DE

    RET

findsym_nextfile:
    LD A, (IX+1)    ; Blocks in use
    CPL             ; Invert it to a normal number
    OR A
    JR NZ, findsym_nextfile_cont
    INC A
findsym_nextfile_cont:
    LD B, A
    LD DE, 64
    PUSH IX
    POP HL
findsym_nextfile_loop:
    ADD HL, DE
    DJNZ findsym_nextfile_loop
    PUSH HL
    POP IX
    LD A, $80
    AND H
    JR Z, findsym_middle_loop
    JR findsym_outer_loop

findsym_not_found:
    LD HL, $ffff
    POP BC        ; Return directory we were looking in.
    POP DE
    RET

;** seeknewdirid
;* Tries to find a Directory IDentifier that is not already in use.
;* OUT A: Garbage
;* OUT BC: Garbage
;* OUT D: New DID on success.
;* OUT HL: $0000 on success, $FFFF on failure.
;* OUT IX: Garbage

seeknewdirid:
    LD D, 1
seeknewdirid_outerloop:
    LD A, 4
seeknewdirid_again:
    OUT (6), A
    PUSH AF
    LD HL, $4000
seeknewdirid_loop:
    PUSH HL
    POP IX
    XOR A
    OR (IX+0)
    JR Z, seeknewdirid_nextblock
    OR (IX+3)
    BIT 7, A
    JR Z, seeknewdirid_nextblock
    LD A, (IX+16)
    CP D
    JR Z, seeknewdirid_advance
seeknewdirid_nextblock:
    LD A, (IX+1)
    CPL
    OR A
    LD BC, 64
    JR NZ, seeknewdirid_advloop
    INC A
seeknewdirid_advloop:
    ADD HL, BC
    DEC A
    JR NZ, seeknewdirid_advloop
    LD A, H
    BIT 7, A
    JR Z, seeknewdirid_loop
    POP AF
    INC A
    CP 16
    JR NZ, seeknewdirid_again
    LD HL, $0000
    RET

seeknewdirid_advance:
    POP AF
    INC D
    XOR A
    OR D
    JR NZ, seeknewdirid_outerloop
    LD HL, $FFFF
    RET

;** findparentdir
;* Returns the parent directory of a specified directory.
; IN D: DID of the specified directory.
; OUT A: Garbage
; OUT D: DID of the parent directory, or $00 (root-directory) on failure.
; OUT E: Garbage
; OUT HL: $0000 on success, $FFFF on failure.
; OUT IX: Pointer to the start of the file-header of the specified directory (on success).

findparentdir:  ; Find the parent of the current directory
                ; IN D: current directory, OUT D: parent directory.

    IN A, (6)
    LD E, A
    LD A, 4
    OUT (6), A
    PUSH AF
findparentdir_loop:
    LD HL, $4000
findparentdir_loop_loop:
    PUSH HL
    POP IX
    XOR A
    OR (IX+0)
    JR Z, findparentdir_loop_loop_next
    LD A, (IX+3)    ; Flags
    AND %10000000
    JR NZ, findparentdir_loop_loop_next
    LD A, (IX+16)
    CP D
    JR NZ, findparentdir_loop_loop_next
    LD A, '.'
    CP (IX+4)
    JR Z, findparentdir_check
findparentdir_return:
    POP HL
    LD D, (IX+2)
    LD HL, $0000
    JR findparentdir_exit

findparentdir_check:
    XOR A
    CP (IX+5)
    Jr Z, findparentdir_loop_loop_next
    LD A, '.'
    CP (IX+5)
    JR NZ, findparentdir_return
    XOR A
    CP (IX+6)
    JR NZ, findparentdir_return

findparentdir_loop_loop_next:
    LD A, (IX+1)
    CPL
    OR A
    JR NZ, findparentdir_noadd
    INC A
findparentdir_noadd:
    LD B, A
    PUSH DE
    LD DE, 64
findparentdir_addloop:
    ADD HL, DE
    DJNZ findparentdir_addloop
    POP DE
    LD A, H
    AND $80
    JR Z, findparentdir_loop_loop
    POP AF
    INC A
    OUT (6), A
    PUSH AF
    CP 20
    JR NZ, findparentdir_loop
    POP AF
    LD D, 0
    LD HL, $FFFF

findparentdir_exit:
    LD A, E
    OUT (6), A
    RET

;** checkfs
;* Called bye the system at boot-time, checks the filesystem for errors.
;* Errors are automatically repaired.
;* If too much errors are found, the system will format the filesystem and reboot.

checkfs:    ; Check the filesystem (used by the firstrun program)

    LD BC, 0
    LD A, 4
checkfs_loop1:
    OUT (6), A
    PUSH AF
    LD IX, $4000
checkfs_loop2:
    PUSH IX
    POP HL
    LD A, C
    CP H
    LD C, H
    CALL NZ, checkfs_update_spinner

    LD A, (IX+0)
    CP $00
    JR Z, checkfs_cont1
    CP $FF
    JR Z, checkfs_cont1

    ; Error: block is not alive nor death.
    JP checkfs_disable

checkfs_cont1:
    LD A, (IX+0)
    OR A    ; Important: if this block is not in use, skip it.
    JP Z, checkfs_lastcont

    PUSH IX
    POP HL

    ; Every block is 64 bytes.
    ; * 4 >> H = current block

    SLA L
    RL H
    SLA L
    RL H

    ; Since (IX+1) is inverted, it may not be less then H
    ; If it's the same, this means it covers the whole the page from now on.

    LD A, (IX+1)
    CP H

    ; If it's less, then carry = 1
    JP C, checkfs_disable

checkfs_cont2:    ; Most important, make sure a block marked clean is clean!!
    LD A, (IX+0)
    AND (IX+1)
    INC A
    JR NZ, checkfs_cont3
    PUSH BC
    LD B, 62
    PUSH IX
    POP HL
    INC HL
    LD A, $FF
checkfs_cont2_loop:
    INC HL
    CP (HL)
    JP NZ, checkfs_disable_popBC
    DJNZ checkfs_cont2_loop
    POP BC

checkfs_cont3:
    LD A, $FF
    CP (IX+1)
    JP Z, checkfs_lastcont    ; Jump away if this block is not in use
    Ld A, (IX+3)
    AND %1000000
    JP NZ, checkfs_cont3_normaldir
    LD A, '.'
    CP (IX+4)
    JR NZ, checkfs_cont3_normaldir
    CP (IX+5)
    JR NZ, checkfs_cont3_nextcheck
    XOR A
    CP (IX+6)
    JR Z, checkfs_cont3_updir
    JR checkfs_cont3_normaldir

checkfs_cont3_nextcheck:
    XOR A
    CP (IX+5)
    JR NZ, checkfs_cont3_normaldir
    LD A, (IX+2)
    CP (IX+16)
    JR NZ, checkfs_cont3_unlinkparent
    LD D, (IX+2)
    XOR A
    OR D
    JR Z, checkfs_lastcont
    PUSH IX
    PUSH BC
    CALL findparentdir
    POP BC
    POP IX
    LD A, H
    CP $FF
    JP Z, checkfs_disable
    JR checkfs_lastcont

checkfs_cont3_updir:
    ; TODO: check if this dir points to our updir
    LD D, (IX+2)
    XOR A
    OR D
    JR Z, checkfs_cont3_rootupdir
    PUSH IX
    PUSH BC
    CALL findparentdir
    POP BC
    POP IX
    LD A, H
    CP $FF
    Jr Z, checkfs_cont3_unlinkparent
    LD A, D
    CP (IX+16)
    JR NZ, checkfs_cont3_unlinkparent
    JR checkfs_lastcont

checkfs_cont3_rootupdir:
    CP (IX+16)
    JR NZ, checkfs_disable
    JR checkfs_lastcont

checkfs_cont3_normaldir:
    ; TODO: check for ./ and ../
    ; ! Watch out ! :  file-check code also runs here !
    PUSH IX
    PUSH BC
    LD D, (IX+2)

checkfs_cont3_normaldir_loop:
    XOR A
    CP D
    JR Z, checkfs_cont3_normaldir_cont
    CALL findparentdir
    LD A, H
    CP $FF
    JR Z, checkfs_cont3_unlink
    JR checkfs_cont3_normaldir_loop

checkfs_cont3_normaldir_cont:
    POP BC
    POP IX
    JR checkfs_lastcont

checkfs_cont3_unlinkparent:
    PUSH IX
    LD D, (IX+2)
    PUSH BC
    CALL findparentdir
    POP BC
    PUSH IX
    LD A, H
    POP HL
    POP IX
    CP $FF
    JR Z, checkfs_disable
    JR checkfs_cont3_unlink2

checkfs_cont3_unlink:
    POP BC
    POP IX
    PUSH IX
    POP HL
checkfs_cont3_unlink2:
    PUSH IX
    PUSH BC
    CALL unlink_recursive
    POP BC
    POP IX
checkfs_incb:
    INC B
    JP Z, checkfs_format

checkfs_lastcont:
    PUSH BC
    LD A, (IX+1)
    CPL
    OR A
    JR NZ, checkfs_noadd
    INC A
checkfs_noadd:
    PUSH IX
    POP HL
    LD DE, 64
    LD B, A
checkfs_calcloop:
    ADD HL, DE
    DJNZ checkfs_calcloop
    PUSH HL
    POP IX
    POP BC
    LD A, $80
    AND H
    JP Z, checkfs_loop2
    POP AF
    INC A
    CP 20
    JP NZ, checkfs_loop1
    PUSH BC
    LD A, '\n'
    CALL printc
    POP BC
    XOR A
    OR B
    JP NZ, checkfs
    LD A, '\n'
    JP printc   ; Exit

checkfs_disable_popBC:
    POP BC

checkfs_disable:  ; Disable the current block.

    PUSH IX       ; Save IX
    PUSH IX
    POP DE        ; Destination
    LD HL, Zero   ; Disable byte
    PUSH BC
    CALL writebyte
    POP BC
    POP IX        ; Restore IX
    JP checkfs_incb

checkfs_update_spinner:
    POP HL
    POP AF
    PUSH AF
    PUSH HL
    PUSH BC
    PUSH AF

    LD A, '\r'
    CALL printc
    LD HL, checkfs_spinner1
    CALL printstr
    POP AF
    SRL A
    DEC A
    PUSH AF
    DEC A
    JR Z, checkfs_nofirstloop
    LD B, A
    LD A, '='
checkfs_startloop:
    PUSH AF
    PUSH BC
    CALL printc
    POP BC
    POP AF
    DJNZ checkfs_startloop
checkfs_nofirstloop:
    LD A, '>'
    CALL printc
    POP BC
    LD A, 8
    SUB B
    JR Z, checkfs_endspinner
    LD B, A
    LD A, ' '
checkfs_endloop:
    PUSH AF
    PUSH BC
    CALL printc
    POP BC
    POP AF
    DJNZ checkfs_endloop
checkfs_endspinner:
    LD HL, checkfs_spinner2
    CALL printstr
    PUSH IX
    POP HL
    LD A, %00000011
    AND H
    LD HL, checkfs_spinners
    LD E, A
    LD D, 0
    ADD HL, DE
    LD A, (HL)
    CALL printc
    POP BC
    RET

checkfs_format:
    LD HL, checkfserror
    CALL printstr
    LD B, 255
    CALL sleep
    LD HL, firstrun+2
    CALL loadpgm
    LD B, 255
    CALL sleep
    JP BOOT

;** killblk
;* Flag the given block as dead
;* IN HL: The block to 'kill' (in [page][blknr] format).

;* OUT A: Garbage.
;* OUT BC: Garbage.
;* OUT DE: Garbage.
;* OUT HL: $FFFF on failure, $80xx on success.

killblk:
    CALL blk2byte
    LD D, H
    LD E, L
    LD HL, Zero   ; Disable byte
    JP writebyte
