; This is the ROM-code for the flash driver.
; All the .db statement on the bottom are from the
; flash-driver itself that should reside in RAM.

flash_first_blk:  .equ $0400    ; First block of page 4 (sector 1)
flash_last_blk:   .equ $13ff    ; Last block of page 19 (sector 4)

flash_last_page:  .equ $13

flash_swap:       .equ 24
flash_swapsec:    .equ 6

;** flash_init
;* Called by the OS, initializes the flash-driver.

flash_init:
    ; Hmmm, my assembler seems to be messing this up :-(
    ; LD HL, flash_stop - flash_start

    LD DE, $0000 - flash_start
    LD HL, flash_stop
    ADD HL, DE
    PUSH HL

    CALL save_pgmalloc
    LD (flash_code), HL
    LD DE, flash_ram_begin
    LD A, H
    CP D
    JP NZ, flash_error
    LD A, L
    CP E
    JP NZ, flash_error

    ; Ok, it's save to copy the code...
    LD HL, flash_start
    POP BC
flash_init_loop:
    LD A, (HL)
    LD (DE), A
    INC DE
    INC HL
    DEC BC
    LD A, B
    OR C
    JR NZ, flash_init_loop

    ; This variable is used as switch when writing to memory.
    ; Default = 0, when turned to 1, almost all memory is available for writing,
    ; including the OS (sector 0) and the certificate area (sector 5).
    XOR A
    LD (FlashProtMode), A

    ; Turn on flash_writing.
    LD A, $1c
    OUT (6), A    ; Map restricted section page (1C)
    LD B, 1
    CALL Romlock  ; Actual call.
    RET

flash_error:
    LD HL, flash_error_msg
    CALL printstr
    JP halt

flash_error_msg:
 .db "ERROR ALLOCATING FLASH-DRIVER MEMORY: ADDRESSES DON'T MATCH", 0

;** writebyte
;* Writes a single byte some place in the FLASH.
;* IN DE: Destination address (a 'normal' address between $4000 and $7FFF).
;* IN HL: Pointer to the byte to write. This byte <b>must</b> reside in the RAM.
;* OUT A: Garbage
;* OUT BC: Garbage
;* OUT DE: Initial value + 1
;* OUT HL: $FFFF on failure, Initial value + 1 on success.

writebyte:
    LD B, 1   ; Bytes to write
    IN A, (6)
    LD C, A
    JP writeblk_loop


;** seekempty
;* Searches for an empty place in the flash.
;* IN A: (optional, zero to disable) sector to skip, normally sectors 1-4 will be searched.
;* IN B: Amount of needed blocks (1 block = 64 bytes).
;* OUT A: Garbage
;* OUT BC: Garbage
;* OUT DE: Garbage
;* OUT HL: $FFFF on failure, otherwise the first block of your empty space.

seekempty:
    PUSH AF
    PUSH BC
    LD HL, flash_first_blk

seekempty_restart:
    POP BC
    POP DE
    PUSH DE       ; D = sector we don't want to be on.
    PUSH BC       ; Save B (Needed blocks).

    ; Check if we may use this sector.
    ; INFO: This check is enough since _restart wel be called when a page-change is detected
    ;       (since files may not be stored across multiple pages).

    LD A, H
    SRL A
    SRL A
    CP D
    JR NZ, seekempty_next
    INC H
    INC H
    INC H
    INC H
    LD L, 0
seekempty_next:
    LD A, H       ; Page to write on
    CP flash_last_page + 1
    JR Z, seekempty_exit
    OUT (6), A    ; Set page.
    PUSH HL       ; Save block (Stack: [ Needed blocks | Garbage ] [ Current block ]
    CALL blk2byte
    LD A, (HL)
    INC HL
    LD C, (HL)
    AND C         ; A = $FF if block is free (Block is in use, length = 0)
    POP HL        ; HL = current block.
    INC HL        ; HL = next block.
    INC A         : A = 0, if we can use this block.
    JR NZ, seekempty_restart
    DJNZ seekempty_next

    ; HL = last block of the space we are going to use (+1).

    POP BC        ; B = needed blocks.
    PUSH BC       ; Save it again...
    LD A, L
    CP B          ; if A < B the other empty blocks are not on this flash page.
    JR NC, seekempty_norestart
    LD L, 0
    JR seekempty_restart

seekempty_norestart:
    POP BC
    SUB B
    LD L, A   ; HL is now the start of the free space.

    ; Ok we found the block needed.
    ; Reset the stack and return.

    POP AF
    RET

seekempty_exit:
    POP HL
    POP HL
    LD HL, $ffff
    RET

;** blk2byte
;* IN HL: Block-address
;* OUT A: Garbage
;* OUT HL: Pointer to the first byte of that block.
;* This function takes a flash block-address (in [rom-page][blocknr] format),
;* switches to the corresponding flash-page and returns a pointer to the first byte of that block.

blk2byte:
    LD A, H
    OUT (6), A
    LD H, 0
    ADD HL, HL  ;  2
    ADD HL, HL  ;  4
    ADD HL, HL  ;  8
    ADD HL, HL  ; 16
    ADD HL, HL  ; 32
    ADD HL, HL  ; 64
    LD A, $40
    ADD A, H
    LD H, A
    RET

;** copytoram
;* Copies data from the flash to the RAM.
;* IN BC: Amount of bytes to copy.
;* IN HL: Pointer to the source ($4000 to $7FFF).
;* OUT A: Garbage
;* OUT BC: Garbage
;* OUT DE: Garbage
;* OUT HL: $FFFF on failure, otherwise a pointer to the start of your data in the RAM.

copytoram:
    PUSH HL
    PUSH BC
    PUSH BC
    POP HL
    CALL malloc
    LD A, H
    OR L
    JR Z, copytoram_fail
    LD D, H
    LD E, L
    PUSH HL
    POP IX
    POP BC
    POP HL
    LDIR
    PUSH IX
    POP HL
    RET

copytoram_fail:
    POP HL
    POP HL
    LD HL, $ffff
    RET

;** readblk
;* Copies one block (64 bytes) from the flash to a specified place in the RAM.
;* IN DE: Pointer to 64 previous allocated bytes in the RAM.
;* IN HL: Block to copy (in [page][blknr] format, see OS Documentation page).
;* OUT A: Garbage
;* OUT BC: Always $0000
;* OUT DE: End of data in RAM.
;* OUT HL: End of data in FLASH.

readblk:
    PUSH DE
    CALL blk2byte
    POP DE

;** copyblk
;* Copies one block (64 bytes) from the flash to a specified place in the RAM.
;* IN DE: Pointer to 64 previous allocated bytes in the RAM.
;* IN HL: Pointer to the first byte ot the FLASH block to copy ($4000 to $7fc0).
;* OUT A: Garbage
;* OUT BC: Always $0000
;* OUT DE: End of data in RAM.
;* OUT HL: End of data in FLASH.

copyblk:
    PUSH DE
    LD BC, 64
    LDIR
    POP HL
    RET

;** mvrom
;* Copies a file (only phisically) from one place in the FLASH to another.
;* Deletes ("kills") the source afterward.
;* IN DE: First block of some free space in the flash (the destination).
;* IN HL: First block of the source.
;* OUT A: Garbage
;* OUT BC: Garbage
;* OUT DE: Garbage
;* OUT HL: $FFFF on Failure, something else on success.

mvrom:
    PUSH HL
    CALL rom2rom
    LD A, H
    OR L
    JR Z, mvrom_delete
    POP DE
    RET

mvrom_delete:
    POP HL
    CALL blk2byte
    LD D, H
    LD E, L
    LD HL, zero
    CALL writebyte
    RET             ; Pass error or success to the caller.


;** rom2rom
;* Copies a file (only phisically) from one place in the FLASH to another.
;* IN DE: First block of some free space in the flash (the destination).
;* IN HL: First block of the source.
;* OUT A: Garbage
;* OUT BC: Garbage
;* OUT DE: Garbage
;* OUT HL: $FFFF on Failure, $0000 on success.

rom2rom:    ; Copy a file from one piece of flash to another.
            ; HL is the first source block ([page][block]).
            ; DE is the first destination block ([page][block]).

    PUSH DE
    PUSH HL
    LD HL, 64
    CALL malloc
    LD A, H
    OR L
    JP Z, rom2rom_error_pop2
    PUSH HL
    POP IX
    POP HL
    PUSH HL
    CALL blk2byte
    INC HL
    LD A, (HL)
    CPL
    LD B, A

rom2rom_loop:       ; IX = memptr, B = blocks to copy, Stack = srcptr, Stack + 2 = destptr
    POP HL
    PUSH HL
    PUSH IX
    POP DE
    PUSH BC
    CALL readblk
    POP BC
    POP HL
    POP DE
    PUSH DE
    PUSH HL
    PUSH IX
    POP HL
    PUSH BC         ; IX = memptr = HL, Stack = blocks to copy, Stack + 2 = srcptr, Stack + 4 = destptr = DE
    CALL writeblk
    LD A, H
    AND L
    CP $FF
    JR Z, rom2rom_error_pop3
    POP BC
    POP HL
    POP DE
    INC HL
    INC DE
    PUSH DE
    PUSH HL
    DJNZ rom2rom_loop
    POP HL
    POP HL
    PUSH IX
    POP HL
    CALL free
    LD HL, $0000
    RET

rom2rom_error_pop3:
    PUSH IX
    POP HL
    CALL free
    POP HL
rom2rom_error_pop2:
    POP HL
    POP HL
rom2rom_error:
    LD HL, $ffff
    RET

;** backuprom
;* Used internally for sector cleaning (removing "dead" blocks).
;* Copies (phisically) a file to another sector or to the swap sector (sector 6).
;* Deletes ("kills") the source afterward.
;* WARNING: You should have cleaned the swap-page before calling this function.
;* IN HL: Block to move (in normal Flash-driver format, see also the OS Documentation page).
;* OUT A: Garbage
;* OUT BC: Garbage
;* OUT DE: Garbage
;* OUT HL: $FFFF on Failure, $0000 on success.

backuprom:
    PUSH HL
    CALL blk2byte
    INC HL
    LD A, (HL)
    CPL
    LD B, A         ; Needed blocks
    POP HL
    LD A, H
    SRL A
    SRL A           ; Sector to skip.
    PUSH HL
    PUSH BC
    CALL seekempty
    POP BC
    LD A, H
    AND L
    CP $FF
    CALL Z, backuprom_getswapspace
    EX DE, HL
    POP HL
    JP mvrom

backuprom_getswapspace:   ; B = needed blocks.
    LD HL, $8000
    LD DE, $ffc0
backuprom_loop:
    ADD HL, DE
    DJNZ backuprom_loop
    PUSH HL

    LD A, flash_swap
backuprom_mainloop:
    OUT (6), A
    PUSH AF           ; Stack [ page : ... ] [ last block ]
    LD HL, $4000
backuprom_memloop:
    PUSH HL
    POP IY
    LD A, (IY+0)
    AND (IY+1)
    CP $FF
    JR Z, backuprom_toblock
    LD A, (IY+1)
    CPL
    LD DE, 64
    LD B, A
backuprom_addloop:
    ADD HL, DE
    DJNZ backuprom_addloop
    POP AF
    POP DE
    PUSH DE
    PUSH AF
    LD A, D
    CP H
    JR C, backuprom_nextpage
    JR NZ, backuprom_memloop
    LD A, E
    CP L
    JR NC, backuprom_memloop
backuprom_nextpage:
    POP AF
    INC A
    JR backuprom_mainloop   ; No "overflow" can occur, since a sector fits in a sector.
                            ; (At least if I didn't made a mistake ;-)

backuprom_toblock:
    POP AF
    POP AF
    CALL byte2blk
    RET

;** cleansec
;* Cleans a single sector from "dead" blocks, to get some free space.
;* IN A: Sector to clean.
;* OUT A: Garbage
;* OUT BC: Garbage
;* OUT DE: Garbage
;* OUT HL: $FFFF on Failure, $0000 on success.

cleansec:
    CP 5
    JP NC, cleansec_highpage
    PUSH AF
    CALL clearswap
    LD A, H
    CP $FF
    JP Z, cleansec_exit1
    POP AF
    ADD A, A
    ADD A, A  ; A = first of 4 pages.
    PUSH AF
cleansec_outloop:
    OUT (6), A
    PUSH AF
    LD HL, $4000
cleansec_loop:
    LD A, (HL)
    OR A
    JR Z, cleansec_next
    INC HL
    LD A, (HL)
    DEC HL
    CP $FF
    JR Z, cleansec_next
    PUSH HL
    CALL byte2blk
    CALL backuprom
    LD A, H
    CP $FF
    JR Z, cleansec_fail
    POP HL

cleansec_next:
    CALL nextblk
    LD A, H
    CP $FF
    JR NZ, cleansec_loop
    POP AF
    INC A
    PUSH AF
    AND %00000011
    JR Z, cleansec_exit
    POP AF
    JR cleansec_outloop
cleansec_exit:
    POP AF
    POP AF
    PUSH AF
    SRL A
    SRL A
    LD B, A           ; Remember, ClrSec takes B as argument!!
    CALL clrsec
    LD A, H
    CP $FF
    JP Z, cleansec_exit1
    POP AF            ; Page 1/4 to write back to.
    LD B, flash_swap  ; Page 1/4 to write back from.
cleansec_wrtbck:
    PUSH AF
    PUSH BC
    CALL page2page
    LD A, H
    CP $FF
    JR Z, cleansec_wrtbck_fail
    POP BC
    POP AF
    INC A
    PUSH AF
    AND %00000011
    JR Z, cleansec_end
    POP AF
    INC B
    JR cleansec_wrtbck

cleansec_end:
    POP HL
    LD HL, $0000
    RET

cleansec_fail:
    POP HL
cleansec_wrtbck_fail:
    POP HL
cleansec_exit1:
    POP HL
    LD HL, $FFFF
    RET

cleansec_highpage:
    LD HL, $0000    ; Or should we use $FFFF??
    RET

;** byte2blk
;* Calculates the Flash-driver block number from a exact address (and the current flash page).
;* IN HL: Logical address ($4000 to $7FFF).
;* OUT HL: Block number

byte2blk:
    PUSH AF
    SLA L
    RL H
    SLA L
    RL H
    LD L, H
    IN A, (6)
    LD H, A
    POP AF
    RET

;** nextblk
;* Used internally to find the location of the next "file-header" on the current flash page.
;* IN HL: Pointer to the current file header ($4000 to $7fc0)
;* OUT A: Garbage
;* OUT BC: Garbage
;* OUT DE: Garbage
;* OUT HL: Logical address of next file header on this page ($4000 to $7fc0) or $FFFF if no header could be found on the current page.

nextblk:  ; Find the next "file-header" on the current page (or return $FFFF).
    PUSH AF
    PUSH DE
    LD DE, 64
    INC HL
    LD A, (HL)
    DEC HL
    CPL
    OR A
    JR NZ, nextblk_next
    INC A
nextblk_next:
    ADD HL, DE
    DEC A
    JR NZ, nextblk_next
    LD A, H
    AND $80
    JR Z, nextblk_ok
    LD HL, $ffff
nextblk_ok:
    POP DE
    POP AF
    RET

;** page2page
;* Copies a full page to another.
;* Stops copying when an empty block is found.
;* Used by the <code>clrsec</code> routine.
;* IN A: Destination page.
;* IN B: Source page.
;* OUT A: Garbage.
;* OUT BC: Garbage.
;* OUT DE: Garbage.
;* OUT HL: Garbage.

page2page:
    LD D, A
    LD H, B
    LD E, 0
    LD L, 0
page2page_loop:
    PUSH HL
    PUSH DE
    CALL blk2byte
    LD A, (HL)
    CP $FF
    JR NZ, page2page_next
    INC HL
    LD A, (HL)
    CP $FF
    JR Z, page2page_foundend
    POP DE
    POP HL
    PUSH HL
    PUSH DE
    CALL rom2rom
    LD A, H
    CP $FF
    JR Z, page2page_fail
page2page_next:
    POP DE
    POP HL
    INC DE
    INC HL
    XOR A
    OR L
    JR Z, page2page_end
    JR page2page_loop

page2page_fail:
    POP HL
    POP HL
    LD HL, $FFFF
    RET

page2page_foundend:
    POP HL
    POP HL
page2page_end:
    LD HL, $0000
    RET

;** clearswap
;* Checks if there is any data in the swap-sector.
;* If that is the case, the swap space will be wiped out.
;* OUT HL: $0000 on success, $FFFF on failure,

clearswap:
    PUSH AF
    PUSH BC
    PUSH DE
    LD A, (6)
    PUSH AF
    LD A, flash_swap
    LD DE, 64
clearswap_outer_loop:
    OUT (6), A
    PUSH AF
    LD HL, $4000
clearswap_loop:
    LD A, (HL)
    CP $FF
    JR NZ, clearswap_clean
    ADD HL, DE
    LD A, H
    CP $80
    JR NZ, clearswap_loop
    POP AF
    INC A
    CP flash_swap + 4
    JR NZ, clearswap_outer_loop
clearswap_end:
    LD HL, $0000
clearswap_exit:
    POP AF
    OUT (6), A
    POP DE
    POP BC
    POP AF
    RET

clearswap_clean:
    POP AF
    LD B, flash_swapsec
    CALL clrsec
    LD A, H
    CP $FF
    JR NZ, clearswap_end
    JR clearswap_exit

;** falloc
;* Searches for an empty place in the flash.
;* A wrapper around <code>seekempty</code> and <code>trycleansec</code>, to provide automatic sector cleaning.
;* IN B: Amount of needed blocks (1 block = 64 bytes).
;* OUT A: Garbage
;* OUT BC: Garbage
;* OUT DE: Garbage
;* OUT HL: $FFFF on failure, otherwise the first block of your empty space.

falloc:
    XOR A
    PUSH BC
    CALL seekempty
    POP BC
    LD A, H
    AND L
    CP $FF
    RET NZ
    PUSH BC
    CALL trycleansec
    POP BC
    LD A, H
    AND L
    CP $FF
    RET Z
    JR falloc

;** trycleansec
;* Searches for sectors with dead blocks and tries to clean them.
;* OUT A: Garbage
;* OUT BC: Garbage
;* OUT DE: Garbage
;* OUT HL: $FFFF on Failure (no dead blocks or flash-driver error), $0000 on success.

trycleansec:
    ; We have 4 sectors (1 to 4) that can have dead blocks.
    ; Try to find the one with the most dead blocks and clean that one.
    ; If there are none with deadblocks, return $FFFF.

    LD A, 1
    CALL countdeadblks
    LD A, 1
    LD B, 1
trycleansec_loop:
    PUSH HL
    INC A
    CP 5
    JR Z, trycleansec_cont
    LD C, A
    PUSH BC
    CALL countdeadblks
    POP BC
    POP DE
    
    LD A, D
    CP H
    JR C, trycleansec_H
    JR NZ, trycleansec_L
    LD A, E
    CP L
    JR C, trycleansec_H
trycleansec_L:
    LD A, C
    LD H, D
    LD L, E
    JR trycleansec_loop

trycleansec_H:
    LD A, C
    LD B, A
    JR trycleansec_loop

trycleansec_cont:
    POP HL
    LD A, H
    OR L
    JR Z, trycleansec_fail
    LD A, B
    JP cleansec  ; Return value will be passed on.

trycleansec_fail:
    LD HL, $FFFF
    RET

;** countdeadblks
;* Counts the number of dead blocks in the given sector.
;* Used by <code>trycleansec</code>.
;* IN A: Sector to scan.
;* OUT A: Always zero.
;* OUT BC: Number of dead blocks.
;* OUT DE: Garbage.
;* OUT HL: Number of dead blocks.

countdeadblks:
    ADD A, A
    ADD A, A
    LD BC, 0
countdeadblks_loop:
    OUT (6), A
    PUSH AF
    LD HL, $4000
countdeadblks_loop2:
    LD A, (HL)
    OR A
    JR NZ, countdeadblks_next
    INC HL
    LD A, (HL)
    DEC HL
    CPL
    OR A
    JR NZ, countdeadblks_add
    INC A
countdeadblks_add:
    INC BC
    DEC A
    JR NZ, countdeadblks_add
countdeadblks_next:
    PUSH BC
    CALL nextblk
    POP BC
    LD A, H
    CP $FF
    JR NZ, countdeadblks_loop2
    POP AF
    INC A
    PUSH AF
    AND %00000011
    JR Z, countdeadblks_exit
    POP AF
    JR countdeadblks_loop

countdeadblks_exit:
    POP AF
    PUSH BC
    POP HL
    RET

flash_start:

#include "flash.h"

flash_stop:

