;*** base.asm

#include "config.h"

#include "startup.h"

    jp INIT

;Some driver routines:

#include "drivers.h"

; Some functions:

#include "lib.h"

; Some data

#include "data.h"

; Memory locations:

; Everything in vars.h is automatically written to memlocs.h
;#include "vars.h"
#include "memlocs.h"

; Different parts of the OS functionality:

#include "os-parts.h"

; All stuff needed to get something running:

#include "userspc/userspc.h"

; All hardware is ready to work, entering startup phase 1.


;** INIT
;* This function is started from the function BOOT and will initialize all OS Drivers and after this start the program INIT,
;* which should start everything that runs in "user-space" (some services and a login prompt).
INIT:
    LD A, 1
    LD (os_stage), A
    LD DE, $0000          ; Some nice screen :p
    CALL locate           ; Locate up-left
    XOR A
    LD (curscrn), A       ; Screen is not yet buffered.
    LD (multitasking), A  ; We're not yet multitasking
    LD (screen_lock), A   ; Since the screen is not yet initialised,
    Ld (screen_master), A ; and cls wants to lock it.
    CALL cls              ; Clear the screen.
    LD HL, MesWelcome     ; Send out a message.
    CALL puts

    CALL mem_init     ; Init memory.
    CALL proc_init    ; Init Process Manager.
    CALL timer_init   ; Timer Handler.
    CALL screen_init  ; Screen Manager

    ; Core drivers loaded and initialized, entering startup phase 2.

    LD A, 2
    LD (os_stage), A
    CALL cls          ; Clear the screen by hardware.
    CALL clrscrn      ; Clear the screen buffer
                      ; from now on we will only use the screen manager

    CALL flash_init   ; Init the flash driver.

    ; Because the library manager isn't ready, we should have an option
    ; to clear the flash (and reformat it).

    CALL checkclear

    ; Check the filesystem for errors and start the init process.

    LD HL, checkfs_mes
    CALL printstr
    CALL checkfs

    ; Check again, for slow people...
    CALL checkclear

    ; Start the init process
    CALL startinit

    ; Done...
    JP halt

;** checkclear
;* This function is called during booting. It checks if the CLEAR key is pressed. If this is the case, it will call sysclrflash
;* and clears the flash.

checkclear:
    CALL sysinkey
    CP AscBackspc
    JP Z, sysclrflash
    RET

;** boot_timeout
;* A simple timeout-loop for debugging the booting of BAOS.
;* OUT B: Always 0
boot_timeout:
    LD B, $80
boot_timeout_loop:
    HALT
    DJNZ boot_timeout_loop
    RET

;** halt
;* Halts forever, only usefull to stop the OS (which can only be done when the OS itself calls halt), when something critical happends.
halt:
    HALT
    JR halt

;** shutdown
;* This function will power down the calculator. Called by the OS when ON and 2ND keys are pressed at the same time.

shutdown:   ; Currently we only do a real deep-shutdown (not "sleep-like").
            ; System restart will take place after it.

    LD A, %00000001 ; Only wake up on ON-key interrupt, and sleep deep (turn off hardware).
    OUT (3), A

    CALL quickbusywaitforLCD
    LD A, $02   ; Turn off LCD-screen (not really needed, but also for a reaction on powering off).
    OUT ($10), A

shutdown_loop:	 ; Wait until we release the on key (in case someone shuts down wrong way round).
    IN A, (4)
    AND 8
    JR Z, shutdown_loop

    EI			; !!! Very important !!! Make sure interrupts are on, or we will never wake up again ;p
    HALT
    RST $00	; Port 3 will be restored at bootup.




;** standby
;* This function will turn off the calculator, but when the ON-key is pressed, the calculator continous execution.
;* OUT A: Always $0B

standby:
    DI
    LD A, %00000001   ; Only wake up on ON-key interrupt, and sleep deep (turn off hardware).
    OUT (3), A

    CALL quickbusywaitforLCD
    LD A, $02         ; Turn off LCD-screen (not really needed, but also for a reaction on powering off).
    OUT ($10), A

    EI
    HALT              ; Sleep

    DI
    CALL quickbusywaitforLCD
    LD A, $03         ; Turn on LCD-screen
    OUT ($10), A
    CALL quickbusywaitforLCD  ; In case someone was writing...

    IM 1              ; Interrupt Mode 1, Normal System Interrupts.
    LD A, %00001011   ; Enable Timer 1 + ON-key interrupts (+ HALT means low-power state).
    OUT (3), A

    EI                ; Enable interrupts, and...

    RET               ; Resume executing...



;** sysclrflash
;* Will wipeout the filesystem and reboot the calculator. Handy when the filesystem is corrupted and booting fails.
sysclrflash:
    CALL clrscrn
    LD HL, clrflashmsg
    CALL printstr
    CALL waitnokey
    LD B, 1
    CALL ClrSec
    LD B, 2
    CALL ClrSec
    LD B, 3
    CALL ClrSec
    LD B, 4
    CALL ClrSec
    LD HL, clrflashmsg2
    CALL printstr
    CALL getkey
    RST $00

.org $3fff
    RST $38
.end
