;===============================================================;
;                                                               ;
; Alien Breed 5                                                 ;
; Shooting routines                                             ;
;                                                               ;
;===============================================================;

;------------------------------------------------
; clearBulletTable - clear all bulletTable entries
;
; input:    none
; output:   none
;------------------------------------------------
clearBulletTable:
        ld      hl,bulletTable                  ; first byte to clear
        ld      de,bulletTable+1                ; first byte to copy cleared bytes to
        ld      bc,NUM_BULLETS*BULLET_SIZE-1    ; number of bytes to clear
        ld      (hl),0                          ; clear first byte
        ldir                                    ; clear all subsequent bytes
        ret

;------------------------------------------------
; playerShoot - try to create a new player bullet
;
; input:    none
; output:   none
;------------------------------------------------
playerShoot:
        ld      hl,frame
        bit     0,(hl)
        ret     nz                              ; player can only shoot every 2nd frame
        ld      hl,(ammo)                       ; HL => player ammo
        ld      a,l
        or      h
        ret     z                               ; can't shoot if no ammo
; check bulletTable for an empty entry
        ld      ix,bulletTable                  ; IX => start of table
        ld      b,NUM_BULLETS                   ; B = num entries to check
        ld      de,BULLET_SIZE                  ; DE = size of each entry
findEmptyBulletEntry:
        ld      a,(ix)                          ; check type
        or      a                               ; is it empty?
        jr      z,foundEmptyBulletEntry         ; if so, found one
        add     ix,de                           ; IX => next entry
        djnz    findEmptyBulletEntry
        ret                                     ; if no entries found, can't shoot
foundEmptyBulletEntry:
        dec     hl
        dec     hl                              ; decrement ammo (HL still holds the value from earlier)
        bit     7,h                             ; check if ammo went below 0?
        jr      z,saveAmmo                      ; if not, go straight to saving it
        ld      hl,0                            ; otherwise set it to 0
saveAmmo:
        ld      (ammo),hl                       ; save it
        ld      hl,playerDir                    ; HL => player direction
        ld      a,(weapon)                      ; A = weapon type
        ld      (ix+0),a                        ; store it
        dec     a                               ; weapon type will now be between 0-4
        add     a,a
        add     a,a
        add     a,a
        add     a,a
        ld      d,a                             ; D = offset to type in bulletOffsetTable
        ld      a,(hl)                          ; A = player direction
        ld      (ix+1),a                        ; save it as bullet direction
        add     a,a
        add     a,a                             ; A = direction offset for bulletOffsetTable
        add     a,d                             ; A = offset to coords in bulletOffsetTable
        ld      l,a
        ld      h,0
        ld      de,bulletOffsetTable
        add     hl,de                           ; HL => coords to use in bulletOffsetTable
        ld      e,(hl)
        inc     hl
        ld      d,(hl)                          ; DE = x coord offset
        inc     hl
        push    hl                              ; save bulletOffsetTable ptr
        ld      hl,(playerX)
        add     hl,de                           ; HL = bullet x coord
        ld      (ix+2),l                        ; save x coord LSB
        ld      (ix+3),h                        ; save x coord MSB
        pop     hl                              ; restore bulletOffsetTable ptr
        ld      e,(hl)
        inc     hl
        ld      d,(hl)                          ; DE = y coord offset
        ld      hl,(playerY)
        add     hl,de                           ; HL = bullet y coord
        ld      (ix+4),l                        ; save y coord LSB
        ld      (ix+5),h                        ; save y coord MSB
        ld      (ix+6),0                        ; reset animation counter for flamethrower
; if machine gun is the selected weapon, we need to set the blast effect flag in drawBullets
        ld      a,(weapon)
        dec     a                               ; weapon 1 (machine gun)?
        ret     nz                              ; if not, don't set flag
        ld      a,$37                           ; A = "scf" instruction
        ld      (__dbBlast),a                   ; set the code flag
        ret

;------------------------------------------------
; drawBullets - draw player bullets
;
; input:    none
; output:   none
;------------------------------------------------
drawBullets:
#ifndef APP83P
        nop                                     ; will be set to either "or a" or "scf" depending on whether not player shot the machine gun this frame
__dbBlast               = $-1
#else
        call    __dbBlast
#endif
        jr      nc,afterGunBlast                ; if machine gun not shot this frame, don't show the blast effect
        ld      a,(playerDir)
        add     a,a
        add     a,a
        ld      e,a
        ld      d,0                             ; DE = offset to data in bulletOffsetTable
        add     a,a
        push    af                              ; save playerDir * 8 for later
        ld      hl,bulletOffsetTable
        add     hl,de                           ; HL => required data in bulletOffsetTable
        ex      de,hl                           ; DE => required data in bulletOffsetTable
        ld      hl,(playerX)
        ld      bc,(screenX)
        or      a
        sbc     hl,bc                           ; HL = player x on screen
        ex      de,hl                           ; DE = player x on screen; HL => required data in bulletOffsetTable
        push    hl                              ; save ptr to bulletOffsetTable
        bcall(_ldhlind)                         ; HL = x offset
        add     hl,de                           ; L = x coord to display blast effect at
        ld      b,l                             ; B = x coord to display at
        pop     de                              ; restore ptr to bulletOffsetTable
        push    bc                              ; save x coord to display at
        inc     de
        inc     de                              ; DE => y data
        ld      hl,(playerY)
        ld      bc,(screenY)
        or      a
        sbc     hl,bc                           ; HL = player y on screen
        ex      de,hl                           ; DE = player y on screen; HL => required data in bulletOffsetTable
        bcall(_ldhlind)                         ; HL = y offset
        add     hl,de
        pop     bc                              ; B = x coord to display at
        ld      c,l                             ; C = y coord to display at
        pop     af                              ; A = playerDir * 8 (from earlier)
        ld      e,a
        ld      d,0                             ; DE => offset to sprite data
        ld      hl,sprMachineGun
        add     hl,de                           ; HL => sprite to display
        call    putClippedSprite                ; draw blast effect
afterGunBlast:
        ld      ix,bulletTable                  ; IX => start of data
        ld      b,NUM_BULLETS                   ; B = number of bullets to draw
drawBulletsLoop:
        push    bc                              ; save counter
        ld      a,(ix+0)                        ; A = type
        or      a                               ; if 0, it's an empty entry
        jr      z,endDrawBulletsLoop
        dec     a                               ; A = type {0-4}
        jr      z,endDrawBulletsLoop            ; if type = 0, no sprite to draw
        add     a,a
        ld      e,a
        ld      d,0
        ld      hl,bulletSpriteTable
        add     hl,de                           ; HL => address of sprites for this bullet type
        bcall(_ldhlind)                         ; HL => sprites for this bullet type
        ld      a,(ix+1)                        ; A = bullet direction
        add     a,a
        add     a,a
        add     a,a
        ld      e,a
        ld      d,0                             ; DE = direction offse to sprite
        add     hl,de                           ; HL => sprites for this bullet & direction
        push    hl                              ; save sprite ptr
        ld      de,(screenX)
        ld      l,(ix+2)
        ld      h,(ix+3)                        ; HL = bullet x
        or      a
        sbc     hl,de                           ; HL = x coord to display at
        ld      b,l                             ; B = x coord to display at
        ld      de,(screenY)
        ld      l,(ix+4)
        ld      h,(ix+5)                        ; HL = bullet y
        or      a
        sbc     hl,de                           ; HL = y coord to display at
        ld      c,l                             ; C = y coord to display at
        pop     hl                              ; HL => sprite
        ld      a,(ix+0)                        ; A = bullet type
        cp      5                               ; is it a flamethrower bullet?
        jr      nz,drawBulletDraw               ; if not, no further sprite calcs
        ld      a,(ix+6)                        ; A = animation counter
        add     a,a
        add     a,a
        add     a,a
        add     a,a
        add     a,a                             ; A = animation counter * 32
        ld      e,a
        ld      d,0
        add     hl,de                           ; HL => sprite
drawBulletDraw:
        push    ix                              ; save bulletTable ptr
        call    putClippedSprite                ; display bullet
        pop     ix                              ; restore bulletTable ptr
endDrawBulletsLoop:
        pop     bc                              ; restore counter
        ld      de,BULLET_SIZE
        add     ix,de                           ; IX => next entry
        djnz    drawBulletsLoop
        ret

;------------------------------------------------
; moveBullets - move player bullets
;
; input:    none
; output:   none
;------------------------------------------------
moveBullets:
        ld      ix,bulletTable                  ; IX => start of data
        ld      b,NUM_BULLETS                   ; B = number of bullets to move
moveBulletsLoop:
        push    bc                              ; save counter
        ld      a,(ix+0)                        ; A = type
        or      a                               ; if 0, it's an empty entry
        jr      z,endMoveBulletsLoop
        cp      5                               ; is it a flamethrower bullet?
        jr      nz,moveBulletMove               ; if not, skip animation update
        ld      a,(frame)
        and     %00000111                       ; only update animation every 8th frame
        jr      nz,moveBulletMove
        ld      a,(ix+6)                        ; A = animation counter
        inc     a                               ; increment it
        and     %00000011                       ; can only be 0-3
        ld      (ix+6),a                        ; save it
moveBulletMove:
        ld      l,(ix+1)
        ld      h,0                             ; HL = direction
        add     hl,hl
        ld      de,bulletMoveTable
        add     hl,de
        bcall(_ldhlind)                         ; HL => routine to move bullet
        ld      de,endMoveBulletsLoop           ; DE => where to ret to after moving bullet
        push    de                              ; push ret address to stack
        jp      (hl)                            ; jump to moving routine
endMoveBulletsLoop:
        pop     bc                              ; restore counter
        ld      de,BULLET_SIZE
        add     ix,de                           ; IX => next entry
        djnz    moveBulletsLoop
        ret

moveBulletUp:
        ld      e,(ix+4)
        ld      d,(ix+5)                        ; DE = bullet y
        dec     de
        dec     de                              ; decrement * 2
        bit     7,d                             ; check if it went below 0
        jr      nz,removeBullet                 ; if so, it's off the level, so remove the bullet
        ld      (ix+4),e
        ld      (ix+5),d                        ; save new bullet y
        jr      moveBulletCheck
moveBulletDown:
        ld      e,(ix+4)
        ld      d,(ix+5)                        ; DE = bullet y
        inc     de
        inc     de
        ld      hl,(levelHeight)
        ld      h,0
        add     hl,hl
        add     hl,hl
        add     hl,hl                           ; HL = level height in pixels
        bcall(_cphlde)                          ; check if bullet has gone off bottom edge of level
        jr      c,removeBullet                  ; if so, remove the bullet
        ld      (ix+4),e
        ld      (ix+5),d                        ; save new bullet y
        jr      moveBulletCheck
moveBulletLeft:
        ld      e,(ix+2)
        ld      d,(ix+3)                        ; DE = bullet x
        dec     de
        dec     de
        bit     7,d                             ; check if it went under 0
        jr      nz,removeBullet                 ; if so, it's off the level, so remove the bullet
        ld      (ix+2),e
        ld      (ix+3),d                        ; save new bullet x
        jr      moveBulletCheck
moveBulletRight:
        ld      e,(ix+2)
        ld      d,(ix+3)                        ; DE = bullet x
        inc     de
        inc     de
        ld      hl,(levelWidth)
        ld      h,0
        add     hl,hl
        add     hl,hl
        add     hl,hl                           ; HL = level width in pixels
        bcall(_cphlde)                          ; check if bullet has gone off right edge of level
        jr      c,removeBullet                  ; if so, remove the bullet
        ld      (ix+2),e
        ld      (ix+3),d                        ; save new bullet x
        jr      moveBulletCheck
removeBullet:
        ld      (ix+0),0                        ; disable this entry
        ret                                     ; ret to loop
moveBulletCheck:
        ld      a,(ix+0)                        ; A = bullet type
        dec     a
        add     a,a
        add     a,a
        add     a,a
        add     a,a                             ; A = offset to bullet type in bulletCentreTable
        ld      l,a
        ld      a,(ix+1)                        ; A = bullet direction
        add     a,a
        add     a,a
        add     a,l                             ; A = offset to bullet type & direction in bulletCentreTable
        ld      l,a
        ld      h,0
        ld      de,bulletCentreTable
        add     hl,de                           ; HL => required data in bulletCentreTable
        ld      e,(hl)
        inc     hl
        ld      d,(hl)                          ; DE = x offset to centre of bullet
        inc     hl                              ; HL => y offset to centre of bullet
        push    hl                              ; save ptr
        ld      l,(ix+2)
        ld      h,(ix+3)                        ; HL = bullet x
        add     hl,de                           ; HL = x coord of centre of bullet
        ex      de,hl                           ; DE = x coord of centre of bullet
        pop     hl                              ; HL => y offset to centre of bullet
        ld      c,(hl)
        inc     hl
        ld      b,(hl)                          ; BC = y offset to centre of bullet
        ld      l,(ix+4)
        ld      h,(ix+5)                        ; HL = bullet y
        add     hl,bc                           ; HL = y coord of centre of bullet
        call    getTile                         ; A = tile at bullet centre; HL => tile in level data
        cp      TILE_SOLID                      ; solid tile?
        ret     c                               ; if not, ret to loop without terminating bullet
        sub     TILE_POWERDOME                  ; power dome or door tile?
        push    ix                              ; save bullet data ptr
        call    nc,shootTile                    ; if so, run the tile shooting routine
        pop     ix                              ; restore bullet data ptr
removeBulletCheckAnim:
        ld      a,(ix+0)                        ; A = bullet type
        dec     a                               ; is it 0 (machine gun)?
        jr      nz,removeBullet                 ; if not, remove bullet without the explosion animation
; create an explosion animation
; first randomly pick an offset line from bulletWallAnimTable
        ld      b,96
        call    libRandom                       ; A = 0-95
        and     %01110000                       ; A = 0,16,32,48,64,80
        ld      c,a
        ld      b,0                             ; BC = random line offset for bulletWallAnimTable
        ld      a,(ix+1)                        ; A = bullet direction
        add     a,a
        add     a,a
        ld      l,a
        ld      h,0                             ; HL = direction offset for bulletWallAnimTable
        add     hl,bc                           ; HL = total offset for bulletWallAnimTable
        ld      de,bulletWallAnimTable
        add     hl,de                           ; HL => offsets for animation x,y
        ld      e,(hl)
        inc     hl
        ld      d,(hl)                          ; DE = x offset
        inc     hl                              ; HL => y offset
        push    hl
        ld      l,(ix+2)
        ld      h,(ix+3)                        ; HL = bullet x
        add     hl,de
        ex      de,hl                           ; DE = x coord for explosion animation
        pop     hl                              ; HL => y offset
        ld      c,(hl)
        inc     hl
        ld      b,(hl)                          ; BC = y offset
        ld      l,(ix+4)
        ld      h,(ix+5)                        ; HL = bullet y
        add     hl,bc                           ; HL = y coord for explosion animation
        ld      bc,0*256+4                      ; B = animation type; C = initial counter
        call    newAnim
        jr      removeBullet

;------------------------------------------------
; shootTile - find location of a shootable tile that has been shot and damage/destroy it
;
; input:    A = tile number minus TILE_POWERDOME (the first shootable tile)
;           HL => tile in level data
; output:   none
;------------------------------------------------
shootTile:
        cp      TILE_DOOR-TILE_POWERDOME        ; check if a door tile
        ld      c,HITS_DOOR                     ; C = number of hits required to destroy a door
        jr      nc,doShootTile                  ; if it's a door, we can go straight ahead and process it
; otherwise, it's a power dome, which is 2x2 tiles in size, so first we need to
; calc a ptr to the top-left tile of the power dome
        ld      de,(levelFlags)                 ; E = level flags
        bit     LFLAG_POWERDOMES,e              ; check if power domes can be shot
        ret     z                               ; if not, just leave
        ld      c,HITS_POWERDOME                ; C = number of hits req'd to destroy a power dome
        cp      2                               ; check if it's the 3rd or 4th tile of a power dome
        jr      c,shootPowerDomeYOK             ; if not, HL is pointing to the correct row (y coord) in level data
        ld      de,(levelWidth)
        ld      d,0                             ; DE = level width (in blocks)
        or      a                               ; clear carry
        sbc     hl,de                           ; HL => either tile 1 or 2 of the power dome
shootPowerDomeYOK:
        bit     0,a                             ; check if it's the 2nd or 4th tile of a power dome
        jr      z,doShootTile                   ; if not, HL is pointing to the correct tile
        dec     hl                              ; HL => 1st (top left) tile of power dome
doShootTile:
#ifndef APP83P
        ld      de,$0000                        ; DE => previously shot tile in level data (NULL if none)
__tileShotPtr           = $-2
#else
        ld      de,(__tileShotPtr)
#endif
        bcall(_cphlde)                          ; compare pointers to see if this tile is the same as the last one shot
        jr      z,updateShotTile                ; if so, go straight to update
        ld      a,c                             ; A = number of hits req'd to destroy tile (C was set earlier)
        ld      (__tileShotHits),a              ; set number of hits left (ie. tile health)
updateShotTile:
        ld      (__tileShotPtr),hl              ; save ptr to show tile that has been shot this frame
#ifndef APP83P
        ld      a,$00                           ; tile hits (health) (set earlier)
__tileShotHits          = $-1
#else
        ld      a,(__tileShotHits)
#endif
        dec     a
        ld      (__tileShotHits),a              ; save new hits (health)
        or      a                               ; check if it has reached zero
        ret     nz                              ; if not, simply leave
; otherwise, tile has been "destroyed", and thus needs to be updated in level data to a new tile or new set of tiles
        ld      de,$0000
        ld      (__tileShotPtr),de              ; reset ptr to NULL to indicate that no tile has previously been shot
        ld      a,(hl)                          ; A = tile that has been "destroyed"
        cp      TILE_DOOR                       ; is it a door?
        jp      nc,openDoor                     ; if so, open it and ret
; otherwise, it's a power dome, which is a little more complicated
        call    destroy2PDTiles                 ; destroy 1st & 2nd tile
        ld      de,(levelWidth)
        ld      d,0
        dec     e
        add     hl,de                           ; HL => 3rd tile
        call    destroy2PDTiles                 ; destroy 3rd & 4th tile
        jp      lowerObjectivesCount            ; lower objectives count & ret
destroy2PDTiles:
        ld      a,(hl)                          ; A = power dome tile
        sub     4                               ; A = destroyed power dome tile
        call    putTile                         ; place it & draw it
        inc     hl                              ; HL => next tile
        ld      a,(hl)                          ; repeat
        sub     4
        jp      putTile

.end
