;===============================================================;
;                                                               ;
; Alien Breed 5                                                 ;
; Screen scrolling routines                                     ;
;                                                               ;
;===============================================================;

;------------------------------------------------
; scrollUp - scroll the screen up a pixel
;
; input:    none
; output:   none
;------------------------------------------------
scrollUp:
; first, check that we're not already viewing the bottom of the level
        ld      hl,(screenY)
        ld      de,64
        add     hl,de
        ex      de,hl                           ; DE = y pos of bottom of screen + 1
        ld      hl,(levelHeight)
        ld      h,0                             ; HL = level height in blocks
        add     hl,hl
        add     hl,hl
        add     hl,hl                           ; HL = level height in pixels
        bcall(_cphlde)                          ; compare against bottom of screen
        ret     z                               ; if they equal, we're at the bottom of the level, so don't scroll
        ld      hl,__playerScrY
        dec     (hl)                            ; player will move up 1 pixel on screen due to the scroll
        ld      hl,gbuf+12                      ; copy from 2nd pixel row
        ld      de,gbuf                         ; to top pixel row
        ld      bc,63*12                        ; 63 rows to copy
        ldir
        ld      hl,gbuf+(63*12)                 ; HL => where to draw new pixel row
        ld      a,63                            ; A = pixel row number
        ld      b,$23                           ; B = inc hl opcode
; fall through to scrollVertical

;------------------------------------------------
; scrollVertical - scroll screen vertically
;
; input:    HL => start of either top or bottom row of gbuf
;           A = {0, 63}
;           B = {inc hl, dec hl}
; output:   None
;------------------------------------------------
scrollVertical:
        ld      (__svA),a                       ; set pixel row number in code below
        ld      a,b                             ; A = opcode
        ld      (__svIncDec),a                  ; set opcode in code below
        ld      (__svgbuf),hl                   ; save gbuf pointer in code below
        ld      de,11
        add     hl,de
        ld      (__svgbufEndRow),hl             ; save gbuf pointer to end of row in code below
        ld      hl,(__svgbuf)                   ; HL => start of row again
        ld      d,h
        ld      e,l                             ; DE = HL
        inc     de                              ; DE = HL + 1
        ld      bc,11                           ; 11 bytes to clear
        ld      (hl),0                          ; clear first byte
        ldir                                    ; clear full pixel row
        ld      hl,(screenY)                    ; HL = y coord of top of screen
        ld      a,l
        and     %00000111                       ; check if screenY is on an even block before scrolling
        ld      a,$B7                           ; A = "or a"
        jr      nz,svSetTriggerCheck
        ld      a,$37                           ; A = "scf"
svSetTriggerCheck:
        ld      (__svTriggerCheck),a            ; set instruction later in code
#ifndef APP83P
        nop                                     ; inc or dec depending on which opcode was loaded
__svIncDec              = $-1
#else
        call    __svIncDec
#endif
        ld      (screenY),hl
#ifndef APP83P
        ld      de,$0000
__svA                   = $-2
#else
        ld      de,(__svA)
        ld      d,0
#endif
        add     hl,de                           ; HL = y coord of pixel row to draw from tile data
        ld      (tempY),hl                      ; save it for later
        ld      a,(screenX)                     ; A = LSB of screenX
        and     $07                             ; check if left side of screen is currently on an even block
        jr      nz,sh1                          ; if not, go set up to deal with uneven tile scrolling
        ld      b,12                            ; if on an even block, we have a standard 12 bytes to draw
        xor     a
        ld      (tempX),a                       ; no x offset
        ld      (scrollBits),a                  ; 0 bits to shift
        jr      sh2
sh1:
        ld      b,11                            ; 11 standard bytes to draw
        sub     8
        neg
        ld      (tempX),a                       ; save x offset
        ld      (scrollBits),a                  ; save # bits to shift
sh2:
        ld      a,(tempY)
        and     $07                             ; A = lowest 3 bits of y coord to draw, to determine offset in tile sprite
        ld      l,a
        ld      h,0
        ld      (scrollSkip),hl                 ; save tile sprite offset
        push    bc                              ; save byte counter
        ld      hl,(screenX)                    ; HL = screen x coord
        ld      de,(tempX)
        ld      d,0                             ; DE = x offset
        add     hl,de
        ex      de,hl                           ; DE = x coord
        ld      hl,(tempY)                      ; HL = y coord
        call    getTile                         ; HL => tile in tile data
        pop     bc                              ; restore counter
sh3:
        push    bc                              ; save counter
        call    svCheckEnemyTrigger             ; check tile for enemy trigger
        push    hl                              ; save tile ptr
        call    checkE2Trigger                  ; check tile to display and put it in HL
        add     hl,hl
        add     hl,hl
        add     hl,hl
        ld      de,startTiles
        add     hl,de                           ; HL => tile sprite
        ld      de,(scrollSkip)                 ; DE = offset to correct byte in tile sprite
        add     hl,de                           ; HL => byte to display
        ld      a,(tempX)                       ; A = x offset
#ifndef APP83P
        ld      de,$0000                        ; DE => row in gbuf to display byte (set earlier in code)
__svgbuf                = $-2
#else
        ld      de,(__svgbuf)
#endif
        call    putScrollByte                   ; display tile sprite byte
        pop     hl                              ; restore tile data pointer
        inc     hl                              ; increment it to next tile in tile data
        ld      a,(tempX)
        add     a,8                             ; add 8 to x offset
        ld      (tempX),a                       ; and save it
        pop     bc                              ; restore counter
        djnz    sh3                             ; loop
        ld      a,(scrollBits)                  ; A = # bits to shift
        or      a                               ; was it 0?
        ret     z                               ; if so, we're done
; otherwise, we need to display some extra bits at the end of the row, and then the start of the row
        push    af                              ; save # bits to shift
        call    svCheckEnemyTrigger             ; check tile for enemy trigger
        pop     af                              ; restore # bits to shift
        push    hl                              ; save pointer to tile data
        ld      b,a                             ; B = # bits to shift
        call    checkE2Trigger                  ; check tile to display and put it in HL
        add     hl,hl
        add     hl,hl
        add     hl,hl
        ld      de,startTiles
        add     hl,de                           ; HL => tile sprite
        ld      de,(scrollSkip)
        add     hl,de                           ; HL => byte to display
        ld      a,(hl)                          ; A = byte to display
sh4:
        srl     a                               ; shift right
        djnz    sh4                             ; B number of times
#ifndef APP83P
        ld      hl,$0000                        ; HL => last byte of row on gbuf
__svgbufEndRow          = $-2
#else
        ld      hl,(__svgbufEndRow)
#endif
        or      (hl)                            ; display the bits with OR
        ld      (hl),a                          ; and save it
        pop     hl                              ; HL => pointer to tile data
        ld      de,-12
        add     hl,de                           ; go back to find required tile for the start of the row
        call    svCheckEnemyTrigger             ; check tile for enemy trigger
        ld      a,(scrollBits)                  ; A = # bits to shift
        sub     8                               ; we want the opposite amount for this shift as we're shifting the other way
        neg                                     ; make it a positive value
        ld      b,a                             ; B = # bits to shift
        call    checkE2Trigger                  ; check tile to display and put it in HL
        add     hl,hl
        add     hl,hl
        add     hl,hl
        ld      de,startTiles
        add     hl,de                           ; HL => tile sprite
        ld      de,(scrollSkip)
        add     hl,de                           ; HL => byte to display
        ld      a,(hl)                          ; A = byte to display
sh5:
        add     a,a                             ; shift left
        djnz    sh5                             ; B number of times
        ld      hl,(__svgbuf)                   ; HL => first byte of row on gbuf
        or      (hl)                            ; display the bits with OR
        ld      (hl),a                          ; and save it
        ret

;------------------------------------------------
; scrollDown - scroll the screen down a pixel
;
; input:    none
; output:   none
;------------------------------------------------
scrollDown:
; first, check that we're not already viewing the top of the level
        ld      hl,(screenY)
        ld      a,l
        or      h                               ; is screen y pos 0?
        ret     z                               ; if so, we're at the top of the level, so don't scroll
        ld      hl,__playerScrY
        inc     (hl)                            ; player will move down 1 pixel on screen as a result of the scroll
        ld      hl,gbuf+755                     ; copy from 2nd last pixel row
        ld      de,gbuf+767                     ; to bottom pixel row
        ld      bc,63*12                        ; 63 rows to copy
        lddr
        ld      hl,gbuf                         ; HL => where to draw new pixel row
        xor     a                               ; A = pixel row number
        ld      b,$2B                           ; B = dec hl opcode
        jp      scrollVertical

;------------------------------------------------
; scrollLeft - scroll the screen left a pixel
;
; input:    none
; output:   none
;------------------------------------------------
scrollLeft:
; first, check that we're not already viewing the far right of the level
        ld      hl,(screenX)
        ld      de,96
        add     hl,de
        ex      de,hl                           ; DE = x pos of right of screen + 1
        ld      hl,(levelWidth)
        ld      h,0                             ; HL = level width in blocks
        add     hl,hl
        add     hl,hl
        add     hl,hl                           ; HL = level width in pixels
        bcall(_cphlde)                          ; compare against right of screen
        ret     z                               ; if they equal, we're at the far right of the level, so don't scroll
        ld      hl,__playerScrX
        dec     (hl)                            ; player will move left 1 pixel on screen due to the scroll

        ld      hl,gbuf+(64*12)-1               ; HL => last byte of gbuf
        ld      c,64                            ; C = 64 rows to shift
sl1:
        ld      b,3                             ; B = loop 3 times (4 shifts per loop, equals total of 12 bytes per row)
        or      a                               ; clear carry flag
sl2:
        rl      (hl)                            ; rotate left 4 bytes
        dec     hl
        rl      (hl)
        dec     hl
        rl      (hl)
        dec     hl
        rl      (hl)
        dec     hl
        djnz    sl2                             ; and loop
        dec     c                               ; decrement row counter
        jr      nz,sl1                          ; and loop
; draw tiles at right of screen
        ld      hl,(screenX)
        ld      a,l
        and     %00000111                       ; check if screenX is on an even block before scrolling
        ld      a,$B7                           ; A = "or a"
        jr      nz,slSetTriggerCheck
        ld      a,$37                           ; A = "scf"
slSetTriggerCheck:
        ld      (__slTriggerCheck),a            ; set instruction later in code
        inc     hl
        ld      (screenX),hl                    ; inc screen x coord and save it
        ld      de,95
        add     hl,de                           ; HL = x coord of right side of screen
        ld      (tempX),hl                      ; save it for later
        ld      a,l                             ; A = LSB of x coord of right side of screen
        and     $07                             ; A = x offset
        ld      hl,pixelTable+1
        ld      e,a
        ld      d,0
        add     hl,de
        ld      a,(hl)                          ; A = mask to use
        ld      (scrollMask),a                  ; save it
        ld      de,(tempX)                      ; DE = x coord
        ld      hl,(screenY)                    ; HL = y coord
        call    getTile                         ; A = tile #; HL => tile in tile data
        ld      (scrollPtr),hl                  ; save pointer to tile in tile data
        ld      a,(screenY)                     ; A = LSB of screen y coord
        and     $07
        ld      l,a
        ld      h,0                             ; HL = offset to use for tile sprite
        ld      (scrollSkip),hl                 ; save it
        ld      (scrollBits),a                  ; save # of bytes that will need to be displayed for last tile if screen y coord isn't an exact block
        sub     8
        neg                                     ; A = # rows to display for first tile
        ld      ix,gbuf+11                      ; IX => end of first row of gbuf
        ld      c,a                             ; C = # rows to display for first tile
        ld      b,8                             ; B = 8 tiles to display
sl3:
        ld      hl,(scrollPtr)                  ; HL => tile in tile data
        push    bc                              ; save counters
        push    hl                              ; save tile data ptr
#ifndef APP83P
        nop                                     ; "or a" or "scf" (set earlier in code)
__slTriggerCheck        = $-1
#else
        call    __slTriggerCheck                ; "or a" or "scf"
#endif
        call    c,checkEnemyTrigger             ; only call if screenY was previously on an even block before scrolling
        pop     hl                              ; restore tile data ptr
        pop     bc                              ; restore counters
        call    checkE2Trigger                  ; check tile to display and put it in HL
        add     hl,hl
        add     hl,hl
        add     hl,hl
        ld      de,startTiles
        add     hl,de                           ; HL => tile sprite
        ld      de,(scrollSkip)
        add     hl,de                           ; HL => byte to get pixel from
        ld      de,12                           ; will be used after sl5
sl4:
        ld      a,(scrollMask)                  ; A = pixel mask
        and     (hl)                            ; apply it to tile sprite byte
        jr      z,sl5                           ; if blank pixel, don't turn pixel on in gbuf
        set     0,(ix)                          ; turn pixel on in gbuf
sl5:
        add     ix,de                           ; IX => end of next row of gbuf
        inc     hl                              ; HL => next byte of tile sprite
        dec     c                               ; dec byte counter
        jr      nz,sl4                          ; loop
        ld      hl,0
        ld      (scrollSkip),hl                 ; start at first byte of tile sprite next time
        ld      hl,(scrollPtr)                  ; HL => tile in tile data
        ld      de,(levelWidth)
        ld      d,0                             ; DE = level width
        add     hl,de                           ; HL => next tile to display in tile data
        ld      (scrollPtr),hl                  ; save pointer
        ld      c,8                             ; C = 8 bytes to display for next sprite
        djnz    sl3                             ; loop
        ld      a,(scrollBits)                  ; A = # bytes that need to be displayed for last 9th tile
        or      a                               ; if 0, that means screen y coord is currently on an exact block, so no 9th tile to display
        ret     z
        ld      b,1                             ; B = 1 tile to display
        ld      c,a                             ; C = # bytes (rows) to display for this tile
        xor     a
        ld      (scrollBits),a                  ; reset as this will be the last tile to display
        jr      sl3                             ; go back and display this tile, routine will finish after this last loop

;------------------------------------------------
; scrollRight - scroll the screen right a pixel
;
; input:    none
; output:   none
;------------------------------------------------
scrollRight:
; first, check that we're not already viewing the far left of the level
        ld      hl,(screenX)
        ld      a,l
        or      h                               ; is screen x pos 0?
        ret     z                               ; if so, we're at the far left of the level, so don't scroll
        ld      hl,__playerScrX
        inc     (hl)                            ; player will move right 1 pixel on screen as a result of the scroll

        ld      hl,gbuf                         ; HL => start of gbuf
        ld      c,64                            ; C = 64 rows to shift
sr1:
        ld      b,3                             ; B = loop 3 times (4 shifts per loop, equals total of 12 bytes per row)
        or      a                               ; clear carry flag
sr2:
        rr      (hl)                            ; rotate right 4 bytes
        inc     hl
        rr      (hl)
        inc     hl
        rr      (hl)
        inc     hl
        rr      (hl)
        inc     hl
        djnz    sr2                             ; and loop
        dec     c                               ; dec row counter
        jr      nz,sr1                          ; and loop
; draw tiles at left of screen
        ld      hl,(screenX)
        ld      a,l
        and     %00000111                       ; check if screenX is on an even block before scrolling
        ld      a,$B7                           ; A = "or a"
        jr      nz,srSetTriggerCheck
        ld      a,$37                           ; A = "scf"
srSetTriggerCheck:
        ld      (__srTriggerCheck),a            ; set instruction later in code
        dec     hl
        ld      (screenX),hl                    ; dec screen x coord and save it
        ld      a,l                             ; A = LSB of x coord of left side of screen
        and     $07                             ; A = x offset
        ld      hl,pixelTable+1
        ld      e,a
        ld      d,0
        add     hl,de
        ld      a,(hl)                          ; A = mask to use
        ld      (scrollMask),a                  ; save it
        ld      de,(screenX)                    ; DE = x coord
        ld      hl,(screenY)                    ; HL = y coord
        call    getTile                         ; A = tile #; HL => tile in tile data
        ld      (scrollPtr),hl                  ; save pointer to tile in tile data
        ld      a,(screenY)                     ; A = LSB of screen y coord
        and     $07
        ld      l,a
        ld      h,0                             ; HL = offset to use for tile sprite
        ld      (scrollSkip),hl                 ; save it
        ld      (scrollBits),a                  ; save # of bytes that will need to be displayed for last tile if screen y coord isn't an exact block
        sub     8
        neg                                     ; A = # rows to display for first tile
        ld      ix,gbuf                         ; IX => start of first row of gbuf
        ld      c,a                             ; C = # rows to display for first tile
        ld      b,8                             ; B = 8 tiles to display
sr3:
        ld      hl,(scrollPtr)                  ; HL => tile in tile data
        push    bc                              ; save counters
        push    hl                              ; save tile data ptr
#ifndef APP83P
        nop                                     ; "or a" or "scf" (set earlier in code)
__srTriggerCheck        = $-1
#else
        call    __srTriggerCheck
#endif
        call    c,checkEnemyTrigger             ; only call if screenY was previously on an even block before scrolling
        pop     hl                              ; restore tile data ptr
        pop     bc                              ; restore counters
        call    checkE2Trigger                  ; check tile to display and put it in HL
        add     hl,hl
        add     hl,hl
        add     hl,hl
        ld      de,startTiles
        add     hl,de                           ; HL => tile sprite
        ld      de,(scrollSkip) 
        add     hl,de                           ; HL => byte to get pixel from
        ld      de,12                           ; will be used after sr5
sr4:
        ld      a,(scrollMask)                  ; A = pixel mask
        and     (hl)                            ; apply it to tile sprite byte
        jr      z,sr5                           ; if blank pixel, don't turn pixel on in gbuf
        set     7,(ix)                          ; turn pixel on in gbuf
sr5:
        add     ix,de                           ; IX => start of next row of gbuf
        inc     hl                              ; HL => next byte of tile sprite
        dec     c                               ; dec byte counter
        jr      nz,sr4                          ; loop
        ld      hl,0
        ld      (scrollSkip),hl                 ; start at first byte of tile sprite next time
        ld      hl,(scrollPtr)                  ; HL => tile in tile data
        ld      de,(levelWidth)
        ld      d,0                             ; DE = level width
        add     hl,de                           ; HL => next tile to display in tile data
        ld      (scrollPtr),hl                  ; save pointer
        ld      c,8                             ; C = 8 bytes to display for next sprite
        djnz    sr3                             ; loop
        ld      a,(scrollBits)                  ; A = # bytes that need to be displayed for last 9th tile
        or      a                               ; if 0, that means screen y coord is currently an exact block, so no 9th tile to display
        ret     z
        ld      b,1                             ; B = 1 tile to display
        ld      c,a                             ; C = # bytes (rows) to display for this tile
        xor     a
        ld      (scrollBits),a                  ; reset as this will be the last tile to display
        jr      sr3                             ; go back and display this tile, routine will finish after this loop

;------------------------------------------------
; putScrollByte - display a byte of a tile sprite to a row on the gbuf at a specified x offset
;
; input:    A = x offset
;           HL => byte to display
;           DE => start of row on gbuf
; output:   none
;------------------------------------------------
putScrollByte:
        ex      de,hl                           ; HL => start of row on gbuf, DE => byte to display
        ld      b,0
        ld      c,a                             ; BC = x offset
        and     $07                             ; A = number of bits to shift / rotate right
        ld      (__psb),a                       ; store it later in the code
        srl     c
        srl     c
        srl     c                               ; BC = block x offset
        add     hl,bc                           ; HL => first byte to write to
        ld      a,(de)                          ; A = byte to display
        ld      d,a                             ; D = byte to display
        ld      e,b                             ; E = 0 (B is 0 from earlier)
#ifndef APP83P
        ld      a,$00                           ; A = number of bits to shift / rotate right (set earlier)
__psb                   = $-1
#else
        ld      a,(__psb)
#endif
        or      a                               ; any bits to shift / rotate?
        jr      z,psbSkip                       ; if not, skip the shift / rotate
psbLoop:
        srl     d                               ; shift first byte right
        rr      e                               ; rotate second byte right, which will add the carry flag to the MSB if it was set
        dec     a                               ; dec counter
        jr      nz,psbLoop                      ; loop again if req'd
psbSkip:
        ld      a,(hl)                          ; get byte from gbuf
        or      d                               ; add our first byte over it with OR
        ld      (hl),a                          ; save it
        inc     hl                              ; HL => next byte
        ld      a,(hl)                          ; repeat with second byte
        or      e
        ld      (hl),a
        ret

;------------------------------------------------
; svCheckEnemyTrigger - routine used in scrollVertical to check for enemy triggers
;
; input:    HL => tile in tile data to check
;           (__svTriggerCheck) = either "or a" or "scf"
; output:   HL => tile in tile data to check
;------------------------------------------------
svCheckEnemyTrigger:
        push    hl                              ; save tile data pointer
#ifndef APP83P
        .db     $00                             ; "or a" or "scf" (set before this routine was called code)
__svTriggerCheck        = $-1
#else
        call    __svTriggerCheck
#endif
        call    c,checkEnemyTrigger             ; only call if screenY was previously on an even block before scrolling
        pop     hl                              ; restore tile data ptr
        ret

.end
