;===============================================================;
;                                                               ;
; Alien Breed 5                                                 ;
; Player routines                                               ;
;                                                               ;
;===============================================================;

;------------------------------------------------
; playerUp - try to move player up
;
; input:    none
; output:   none
;------------------------------------------------
playerUp:
        xor     a
        ld      (playerDir),a                   ; set direction
; check if a solid tile is in the way
        ld      de,(playerX)
        inc     de
        ld      hl,(playerY)
        call    checkTile                       ; check tile at (playerX+1,playerY)
        ret     c
        ld      hl,(playerX)
        ld      de,6
        add     hl,de
        ex      de,hl
        ld      hl,(playerY)
        call    checkTile                       ; check tile at (playerX+6,playerY)
        ret     c
; check if at top of level
        ld      hl,(playerY)
        ld      a,h
        or      l                               ; check if at top of level
        ret     z                               ; if so, can't move
        dec     hl
        ld      (playerY),hl                    ; save new y coord
        ld      hl,__playerScrY
        dec     (hl)                            ; player has moved up on screen by 1 pixel
        ret

;------------------------------------------------
; playerDown - try to move player down
;
; input:    none
; output:   none
;------------------------------------------------
playerDown:
        ld      a,1
        ld      (playerDir),a                   ; set direction
; check if a solid tile is in the way
        ld      de,(playerX)
        inc     de
        ld      hl,(playerY)
        ld      bc,7
        add     hl,bc
        call    checkTile                       ; check tile at (playerX+1,playerY+7)
        ret     c
        ld      hl,(playerX)
        ld      de,6
        add     hl,de
        ex      de,hl
        ld      hl,(playerY)
        ld      bc,7
        add     hl,bc
        call    checkTile                       ; check tile at (playerX+6,playerY+7)
        ret     c
; check if at bottom of level
        ld      de,(playerY)
        ld      hl,(levelHeight)
        ld      h,0
        dec     l
        add     hl,hl
        add     hl,hl
        add     hl,hl                           ; HL = level height in pixels - 8
        ex      de,hl                           ; DE = level height in pixels - 8; HL = player y coord
        bcall(_cphlde)                          ; check if at bottom of level
        ret     z                               ; if so, can't move
        inc     hl
        ld      (playerY),hl                    ; save new y coord
        ld      hl,__playerScrY
        inc     (hl)                            ; player has moved down on screen by 1 pixel
        ret

;------------------------------------------------
; playerLeft - try to move player left
;
; input:    none
; output:   none
;------------------------------------------------
playerLeft:
        ld      a,2
        ld      (playerDir),a                   ; set direction
; check if a solid tile is in the way
        ld      de,(playerX)
        ld      hl,(playerY)
        inc     hl
        call    checkTile                       ; check tile at (playerX,playerY+1)
        ret     c
        ld      de,(playerX)
        ld      hl,(playerY)
        ld      bc,6
        add     hl,bc
        call    checkTile                       ; check tile at (playerX,playerY+6)
        ret     c
; check if at far left of level
        ld      hl,(playerX)
        ld      a,h
        or      l                               ; check if at far left of level
        ret     z                               ; if so, can't move
        dec     hl
        ld      (playerX),hl                    ; save new x coord
        ld      hl,__playerScrX
        dec     (hl)                            ; player has moved left on screen by 1 pixel
        ret

;------------------------------------------------
; playerRight - try to move player right
;
; input:    none
; output:   none
;------------------------------------------------
playerRight:
        ld      a,3
        ld      (playerDir),a                   ; set direction
; check if a solid tile is in the way
        ld      hl,(playerX)
        ld      de,7
        add     hl,de
        ex      de,hl
        ld      hl,(playerY)
        inc     hl
        call    checkTile                       ; check tile at (playerX+7,playerY+1)
        ret     c
        ld      hl,(playerX)
        ld      de,7
        add     hl,de
        ex      de,hl
        ld      hl,(playerY)
        ld      bc,6
        add     hl,bc
        call    checkTile                       ; check tile at (playerX+7,playerY+6)
        ret     c
; check if at far right of level
        ld      de,(playerX)
        ld      hl,(levelWidth)
        ld      h,0
        dec     l
        add     hl,hl
        add     hl,hl
        add     hl,hl
        ex      de,hl                           ; DE = level width in pixels - 8; HL = player x coord
        bcall(_cphlde)                          ; check if at far right of level
        ret     z                               ; if so, can't move
        inc     hl
        ld      (playerX),hl                    ; save new x coord
        ld      hl,__playerScrX
        inc     (hl)                            ; player has moved right on screen by 1 pixel
        ret

;------------------------------------------------
; playerUse - check for a usable tile in front of player and if one is found, use it
;
; input:    none
; output:   none
;------------------------------------------------
playerUse:
        ld      hl,(playerDir)
        ld      h,0                             ; HL = player direction
        add     hl,hl
        add     hl,hl                           ; *4
        ld      de,playerUseCoordsTable
        add     hl,de                           ; HL => required offsets
        ld      c,(hl)
        inc     hl
        ld      b,(hl)                          ; BC = x offset
        inc     hl
        ex      de,hl                           ; DE => y offset
        ld      hl,(playerX)
        add     hl,bc                           ; HL = x coord to check
        ex      de,hl                           ; DE = x coord to check; HL => y offset
        ld      c,(hl)
        inc     hl
        ld      b,(hl)                          ; BC = y offset
        ld      hl,(playerY)
        add     hl,bc                           ; HL = y coord to check
        call    getTile                         ; A = tile
; check table to see if it's a valid "use" tile
        ex      de,hl                           ; DE => tile in level data
        ld      bc,NUM_USE_TILES*256+0          ; B = num entries to check; C = index counter (starting at 0)
        ld      hl,playerUseTileTable           ; HL => start of table
checkPlayerUseTile:
        cp      (hl)                            ; check if tile matches an entry
        jr      z,playerUseFound                ; if it matches, go to next step
        inc     hl                              ; HL => next entry
        inc     c                               ; inc index counter
        djnz    checkPlayerUseTile              ; loop
        ret                                     ; no useable tiles found, so leave

playerUseFound:
        ld      l,c
        ld      h,0                             ; HL = index counter
        add     hl,hl                           ; *2
        ld      bc,playerUseRoutines
        add     hl,bc                           ; HL => routine to execute
        bcall(_ldhlind)                         ; HL = routine to execute
        jp      (hl)                            ; go to routine; note that DE => tile in level data

;------------------------------------------------
; drawPlayer - draw player on gbuf
;
; input:    none
; output:   none
;------------------------------------------------
drawPlayer:
        ld      a,(playerInv)
        bit     0,a                             ; only draw player if bit 0 is reset
        ret     nz
#ifndef APP83P
        ld      bc,$0000                        ; B = player x on screen; C = player y on screen (will be initialised during drawLevel routine when loading level)
__playerScrX            = $-1
__playerScrY            = $-2
#else
        ld      bc,(__playerScrY)
#endif
        ld      a,(playerDir)
        add     a,a
        add     a,a
        add     a,a
        add     a,a                             ; A = player direction * 16 (offset to sprite)
        ld      e,a
        ld      d,0
        ld      hl,sprPlayer
        add     hl,de                           ; HL => masked sprite to display if player wasn't hurt this frame
        ld      a,(playerHurt)
        or      a                               ; was player hurt this frame?
        jr      z,doDrawPlayer                  ; if so, go ahead and draw
        ld      de,64
        add     hl,de                           ; HL => masked sprite for hurt player
doDrawPlayer:
        jp      putClippedMaskedSprite

;------------------------------------------------
; checkTile - check a tile to see if it's a wall, trigger, pickup item, etc.
;
; input:    DE = x
;           HL = y
; output:   CA = 1 if can't walk through
;------------------------------------------------
checkTile:
        call    getTile                         ; A = tile number
        cp      TILE_DOOR                       ; is it a key door?
        jr      nc,tryOpenDoor                  ; if so, try to open it
        cp      TILE_UNOPENALIENHOLE            ; is it an unopen alien hole?
        jr      z,activateAlienHole             ; if so, try to activate it
        cp      TILE_MISCTRIGGER                ; is it a misc trigger?
        jr      z,triggerAutoDestruct           ; if so, trigger auto destruct sequence
        cp      TILE_BOMB                       ; is it a bomb?
        jr      z,triggerBomb                   ; if so, trigger bomb
        cp      TILE_ENEMYTRIGGER+1             ; is it an enemy2 trigger?
        jr      z,checkTileMove                 ; if so, can always walk on it (in case player is on bare/exposed floor)
#ifndef APP83P
        cp      TILE_SOLID                      ; is it a tile that can't be walked through?
; if player is standing on tile 0 (bare/exposed floor type), then code is modified so that the player
; can only walk on tiles 0-4 inclusive; this check is performed in the main game loop
__ctTileType            = $-1
#else
        ld      bc,(__ctTileType)               ; C = (__ctTileType)
        cp      c
#endif
        jr      c,checkEnergyShield             ; if not, check if it's an energy shield
checkTileNoMove:
#ifndef APP83P
        nop                                     ; will be set to either "or a" or "scf"
__noWallsCheat          = $-1
#else
        call    __noWallsCheat
#endif
        ccf                                     ; complement carry
        ret
activateAlienHole:
        call    loadAlienHole                   ; try to load alien hole if one isn't already active
        jr      checkTileMove                   ; ok to move
triggerBomb:
        ld      a,TILE_TRIGGEREDBOMB            ; A = tile to set
        call    putTile                         ; place the tile & display it
; fall through to triggerAutoDestruct
triggerAutoDestruct:
        ld      a,1
        ld      (objectives),a                  ; set auto destruct sequence
        call    putDestructBlock                ; put the "destruct" block
        jr      checkTileMove                   ; ok to move
tryOpenDoor:
        ld      a,(keys)
        or      a                               ; does player have any key passes?
        jr      z,checkTileNoMove               ; if not, can't move
        dec     a
        ld      (keys),a                        ; use a key and save new value
        ld      a,(hl)                          ; A = tile number
        call    openDoor                        ; open the door
        jr      checkTileMove                   ; indicate that it's ok to move and ret
checkEnergyShield:
        cp      TILE_ENERGYSHIELD               ; is tile an energy shield
        jr      c,checkPickups                  ; if not, check pickups
        sub     TILE_ENERGYSHIELD               ; A = direction of energy shield
        ld      hl,playerDir
        cp      (hl)                            ; compare energy shield direction to player direction
        jr      nz,checkTileNoMove              ; can't move if directions don't match
checkTileMove:
        or      a                               ; clear carry
        ret
checkPickups:
        push    hl                              ; save tile data ptr
        sub     PICKUPS                         ; is tile a key pass?
        jr      z,getKeyPass
        dec     a                               ; is it ammo?
        jr      z,getAmmo
        dec     a                               ; is it 10 credits?
        jr      z,get10Credits
        dec     a                               ; is it 100 credits?
        jr      z,get100Credits
        dec     a                               ; is it first aid?
        jr      z,getFirstAid
        dec     a                               ; is it an extra life?
        jr      z,getExtraLife
        pop     hl                              ; restore tile data ptr
        or      a                               ; no pickup, so clear carry and leave
        ret
getKeyPass:
        ld      hl,keys
        inc     (hl)                            ; add a key
        jr      getPickupScore
getAmmo:
        ld      hl,getPickupScore               ; HL => where to ret to
        push    hl                              ; put it on the stack
addAmmoClip:
        ld      hl,(ammo)
        ld      l,255                           ; L = full clip
        inc     h                               ; H = number of spare clips
        ld      a,h
        cp      5                               ; check to make sure no more than 4 spare clips
        jr      c,setAmmo
        ld      h,4                             ; if too many, set it to 4 spare clips
setAmmo:
        ld      (ammo),hl
        ret                                     ; ret either to getPickupScore or intexBuyAmmoClip depending on where this routine was called from
get10Credits:
        ld      de,10
        jr      getCredits
get100Credits:
        ld      de,100
getCredits:
        ld      hl,(credits)
        add     hl,de                           ; add credits
        ld      (credits),hl
        jr      getPickupScore
getFirstAid:
        ld      a,90
        ld      (showHealth),a                  ; set counter to show health meter for the next A frames
        ld      hl,health
        ld      a,MAX_HEALTH/4
        add     a,(hl)                          ; add health
        ld      (hl),a
        cp      MAX_HEALTH                      ; check it's not over the max
        jr      c,getPickupScore
        ld      (hl),MAX_HEALTH                 ; if it was, set it to MAX_HEALTH
        jr      getPickupScore
getExtraLife:
        ld      hl,lives
        inc     (hl)                            ; add a life
getPickupScore:
        ld      hl,SCORE_PICKUP
        call    addScore                        ; increase score
        pop     hl                              ; restore tile data ptr
        ld      a,TILE_FLOOR
        call    putTile                         ; clear pickup from level
        or      a                               ; clear carry flag
        ret

;------------------------------------------------
; addScoreDiff - add to the players score, adjusting for difficulty
;
; input:    HL = base value to add before adjusting for difficulty
; output:   HL = new score
;------------------------------------------------
addScoreDiff:
        ld      a,(difficulty)
        or      a                               ; check difficulty
        jr      z,addScore                      ; if NORMAL, don't double the value
        add     hl,hl
; fall through to addScore

;------------------------------------------------
; addScore - add to the players score
;
; input:    HL = value to add
; output:   HL = new score
;------------------------------------------------
addScore:
        ld      de,(score)
        add     hl,de
        ld      (score),hl                      ; save new score
        ret

;------------------------------------------------
; viewRadarScanner - view radar scanner (if it is owned)
;
; input:    none
; output:   none
;------------------------------------------------
viewRadarScanner:
        ld      hl,weaponTable                  ; HL => weapon flags (radar scanner is stored as a 6th weapon)
        bit     5,(hl)                          ; check if available
        ret     z                               ; if not, leave
        call    saveVideo                       ; save gbuf
        call    fadeOutAndClear
        ld      de,28*256+29
        ld      hl,strScanning
        call    vPuts                           ; "SCANNING..."
        call    libFastCopy
        ld      b,80
        call    wait
        ld      a,(objectives)
        dec     a                               ; check if self-destruct sequence is enabled
        call    nz,showMap                      ; if not, show the level map
        call    fadeOutAndClear
        jp      restoreVideo                    ; restore gbuf and leave

;------------------------------------------------
; selectWeaponX - select a weapon
;
; input:    none
; output:   none
;------------------------------------------------
selectWeapon1:
        xor     a
        inc     a                               ; A = weapon #, and also NZ (player always has this weapon)
selectWeapon:
        ret     z                               ; leave it Z (player doesn't have this weapon)
        ld      (weapon),a                      ; set new weapon type
        ret
selectWeapon2:
        ld      a,2                             ; A = weapon #
        ld      hl,weaponTable
        bit     1,(hl)                          ; check if weapon available
        jr      selectWeapon
selectWeapon3:
        ld      a,3                             ; A = weapon #
        ld      hl,weaponTable
        bit     2,(hl)                          ; check if weapon available
        jr      selectWeapon
selectWeapon4:
        ld      a,4                             ; A = weapon #
        ld      hl,weaponTable
        bit     3,(hl)                          ; check if weapon available
        jr      selectWeapon
selectWeapon5:
        ld      a,5                             ; A = weapon #
        ld      hl,weaponTable
        bit     4,(hl)                          ; check if weapon available
        jr      selectWeapon

;------------------------------------------------
; displayHealthMeter - display player health meter
;
; input:    none
; output:   none
;------------------------------------------------
displayHealthMeter:
        ld      hl,gbuf+(50*12)                 ; HL => where on gbuf to start bottom of meter
        ld      a,(health)                      ; A = player health (0-127)
displayMeterDiv4:                               ; hack for pause routine to use when displaying ammo meter
        srl     a
        srl     a                               ; A = player health / 4 (0-31)
; fall through to displayMeter

;------------------------------------------------
; displayMeter - display a meter (used for health & ammo)
;
; input:    HL => where on gbuf to start (the meter is drawn from the bottom up)
;           A = meter fill value to display (0-31)
; output:   none
;------------------------------------------------
displayMeter:
        ld      c,a                             ; C = meter fill value
        ld      de,-12                          ; DE = offset to next line up on gbuf
        ld      a,(hl)
        and     %10000000
        ld      (hl),a                          ; clear white space below meter
        add     hl,de                           ; HL => next line up on gbuf
        ld      a,(hl)
        and     %10000000
        or      %00111110
        ld      (hl),a                          ; draw bottom border of meter
        add     hl,de                           ; HL => next line up on gbuf
        ld      b,31                            ; B = 31 lines of meter to draw
displayMeterLoop:
        push    de                              ; save gbuf line offset
        ld      a,(hl)                          ; get byte from gbuf
        and     %10000000                       ; apply mask to clear where meter is displayed
        or      %00100010                       ; apply meter side outlines
        ld      d,a                             ; D = potential byte to be written back to gbuf
        ld      a,c                             ; A = meter value counter
        or      a                               ; check if it's 0
        ld      a,d                             ; get byte back
        jr      z,displayMeterDraw              ; if so, just draw the byte already have
        dec     c                               ; dec meter value counter
        or      %00011100                       ; apply meter fill to byte for gbuf
displayMeterDraw:
        ld      (hl),a                          ; write byte back to gbuf
        pop     de                              ; restore gbuf line offset
        add     hl,de                           ; HL => next line up on gbuf
        djnz    displayMeterLoop
        ld      a,(hl)
        and     %10000000
        or      %00111110
        ld      (hl),a                          ; draw top border of meter
        add     hl,de                           ; HL => next line up on gbuf
        ld      a,(hl)
        and     %10000000
        ld      (hl),a                          ; clear white space above meter
        ret

.end
