;===============================================================;
;                                                               ;
; Alien Breed 5                                                 ;
; Miscellaneous routines                                        ;
;                                                               ;
;===============================================================;

;------------------------------------------------
; waitKey - wait for a key press
;
; input:    none
; output:   A = key pressed (_getcsc style)
;------------------------------------------------
waitKey:
        ei                                      ; enable interrupts
        res     apdrunning,(iy+apdflags)        ; disable apd
wKeyLoop:
        halt
        bcall(GET_KEY)                          ; A = key pressed
        or      a                               ; check if 0
        jr      z,wKeyLoop                      ; if so, no key was pressed
        set     apdrunning,(iy+apdflags)        ; enable apd
        ret

;------------------------------------------------
; waitEnter - wait for [2nd]
;
; input:    none
; output:   none
;------------------------------------------------
wait2nd:
        call    waitKey                         ; get a key press
        cp      GK_2ND                          ; [2nd]?
        jr      nz,wait2nd                      ; if not, wait again
        ret

;------------------------------------------------
; wait - wait for a specified time
;
; input:    B = number of times to halt
; output:   none
;------------------------------------------------
wait:
        ei
        res     apdrunning,(iy+apdflags)        ; disable apd
waitLoop:
        halt
        djnz    waitLoop
        set     apdrunning,(iy+apdflags)        ; enable apd
        ret

;------------------------------------------------
; saveVideo - save gbuf to videoBuffer
;
; input:    none
; output:   none
;------------------------------------------------
saveVideo:
        ld      hl,gbuf
        ld      de,videoBuffer
copyVideo:
        ld      bc,768
        ldir
        ret

;------------------------------------------------
; restoreVideo - restore videoBuffer to gbuf
;
; input:    none
; output:   none
;------------------------------------------------
restoreVideo:
        ld      hl,videoBuffer
        ld      de,gbuf
        jr      copyVideo

;------------------------------------------------
; fadeOut - fade out by decreasing the contrast
;
; input:    none
; output:   (contrastBuffer) = original system contrast setting
;------------------------------------------------
fadeOut:
        ld      a,(contrast)
        ld      (contrastBuffer),a              ; save original contrast setting
        ld      b,20                            ; decrement a max of 20 times
fadeOutLoop:
        ld      c,b                             ; C = dec counter
        ld      a,(contrast)
        dec     a
        push    af
        call    setContrast                     ; save new contrast and set it
        pop     af
        or      a                               ; is contrast 0?
        ret     z                               ; if so, leave
        ld      b,6
        call    wait
        ld      b,c                             ; B = dec counter
        djnz    fadeOutLoop                     ; loop
        ret

;------------------------------------------------
; fadeIn - fade in by increasing the contrast back to the original system setting
;
; input:    (contrastBuffer) = original system contrast setting
; output:   none
;------------------------------------------------
fadeIn:
        ld      a,(contrast)
        inc     a
        push    af
        call    setContrast                     ; save new contrast and set it
        pop     af
        ld      hl,contrastBuffer               ; HL => value that we're aiming to return to
        cp      (hl)                            ; have we hit the limit?
        ret     z                               ; if so, leave
        ld      b,6
        call    wait
        jr      fadeIn

;------------------------------------------------
; restoreContrast - instantly restore lcd constrast to original system setting
;
; input:    (contrastBuffer) = original system setting
; output:   none
;------------------------------------------------
restoreContrast:
        ld      a,(contrastBuffer)              ; A = value to set
; fall through to setContrast

;------------------------------------------------
; setContrast - set contrast level
;
; input:    A = contrast value (unmasked, etc.)
; output:   none
;------------------------------------------------
setContrast:
        ld      (contrast),a                    ; save new contrast
        add     a,$18
        or      $C0                             ; bit mask
        bcall(_lcdbusy)                         ; wait for lcd
        out     ($10),a                         ; set contrast
        ret

;------------------------------------------------
; fadeOutAndClear - fade out the screen and then restore contrast with a clear screen
;
; input:    none
; output:   none
;------------------------------------------------
fadeOutAndClear:
        call    fadeOut
        bcall(_grbufclr)
        call    libFastCopy
        jr      restoreContrast

;------------------------------------------------
; invertLine - invert lines on gbuf
;
; input:    HL => where to start inverting
;           B = number of gbuf lines to invert
; output:   none
;------------------------------------------------
invertLine:
        ld      c,12                            ; C = 12 bytes per gbuf line
invertLineLoop:
        ld      a,(hl)                          ; read byte
        cpl                                     ; invert it
        ld      (hl),a                          ; save it
        inc     hl                              ; HL => next byte
        dec     c                               ; dec byte counter
        jr      nz,invertLineLoop               ; loop until all bytes on line have been inverted
        djnz    invertLine                      ; loop until all lines have been inverted
        ret

;------------------------------------------------
; absHL - check if HL is negative, and if so, make it positive
;
; input:    HL = value to check
; output:   HL = abs(HL)
;------------------------------------------------
absHL:
        bit     7,h                             ; check if MSB is set
        ret     z                               ; if not, HL is a positive value
        ld      a,h
        cpl
        ld      h,a
        ld      a,l
        cpl
        ld      l,a
        ret

;------------------------------------------------
; setShowHL - show HL in small font at specified position
;
; input:    HL = value to show
;           DE = penrow, pencol
; output:   none
;------------------------------------------------
setShowHL:
        ld      (pencol),de

;------------------------------------------------
; showHL - show HL in small font at cursor position
;
; input:    HL = value to show
; output:   None
;------------------------------------------------
showHL:
        ld      de,showHLBuffer+5               ; DE => end of string buffer
        ld      b,5                             ; B = number of characters to calculate
        xor     a
        ld      (de),a                          ; set NULL terminator at end of string buffer and work backwards from there
showHLMake:
        dec     de                              ; DE => next character back
        bcall(_divHLby10)                       ; A = number to display
        add     a,'0'                           ; A = character to display
        ld      (de),a                          ; store it
        djnz    showHLMake
        ex      de,hl                           ; HL => start of string
; now check string and eliminate all leading zeros, and adjust pencol accordingly (string is right-aligned)
        ld      b,4                             ; B = number of characters to check (last character is always shown)
        ld      a,(pencol)                      ; A = pencol
showHLCheck:
        ld      c,a                             ; C = pencol
        ld      a,(hl)                          ; A = char
        cp      '0'                             ; is it a 0?
        ld      a,c                             ; A = pencol
        jr      nz,showHLReady                  ; if not, ready to show string
        add     a,4                             ; A = new pencol (skipping where this character would have been shown)
        inc     hl                              ; HL => next char in string
        djnz    showHLCheck
showHLReady:
        ld      (pencol),a                      ; set new pencol
        bcall(_vputs)
        ret

;------------------------------------------------
; showStrings - show a set of strings
;
; input:    HL => string data
; output:   HL => after string data
;------------------------------------------------
showStrings:
        ld      b,(hl)                          ; B = number of strings to display
        inc     hl
showStringsLoop:
        push    bc
        ld      d,(hl)                          ; D = penrow value
        inc     hl
        ld      e,(hl)                          ; E = pencol value
        inc     hl                              ; HL => string
        call    vPuts
        pop     bc
        djnz    showStringsLoop
        ret

;------------------------------------------------
; getString - get a string of characters from the user
;
; input:    HL => where to store them
;           A = "scf" to enable pressing [ENTER]; "or a" to disable pressing [ENTER]
; output:   none
;------------------------------------------------
getString:
        ld      (__getStringEnter),a            ; save instruction regarding pressing [ENTER]
        push    hl                              ; save ptr
        ld      hl,strBlankString
        ld      de,35*256+28
        push    de                              ; save cursor pos
        call    vPuts                           ; "__________"
        pop     hl                              ; HL => value for pencol to put cursor to start of the '_'
        ld      (pencol),hl
        pop     hl                              ; restore ptr
        ld      b,10                            ; B = max 10 characters
getStringLoop:
        push    bc                              ; save counter
        push    hl                              ; save ptr
        call    libFastCopy                     ; update gbuf to lcd
getChar:
        call    waitKey                         ; A = key press
        cp      GK_DEL                          ; [DEL]?
        jr      z,gsBackspace                   ; if so, try to backspace
        cp      GK_ENTER                        ; [ENTER]?
        jr      z,gsEnter                       ; if so, try to finish early
        dec     a
        ld      e,a
        ld      d,0                             ; DE = offset for charTable
        ld      hl,charTable
        add     hl,de                           ; HL => character entered
        ld      a,(hl)
        cp      '.'                             ; a full stop means invalid character
        jr      z,getChar                       ; if invalid, get another character
        pop     hl                              ; restore ptr
        ld      (hl),a                          ; save character
        bcall(_vputmap)                         ; display character
        inc     hl                              ; update ptr for next character
        pop     bc                              ; restore counter
        djnz    getStringLoop
        ret
gsEnter:
#ifndef APP83P
        nop                                     ; instruction will be set at start of routine, either "scf" or "or a"
__getStringEnter        = $-1
#else
        call    __getStringEnter
#endif
        jr      nc,getChar                      ; if NC, [ENTER] disabled, so get another key press
; otherwise, fill remainder of string storage with spaces and leave
        pop     hl                              ; restore ptr
        pop     bc                              ; restore counter
gsFillSpace:
        ld      (hl),' '                        ; write space
        inc     hl                              ; update ptr
        djnz    gsFillSpace                     ; loop until string is filled
        ret
gsBackspace:
        pop     hl                              ; restore ptr
        pop     bc                              ; restore counter
        ld      a,b
        cp      10                              ; check if at first character
        jr      z,getStringLoop                 ; if so, can't backspace, so go back to loop
        inc     b                               ; inc counter, as we'll need an extra character due to backspacing
        dec     hl                              ; dec ptr
        ld      a,(pencol)
        sub     4
        ld      (pencol),a                      ; move pencol back to write '_'
        ld      c,a                             ; C = pencol (will be needed again after writing the '_')
        ld      a,'_'
        bcall(_vputmap)                         ; display '_'
        ld      a,c
        ld      (pencol),a                      ; set pencol back again to be ready for next character
        jr      getStringLoop                   ; return to loop

;------------------------------------------------
; MirageOS / DoorsCS routines that need to be written for Ion/App versions
;------------------------------------------------
#ifdef NOTMOSDCS

;------------------------------------------------
; directIn - read keypresses from a key group
;
; input:    A = key group code
; output:   A = key presses from specified group
;------------------------------------------------
directIn:
        ld      b,a                             ; B = key group code
        ld      a,$FF
        out     (1),a                           ; clear key port
        ld      a,b
        out     (1),a                           ; put key group code to key port
        in      a,(1)                           ; A = key presses from that key group
        ret

;------------------------------------------------
; vPuts - same as _vputs but sets cursor position
;
; input:    DE = value to store to (pencol)
;           HL => string to display
; output:   HL => next string
;------------------------------------------------
vPuts:
        ld      (pencol),de
#ifdef APP83P                                   ; flash app version needs to write the string to a location in ram
        ld      de,strBuffer                    ; DE => where to write it to
vPutsCopy:
        ld      a,(hl)                          ; A = character
        ld      (de),a                          ; save it
        inc     hl
        inc     de
        or      a                               ; check if NULL terminator
        jr      nz,vPutsCopy                    ; if not, do another char
        push    hl                              ; preserve pointer to next string
        ld      hl,strBuffer                    ; HL => string to display
        bcall(_vputs)
        pop     hl                              ; HL => next string
#else
        bcall(_vputs)
#endif
        ret

#else

vPuts           = setVPuts                      ; redirection to MirageOS / DoorsCS library function

#endif

;------------------------------------------------
; library routines for Ion on TI-83+ must be included as part of the program so that they don't exceed the TI-83+ $C000 execution limit
; thanks to JoeW for the source code to these, taken direction from Ion
;
; also included for the 83+ app version
;------------------------------------------------
#ifdef LIB83P

;-----> Generate a random number
; input b=upper bound
; ouput a=answer 0<=a<b
; all registers are preserved except: af and bc
libRandom:
    push    hl
    push    de
    ld  hl,(__randData)
    ld  a,r
    ld  d,a
    ld  e,(hl)
    add hl,de
    add a,l
    xor h
    ld  (__randData),hl
    sbc hl,hl
    ld  e,a
    ld  d,h
randomLoop:
    add hl,de
    djnz    randomLoop
    ld  a,h
    pop de
    pop hl
    ret
#ifndef APP83P
__randData:
    .dw 0
#endif

;-----> Copy the gbuf to the screen (fast)
;Input: nothing
;Output:graph buffer is copied to the screen
libFastCopy:
    di
    ld  a,$80               ; 7
    out ($10),a             ; 11
    ld  hl,gbuf-12-(-(12*64)+1)     ; 10
    ld  a,$20               ; 7
    ld  c,a             ; 4
    inc hl              ; 6 waste
    dec hl              ; 6 waste
fastCopyAgain:
    ld  b,64                ; 7
    inc c               ; 4
    ld  de,-(12*64)+1           ; 10
    out ($10),a             ; 11
    add hl,de               ; 11
    ld  de,10               ; 10
fastCopyLoop:
    add hl,de               ; 11
    inc hl              ; 6 waste
    inc hl              ; 6 waste
    inc de              ; 6
    ld  a,(hl)              ; 7
    out ($11),a             ; 11
    dec de              ; 6
    djnz    fastCopyLoop            ; 13/8
    ld  a,c             ; 4
    cp  $2B+1               ; 7
    jr  nz,fastCopyAgain        ; 10/1
    ret                 ; 10

;-----> Detect a file
; input:
;   hl=place to start looking
;   ix->first line of data (0 terminated)
; output:
;   de=place stopped + 1
;   hl->program data (after the string)
;   z=0 if found, z=1 if not found.
;   All registers destroyed

libDetect: ld  de,(ptemp)
    bcall(_cphlde)
    ld  a,(hl)
    jr  nz,detectContinue
    inc a
    ret
detectContinue:
    push    hl
#ifndef APP83P
    cp $06
#else
    cp $15
#endif
    jr  nz,detectSkip

    dec hl
    dec hl
    dec hl  ; hl->lsb ptr
    ld  e,(hl)
    dec hl
    ld  d,(hl)
    dec hl  ; hl->page
    ld  a,(hl)
    or  a
    push    af
    ld  h,d
    ld  l,e ; hl & de->program
    jr  z,detectNoMove
    push    hl
    bcall(_memfree)
    ld  bc,64
    sbc hl,bc
    pop hl
    jr  c,detectNotEnough
    ld  de,($9820)
    push    ix
    push    hl
    push    de
    bcall(_flashToRam)
    pop hl
    push    hl
    pop ix
    ld  a,10
    add a,(ix+9)
    ld  e,a
    ld  d,0 ; de=flash offset
    add hl,de
    ex  (sp),hl
    add hl,de
    pop de
    ex  de,hl   ; hl-temp, de-perm
    pop ix
detectNoMove:
    inc de
    inc de
    ld  c,(hl)
    inc hl
    ld  b,(hl)
    inc hl  ; hl->data in ram
    push    bc
    push    ix
    pop bc
detectCheck:
    ld  a,(bc)
    or  a
    jr  z,detectFound
    cp  (hl)
    inc bc
    inc de
    inc hl
    jr  z,detectCheck
detectBad:
    pop bc
detectNotEnough:
    pop af
detectSkip:
    pop hl
    ld  bc,-6
    add hl,bc
    ld  b,(hl)
    dec hl
detectNameLoop2:
    dec hl
    djnz    detectNameLoop2
    jr  libDetect
detectFound:
    pop hl
    ; hl=size, de->data
    pop af  ; a=page, f=(or a)
    jr  z,detectInRam
    push    de  ; data
    push    af
    push    hl
    bcall(_enoughRam)
    pop bc
    jr  c,detectBad
    pop af
    pop hl
    ld  de,($9820)  ; tempMem
    push    de
    bcall(_flashToRam)
    pop de
detectInRam:    ; de->data in RAM
    pop hl  ; hl->vat location
    ld  bc,-6
    add hl,bc
    ld  b,(hl)
    inc b
detectNameLoop1:
    dec hl
    djnz    detectNameLoop1
    ex  de,hl
    xor a
    ret

#endif

.end
