;===============================================================;
;                                                               ;
; Alien Breed 5                                                 ;
; Level routines                                                ;
;                                                               ;
;===============================================================;

;------------------------------------------------
; loadLevel - load current level
;
; input:    none
; output:   none
;------------------------------------------------
loadLevel:
#ifdef APP83P
        ld      a,%11100000
        ld      (__loadingIndicator),a          ; set start state of loading indicator
#endif
        ld      hl,(level)
        srl     l                               ; L = level number / 2 (2 levels per data file)
        ld      h,6                             ; H = 6 bytes per data file name
        bcall(_htimesl)
        ex      de,hl                           ; DE = offset to string
        ld      ix,episode1SearchStr            ; IX => start of search strings
        add     ix,de                           ; IX => relevant search string
        ld      hl,(vat)                        ; HL => start of VAT
        call    libDetect                       ; search for an episode file
        jr      z,dataFileFound                 ; if found, continue loading
        pop     hl                              ; clear RET ptr from stack
        jp      dataFileMissing

dataFileFound:
        ld      (levelFilesPtr),hl              ; save a pointer to start of compressed level files
        xor     a
        ld      (state),a                       ; make sure level is started in normal game state
        ld      (playerDead),a
        ld      (playerInv),a
        ld      (playerHurt),a
        ld      (showHealth),a
        ld      l,a
        ld      h,a                             ; HL = 0
        ld      (__tileShotPtr),hl              ; reset ptr to previously shot tile in level data

        call    clearAnimTable                  ; clear animTable
        call    clearBulletTable                ; clear bulletTable
        call    clearAllEnemyTables             ; clear enemyTable, aholeTable

        ld      a,(level)                       ; A = level number
        and     %00000001                       ; A = file number (0-1)
        ld      b,a                             ; B = file number
        ld      hl,(levelFilesPtr)              ; HL => start of compressed level files
        ld      de,(tempProg)                   ; DE => where to decompress level to
        ld      ix,gbuf                         ; IX => 1024 of temp mem for huffExtr
        call    huffExtr                        ; extract current level

        call    showObjectives                  ; display mission objectives

        ld      de,levelWidth                   ; DE => where to store level stats
        ld      bc,LEVEL_STATS                  ; BC = number of bytes to copy
        ldir                                    ; copy level stats
        ld      (tileDataPtr),hl                ; save pointer to tile data

; make enemy & boss triggers look like the right type of floor
        ld      a,(levelFlags)                  ; A = flags for level
        bit     LFLAG_LEVELTYPE,a               ; check if normal or hatchery
        ld      hl,tileTiledFloor
        jr      z,startFixTiles
        ld      hl,tileHatchery
startFixTiles:
        ld      (__fixTiles),hl                 ; save ptr to tile to copy
        ld      hl,tileFixTable                 ; HL => tile numbers to paste to
        ld      b,NUM_TILES_FIX                 ; B = number of tiles to paste over
fixTilesLoop:
        push    bc                              ; save counter
        push    hl                              ; save destination tile number ptr
        ld      l,(hl)
        ld      h,0
        add     hl,hl
        add     hl,hl
        add     hl,hl
        ld      de,startTiles
        add     hl,de                           ; HL => where to paste to
#ifndef APP83P
        ld      de,$0000                        ; DE => where to copy from (will be set from earlier)
__fixTiles               = $-2
#else
        ld      de,(__fixTiles)
#endif
        ld      bc,8                            ; BC = 8 bytes to copy & paste
        ex      de,hl                           ; swap source & destination
        ldir                                    ; copy tile
        pop     hl
        inc     hl                              ; HL => number of next tile to copy over
        pop     bc
        djnz    fixTilesLoop

; prepare to calculate all wall tiles
        ld      a,(levelHeight)
        dec     a
        ld      c,a                             ; C = y to start at (bottom of level, and work back up)
calcWallsLoop1:
        ld      a,(levelWidth)
        dec     a
        ld      b,a                             ; B = x to start at (far right of level, and work back to the left)
calcWallsLoop2:
        push    bc                              ; save x/y pos
        call    checkWall                       ; check tile to see if it's a wall
        jr      nc,endCWLoop                    ; if not, go to next tile
        ld      (__wallTilePtr),hl              ; save ptr to tile in tile data
        xor     a
        ld      (__wallFlags),a                 ; reset wall flags (one bit for each tile surrounding the current tile)
        ld      de,wallMaskTable                ; DE => mask table to use for wall flags
        ld      hl,wallOffsetTable              ; HL => coord offset table to find all surrounding tile positions
        ld      a,8                             ; A = 8 surrounding tiles to check
findSurroundingWallsLoop:
        pop     bc                              ; get our centre x/y pos
        push    bc                              ; .. and put it back on the stack for later
        push    af                              ; save the loop counter
        push    de                              ; save mask table ptr
        ld      a,b
        add     a,(hl)                          ; apply x offset
        ld      b,a                             ; B = x pos to check
        inc     hl
        ld      a,c
        add     a,(hl)                          ; apply y offset
        ld      c,a                             ; C = y pos to check
        inc     hl
        push    hl                              ; save offset table ptr
        call    checkWall                       ; check tile to see if it's a wall
        pop     hl                              ; restore offset table ptr
        pop     de                              ; restore mask table ptr
        jr      nc,endFSWLoop
        ld      a,(de)                          ; A = bit mask to OR
#ifndef APP83P
        or      $00                             ; apply it with the flags byte (self-modifying code)
__wallFlags             = $-1
#else
        ld      bc,(__wallFlags)                ; C = (__wallFlags)
        or      c
#endif
        ld      (__wallFlags),a                 ; save updated flags
endFSWLoop:
        inc     de                              ; update mask table ptr
        pop     af                              ; restore loop counter
        dec     a
        jr      nz,findSurroundingWallsLoop     ; loop until it reaches 0

        ld      a,(__wallFlags)                 ; get flag byte
        and     $0F                             ; first we want to deal with 4 lsb
        ld      e,a
        ld      d,0
        ld      hl,wallTable
        add     hl,de                           ; HL => offset value for tile
        ld      a,(hl)                          ; A = offset value
#ifndef APP83P
        ld      hl,0                            ; HL => tile in tile data (set from earlier)
__wallTilePtr           = $-2
#else
        ld      hl,(__wallTilePtr)
#endif
        add     a,(hl)                          ; A = new tile value
        ld      (hl),a                          ; set it
        sub     TILE_COMPLEXWALL                ; check if new tile is a "complex" wall
        jr      c,endCWLoop                     ; if not, we're done processing this tile
; otherwise, we need to check the flags of the diagonals
        push    hl                              ; save tile data ptr
        ld      e,a
        ld      d,0                             ; DE = offset for wallComplexTable
        ld      hl,wallComplexTable
        add     hl,de                           ; HL => mask to apply to check correct diagonal block flag bits
        ld      a,(__wallFlags)                 ; A = flag byte
        cpl                                     ; invert it
        and     (hl)                            ; apply mask to it
        pop     hl                              ; restore tile data ptr
        jr      nz,endCWLoop                    ; if any applicable corner blocks were not walls, leave the tile alone
; otherwise, increase wall tile by 5 to get it's counterpart, which makes the wall tile the "more solid" version
        ld      a,5
        add     a,(hl)                          ; A = new wall tile
        ld      (hl),a                          ; save it

endCWLoop:
        pop     bc                              ; restore x/y pos
        dec     b                               ; dec x pos
        bit     7,b
        jr      z,calcWallsLoop2                ; if x hasn't gone lower than 0, loop again

        push    bc                              ; save y pos
#ifndef APP83P
        ld      a,%11100000                     ; moving pixels to display (similar to run indicator)
__loadingIndicator       = $-1
#else
        ld      a,(__loadingIndicator)
#endif
        ld      hl,gbuf+768-1
        ld      (hl),a                          ; write the byte
        rrca                                    ; rotate the pixel right
        ld      (__loadingIndicator),a          ; save it for next time
        call    libFastCopy                     ; update lcd
        pop     bc                              ; restore y pos

        dec     c                               ; dec y pos
        bit     7,c
#ifndef APP83P
        jr      z,calcWallsLoop1                ; if y hasn't gone lower than 0, loop again
#else
        jp      z,calcWallsLoop1
#endif
        ld      hl,gbuf+768-1
        ld      (hl),$00                        ; clear loading indicator
        ld      hl,str2nd
        ld      de,57*256+77
        call    vPuts                           ; "[2nd]"
        call    libFastCopy
        call    wait2nd
; fall through to drawLevel

;------------------------------------------------
; drawLevel - draw the level and scroll if necessary to centre the player
;
; input:    none
; output:   none
;------------------------------------------------
drawLevel:
        bcall(_grbufclr)
; calc left side of screen
        ld      a,(startX)
        sub     5
        jr      c,startDrawLeft                 ; if close to left edge of level, start drawing from left edge
        add     a,11-1
        ld      hl,(levelWidth)                 ; L = level width
        cp      l
        jr      nc,startDrawRight               ; if close to right edge of level, start drawing from right edge - 11
        sub     11-1                            ; otherwise, player will be centred
saveScreenX:
        ld      (screenX),a                     ; screenX is currently only stored as 1 byte block coord, later it will be converted to absolute pixel value
; calc top side of screen
        ld      a,(startY)
        sub     3
        jr      c,startDrawTop                  ; if close to top edge of level, start drawing from top edge
        add     a,7-1
        ld      hl,(levelHeight)                ; L = level height
        cp      l
        jr      nc,startDrawBottom              ; if close to bottom edge of level, start drawing from bottom edge - 7
        sub     7-1                             ; otherwise, player will be centred
saveScreenY:
        ld      (screenY),a                     ; screenY is currently only stored as 1 byte block coord, later it will be converted to absolute pixel value
        jr      afterCalcScreenXY               ; skip past sub routines to next step
startDrawLeft:
        xor     a
        jr      saveScreenX
startDrawRight:
        ld      a,l                             ; A = level width
        sub     12
        jr      saveScreenX
startDrawTop:
        xor     a
        jr      saveScreenY
startDrawBottom:
        ld      a,l                             ; A = level height
        sub     8
        jr      saveScreenY

afterCalcScreenXY:
; start drawing level
        xor     a
        ld      (__dlY),a                       ; reset y offset / counter
drawLevelNextY:
        xor     a
        ld      (__dlX),a                       ; reset x offset / counter
        ld      a,(__dlY)
        ld      hl,screenY
        add     a,(hl)                          ; A = top of screen y + offset
        ld      hl,(levelWidth-1)               ; H = level width
        ld      l,a                             ; L = top of screen y + offset
        bcall(_htimesl)                         ; HL = offset to row in tile data
        ld      de,(tileDataPtr)
        add     hl,de                           ; HL => row in tile data
        ld      de,(screenX)
        ld      d,0
        add     hl,de                           ; HL => where in tile data to start drawing this row from
drawLevelNextX:
        push    hl                              ; save pointer to current position in tile data
        ld      l,(hl)
        ld      h,0                             ; HL = tile number
        add     hl,hl
        add     hl,hl
        add     hl,hl
        ld      de,startTiles
        add     hl,de                           ; HL => tile sprite
        ex      de,hl                           ; DE => tile sprite
#ifndef APP83P
        ld      l,$00
__dlY                   = $-1
        ld      a,$00
__dlX                   = $-1
#else
        ld      hl,(__dlY)                      ; L = (__dlY)
        ld      a,(__dlX)
#endif
        call    putAlignedSprite                ; draw tile
        pop     hl                              ; restore pointer to current position in tile data
        inc     hl                              ; and increment it
        ld      a,(__dlX)
        inc     a                               ; increment x offset / counter
        ld      (__dlX),a
        cp      12                              ; have we reached the end of this row?
        jr      c,drawLevelNextX                ; if not, do another tile
        ld      a,(__dlY)
        inc     a                               ; increment y offset / counter
        ld      (__dlY),a
        cp      8                               ; have we reached the bottom of the screen?
        jr      c,drawLevelNextY                ; if not, do another row

; calculate player start position on screen
        ld      a,(startX)                      ; A = start x pos (block value)
        ld      hl,screenX                      ; HL => screen x pos (block value)
        sub     (hl)                            ; A = player x pos on screen (block value)
        add     a,a
        add     a,a
        add     a,a                             ; A = player x pos on screen absolute value
        ld      (__playerScrX),a                ; save it
        inc     hl
        inc     hl                              ; HL => screen y pos (block value)
        ld      a,(startY)
        sub     (hl)
        add     a,a
        add     a,a
        add     a,a
        ld      (__playerScrY),a                ; ditto for player y pos on screen

; calc screenX & screenY from block values out to absolute pixel values
        ld      hl,(screenX)
        ld      h,0
        add     hl,hl
        add     hl,hl
        add     hl,hl
        ld      (screenX),hl
        ld      hl,(screenY)
        ld      h,0
        add     hl,hl
        add     hl,hl
        add     hl,hl
        ld      (screenY),hl

; calc absolute player coords from startX & startY
        ld      hl,(startX)
        ld      h,0
        add     hl,hl
        add     hl,hl
        add     hl,hl
        ld      (playerX),hl
        ld      hl,(startY)
        ld      h,0
        add     hl,hl
        add     hl,hl
        add     hl,hl
        ld      (playerY),hl

; check if should scroll screen to the right
        ld      a,(__playerScrX)                ; A = player x pos on screen
        cp      48                              ; if player x pos on screen is 48 or more,
        jr      nc,afterDLScrollRight           ; then we're near the right edge of the level, so don't scroll
; try to scroll right 4 times
        ld      b,4
drawLevelScrollRight:
        push    bc
        call    scrollRight
        pop     bc
        djnz    drawLevelScrollRight

afterDLScrollRight:
; check if should scroll screen down
        ld      a,(__playerScrY)                ; A = player y pos on screen
        cp      32                              ; if player y pos on screen is 32 or more,
        ret     nc                              ; then we're near the bottom edge of the level, so don't scroll
; try to scroll down 4 times
        ld      b,4
drawLevelScrollDown:
        push    bc
        call    scrollDown
        pop     bc
        djnz    drawLevelScrollDown
        ret

;------------------------------------------------
; getTile - get a tile given the exact coordinates
;
; input:    DE = x
;           HL = y
; output:   A = tile number
;           HL => tile in tile data
;------------------------------------------------
getTile:
        push    de                              ; save x coord
        ld      a,8
        bcall(_divhlbya)                        ; HL = block y coord (H will be clear)
        ld      a,(levelWidth)
        ld      h,a
        bcall(_htimesl)                         ; HL = offset in tile data to row
        ld      de,(tileDataPtr)
        add     hl,de                           ; HL => row in tile data
        ex      (sp),hl                         ; HL = x coord; pointer to row in tile data now on stack
        ld      a,8
        bcall(_divhlbya)                        ; HL = block x coord
        pop     de                              ; DE => row in tile data
        add     hl,de                           ; HL => tile in tile data
        ld      a,(hl)                          ; A = tile number
        ret

;------------------------------------------------
; putTile - put a tile into a level and draw it on the screen
;
; input:    HL => tile location in level data
;           A = tile number to put
; output:   HL => tile location in level data
;------------------------------------------------
putTile:
        push    hl                              ; save data ptr
        ld      (hl),a                          ; save new tile number in level data
        push    af                              ; save tile num
        call    getBlockXYFromHL                ; A = x; L = y; H = 0; ie. HL = y
        add     hl,hl
        add     hl,hl
        add     hl,hl
        ld      de,(screenY)
        or      a                               ; clear carry
        sbc     hl,de                           ; L = y pos to display tile at on screen, H = 0
        ld      c,l                             ; C = y pos to display tile at on screen
        ld      l,a                             ; HL = x coord in level of tile
        add     hl,hl
        add     hl,hl
        add     hl,hl
        ld      de,(screenX)
        or      a
        sbc     hl,de
        ld      b,l                             ; B = x pos to display tile at on screen
        push    bc                              ; save x,y draw pos
        ld      hl,sprWhiteSpace
        call    putClippedMaskedSprite          ; clear the 8x8 position on screen where new tile will be displayed
        pop     bc                              ; restore x,y draw pos
        pop     af                              ; restore tile num to draw
        ld      l,a
        ld      h,0
        add     hl,hl
        add     hl,hl
        add     hl,hl
        ld      de,startTiles
        add     hl,de                           ; HL => tile sprite
        call    putClippedSprite                ; draw tile sprite
        pop     hl                              ; restore data ptr
        ret

;------------------------------------------------
; getBlockXYFromHL - get x & y tile coords from a pointer to a tile in level data
;
; input:    HL => tile in level data
; output:   A = x (block value)
;           L = y (block value)
;------------------------------------------------
getBlockXYFromHL:
        ld      de,(tileDataPtr)                ; DE => start of level tile data
        or      a                               ; clear carry
        sbc     hl,de                           ; HL = offset to tile in level data
        ld      a,(levelWidth)
        bcall(_divhlbya)                        ; L = dividend (y); A = remainder (x)
        ret

;------------------------------------------------
; getExactXYFromHL - get x & y absolute coords a from a pointer to a tile in level data
;
; input:    HL => tile in level data
; output:   DE = x
;           HL = y
;------------------------------------------------
getExactXYFromHL:
        call    getBlockXYFromHL                ; get the block values first
        ld      h,0                             ; HL = y (block)
        add     hl,hl
        add     hl,hl
        add     hl,hl                           ; HL = y (absolute)
        ex      de,hl                           ; DE = y (absolute)
        ld      l,a
        ld      h,0                             ; HL = x (block)
        add     hl,hl
        add     hl,hl
        add     hl,hl                           ; HL = x (absolute)
        ex      de,hl                           ; DE = x; HL = y (both absolute)
        ret

;------------------------------------------------
; checkWall - check block coords to see if tile is a wall tile
;
; input:    B = block x coord
;           C = block y coord
; output:   HL => tile in tile data
;           CA = is a wall tile
; notes:    if x,y pos is outside bounds of the level, will always return NC
;           x,y pos can only be outside of level by one block
;------------------------------------------------
checkWall:
        ld      a,b
        or      c                               ; A = x & y OR'd together
        bit     7,a                             ; check if either was negative
        jr      nz,checkWallBoundError
        ld      a,(levelWidth)
        cp      b                               ; compare level width to x pos
        jr      z,checkWallBoundError           ; if they're equal, outside level
        ld      a,(levelHeight)
        cp      c                               ; ditto for height & y pos
        jr      z,checkWallBoundError
        ld      l,b
        ld      h,0
        add     hl,hl
        add     hl,hl
        add     hl,hl
        ex      de,hl                           ; DE = absolute x pos
        ld      l,c
        ld      h,0
        add     hl,hl
        add     hl,hl
        add     hl,hl                           ; HL = absolute y pos
        call    getTile                         ; A = tile
        cp      TILE_FIRSTWALL                  ; is it higher than the first wall tile?
        ccf                                     ; complement carry
        ret     nc                              ; leave if tile isn't a wall
        cp      TILE_LASTWALL+1                 ; is it higher than the last wall tile?
        ret                                     ; ret regardless with the NC/CA result
checkWallBoundError:
        or      a                               ; clear carry
        ret

;------------------------------------------------
; checkE2Trigger - check if a tile is an enemy2 trigger, and if so, check if it should be represented as a bare floor tile
;
; input:    HL => tile in level data
; output:   HL = tile to work with
;           A = tile to work with
;------------------------------------------------
checkE2Trigger:
        ld      a,(hl)                          ; A = tile
        cp      TILE_ENEMYTRIGGER+1             ; check if enemy2 trigger
        jr      nz,checkE2TriggerDone           ; if not, nothing to check
        ld      d,a                             ; save tile type
        inc     hl
        ld      a,(hl)                          ; A = tile next to the trigger
        or      a                               ; check if it's a bare floor tile
        ld      a,d                             ; A = original tile
        jr      nz,checkE2TriggerDone           ; if the next tile isn't a bare floor tile, stick with original tile
        xor     a                               ; otherwise, we'll display the enemy tile as a bare floor tile
checkE2TriggerDone:
        ld      l,a
        ld      h,0                             ; HL = tile to display
        ret

;------------------------------------------------
; showObjectives - show mission objectives
;
; input:    none
; output:   HL => level header after mission objectives
;------------------------------------------------
showObjectives:
        ld      hl,(tempProg)                   ; HL => start of uncompressed level data, which is the mission objective text
        jp      displayPage                     ; display page of text and ret

;------------------------------------------------
; openDoor - open a normal door
;
; input:    A = closed door tile
;           HL => tile in level data
; output:   
;------------------------------------------------
openDoor:
        sub     OPEN_DOOR                       ; A = new tile number for open door
        push    af                              ; save tile number
        call    putTile                         ; put new tile & draw it
        call    getExactXYFromHL                ; DE = x; HL = y (for new animation)
        pop     af                              ; A = open door tile number
        sub     ANIM_DOOR                       ; A = animation type
        ld      b,a                             ; B = animation type
        ld      c,2                             ; C = initial counter value
        jp      newAnim                         ; set up new animation & ret

;------------------------------------------------
; closeFireDoor - close a fire door
;
; input:    DE => tile in level data
; output:   none
;------------------------------------------------
closeFireDoor:
        ex      de,hl                           ; HL => tile in level data
        ld      a,(hl)                          ; A = open fire door tile
        add     a,CLOSE_FIRE_DOOR               ; A = closed fire door tile
        call    putTile                         ; put tile & display it
        ld      hl,levelFlags                   ; HL => level flags
        bit     LFLAG_FIREDOORS,(hl)            ; check if fire doors affect objectives counter
        ret     z                               ; if not, leave
; otherwise, fall through to lowerObjectivesCount

;------------------------------------------------
; lowerObjectivesCount - try to lower the objectives counter
;
; input:    none
; output:   none
;------------------------------------------------
lowerObjectivesCount:
        ld      hl,objectives                   ; HL => counter
        ld      a,(hl)
        cp      2                               ; is it already below 2?
        ret     c                               ; if so, can't lower it
        dec     (hl)                            ; otherwise, lower it
        ld      a,(hl)                          ; A = new value
        dec     a                               ; is it now 1 (destruct sequence)?
        ret     nz                              ; if not, leave now
; otherwise, put "destruct" block
putDestructBlock:
        ld      hl,(destructTileOffset)         ; HL => offset to tile in level data
        ld      de,(tileDataPtr)                ; DE => start of tile level data
        add     hl,de                           ; HL => tile in level data
        ld      a,(destructTileType)            ; A = tile to put
        jp      putTile                         ; putTile & ret

;------------------------------------------------
; showTimer - show countdown timer in custom font
;
; input:    none
; output:   none
;------------------------------------------------
showTimer:
; first, clear section of gbuf where timer will be displayed
        ld      hl,gbuf+12                      ; HL => second line of gbuf
        ld      de,11                           ; DE = bytes to skip to next gbuf line
        ld      b,9                             ; B = number of lines to clear portion on
showTimerClear:
        ld      a,(hl)
        and     %10000000                       ; clear 7 bits
        ld      (hl),a
        inc     hl
        ld      a,(hl)
        and     %00001111                       ; clear 4 bits
        ld      (hl),a
        add     hl,de                           ; HL => start of next gbuf line
        djnz    showTimerClear                  ; repeat B times
        ld      hl,(time)
        ld      h,0                             ; HL = timer value
        bcall(_divHLby10)                       ; HL = first character to show; A = second
        push    af                              ; save 2nd char
        ld      a,2
        call    showTimerChar                   ; show first char
        pop     af
        ld      l,a
        ld      h,0
        ld      a,7
; fall through to showTimerChar

;------------------------------------------------
; showTimerChar - show a timer character in custom font
;
; input:    HL = number to show
;           A = x pos to show it at
; output:   none
;------------------------------------------------
showTimerChar:
        add     hl,hl
        add     hl,hl
        add     hl,hl                           ; HL = offset to sprite
        ld      de,sprTimerNumbers
        add     hl,de                           ; HL => sprite
        ex      de,hl
        ld      l,2                             ; y pos
        jp      putSprite                       ; putSprite and ret

;------------------------------------------------
; levelExplode - do multiple explosion animations and then game is over
;
; input:    none
; output:   none
;------------------------------------------------
levelExplode:
; do the level exploding animation, which comprises of the following:
; - shake the screen by directly accessing the lcd driver
; - fade out
; - fade back in with the level without sprites
; - do lots of explosion animations
; - go to game over screen
        ld      c,15                            ; C = shake screen 15 times
shakeScreenLoop:
        ld      a,$42                           ; screen offset of 2
        out     ($10),a                         ; send instruction to lcd
        ld      b,10
        call    wait
        ld      a,$40                           ; screen offset of 0
        out     ($10),a
        ld      b,10
        call    wait
        dec     c                               ; dec shake counter
        jr      nz,shakeScreenLoop
        call    fadeOut                         ; fade screen out
        ld      b,20
        call    wait                            ; wait a bit
        call    restoreVideo                    ; restore gbuf without sprites drawn
        call    libFastCopy
        call    fadeIn                          ; fade it back in
        call    clearAnimTable                  ; clear animTable
        ld      b,50                            ; B = 50 frames to loop showing explosions
levelExplosionLoop:
        push    bc                              ; save frame counter
        call    updateAnims                     ; update all explosions
        call    randomExplosion                 ; make a new random explosion on screen
        call    saveVideo                       ; back up gbuf with no explosion sprites
        call    drawAnims                       ; draw animations
        call    libFastCopy
        call    restoreVideo                    ; restore gbuf without sprites
        ld      b,10
        call    wait
        pop     bc                              ; restore frame counter
        djnz    levelExplosionLoop              ; loop
        jp      gameOver

;------------------------------------------------
; randomExplosion - generate a random on screen explosion
;
; input:    none
; output:   none
;------------------------------------------------
randomExplosion:
        ld      b,96-8
        call    libRandom                       ; get a random x offset
        ld      hl,(screenX)                    ; HL = x pos of left of screen
        ld      e,a
        ld      d,0                             ; DE = random x offset
        add     hl,de
        ex      de,hl                           ; DE = x pos for new explosion
        ld      b,64-8
        call    libRandom                       ; get a random y offset
        ld      hl,(screenY)
        ld      c,a
        ld      b,0
        add     hl,bc                           ; HL = y pos for new explosion
        ld      bc,0*256+4                      ; B = animation type; C = counter ini
        jp      newAnim                         ; set up new explosion & ret

;------------------------------------------------
; drawDarkMask - draw dark mask overlay for "dark" levels
;
; input:    none
; output:   none
;------------------------------------------------
drawDarkMask:
        ld      hl,gbuf
        ld      a,(__playerScrY)                ; A = player y pos on gbuf
        sub     22                              ; A = # full $FF lines to draw at top of gbuf
        push    af                              ; save for later
        jr      c,ddmAfterTopLines              ; if less than 0, no $FF lines at top
        jr      z,ddmAfterTopLines              ; same if 0
        call    ddmFullLines                    ; draw A full lines at (HL) on gbuf
ddmAfterTopLines:
; at this point, HL = gbuf ptr; A = mask y pos
        ex      de,hl                           ; DE = gbuf ptr
        ld      hl,darkMaskData                 ; HL => start of mask data
        bit     7,a                             ; check if mask starts above top of screen
        jr      nz,ddmNearTop                   ; if so, player is near the top of screen
        cp      13                              ; check if mask is going to overflow off bottom of screen
        jr      nc,ddmNearBottom                ; if so, player is near the bottom of screen
; otherwise, player is near centre, so the full mask will be used on the y axis
        ld      a,52                            ; A = 52 mask lines to draw
        jr      ddmStartMaskLines
ddmNearTop:
        push    af                              ; save mask y pos (is a negative value)
        neg                                     ; make it positive
        push    hl                              ; save mask data ptr
        push    de                              ; save gbuf ptr
        ld      l,a                             ; L = lines to skip
        ld      h,7                             ; H = 7 bytes per line
        bcall(_htimesl)                         ; HL = offset for darkMaskData
        pop     de                              ; restore gbuf ptr
        pop     bc                              ; BC => start of darkMaskData
        add     hl,bc                           ; HL => correct starting line of darkMaskData
        pop     af                              ; restore mask y pos (negative value)
        add     a,52                            ; A = # mask lines to draw
        jr      ddmStartMaskLines
ddmNearBottom:
        sub     64
        neg                                     ; A = # mask lines to draw
ddmStartMaskLines:
; at this point, DE = gbuf ptr, HL = darkMaskData ptr, A = # lines to use from darkMaskData
        ld      (__ddmLineCounter),a            ; save for later
        ld      a,(__playerScrX)                ; A = player x pos on gbuf
        ld      b,a                             ; save it to B
        srl     a
        srl     a
        srl     a                               ; A = block x pos of player on gbuf
        sub     3                               ; A = number of $FF bytes to have at left of circle mask
        ld      (__ddmLeftOffset),a             ; save it
        ld      a,b                             ; A = player x pos on gbuf
        and     $07                             ; A = number of bits to shift right when applying mask
        ld      (__ddmsShiftCounter),a          ; save it for later
#ifndef APP83P
        ld      bc,$0000                        ; set earlier, B = # lines of mask to draw; C = # $FF bytes to draw at left of each line
__ddmLineCounter        = $-1
__ddmLeftOffset         = $-2
#else
        ld      bc,(__ddmLeftOffset)
#endif
        ex      de,hl                           ; DE = darkMaskData ptr; HL = gbuf ptr

ddmShiftMaskLines:
; make a 23 byte buffer for the line
        push    bc                              ; save line counter & left bytes counter
        push    de                              ; save darkMaskData ptr
        push    hl                              ; save gbuf ptr
        push    af                              ; save shift counter
        ld      hl,maskBuffer                   ; HL => where to store 23 byte buffer
        ld      a,c                             ; A = # $FF bytes at left
        call    ddmWrite8FFBytes                ; put 8 x $FF bytes at start of buffer
        ex      de,hl                           ; DE = maskBuffer ptr; HL = darkMaskData ptr
        ld      bc,7                            ; BC = 7 bytes to copy
        ldir                                    ; copy BC bytes from (HL) to (DE)
        ex      de,hl                           ; HL = maskBuffer ptr
        call    ddmWrite8FFBytes                ; put 8 x $FF bytes at end of buffer
        neg
        add     a,8                             ; A = offset to use for maskBuffer
        ld      e,a
        ld      d,0
        ld      hl,maskBuffer
        add     hl,de                           ; HL => start of portion of maskBuffer that we'll be using
        pop     af                              ; A = shift counter
        or      a                               ; check if no shifts required
        jr      z,ddmsAfterShift                ; if so, skip shifting
        ld      c,a                             ; put it in C
ddmsShiftRight:
        push    hl                              ; save maskBuffer ptr
        ld      b,2                             ; B = 2 lots of 6 bytes to shift each time
        dec     hl                              ; for the first shift, we need bit from the preceding byte
        rr      (hl)                            ; rotate 1st byte right
        inc     hl                              ; update ptr
ddmsShiftLoop:
        rr      (hl)
        inc     hl
        rr      (hl)
        inc     hl
        rr      (hl)
        inc     hl
        rr      (hl)
        inc     hl
        rr      (hl)
        inc     hl
        rr      (hl)
        inc     hl
        djnz    ddmsShiftLoop
        pop     hl                              ; restore maskBuffer ptr
        dec     c
        jr      nz,ddmsShiftRight               ; rotate the buffer right C times
ddmsAfterShift:
        ex      de,hl                           ; DE => start of portion of maskBuffer that we're using
        pop     hl                              ; HL = gbuf ptr
        ld      b,12                            ; B = 12 bytes of bit-shifted maskBuffer to use
ddmsApplyMask:
        ld      a,(de)                          ; A = mask data
        or      (hl)                            ; apply gbuf byte to it
        ld      (hl),a                          ; and write it back to gbuf
        inc     hl                              ; update gbuf ptr
        inc     de                              ; update darkMaskData ptr
        djnz    ddmsApplyMask                   ; loop 12 times
        ex      de,hl                           ; DE = gbuf ptr
        pop     hl                              ; HL = darkMaskData ptr
        ld      bc,7
        add     hl,bc                           ; HL => next line of darkMaskData
        ex      de,hl                           ; DE = updated darkMaskData ptr; HL = gbuf ptr
        pop     bc                              ; restore line counter & left bytes counter
#ifndef APP83P
        ld      a,$00                           ; A = shift counter (saved from earlier)
__ddmsShiftCounter      = $-1
#else
        ld      a,(__ddmsShiftCounter)
#endif
        djnz    ddmShiftMaskLines               ; loop B times

; draw full $FF lines at bottom of gbuf
        pop     bc                              ; B = # lines drawn at top of gbuf
        ld      a,12
        sub     b                               ; A = # full $FF lines to draw at bottom of gbuf
        ret     z                               ; if 0, no $FF lines at bottom
        bit     7,a                             ; check if negative
        ret     nz                              ; if less than 0, no $FF lines at bottom
; fall through to ddmFullLines
ddmFullLines:
; draw A full lines at (HL) on gbuf
        push    hl                              ; save gbuf ptr
        ld      h,12
        ld      l,a
        bcall(_htimesl)                         ; HL = bytes to write $FF to
        ld      b,h
        ld      c,l
        dec     bc                              ; BC = bytes to write minus 1
        pop     hl                              ; restore gbuf ptr
        ld      d,h
        ld      e,l
        inc     de                              ; DE = HL + 1
        ld      (hl),$FF                        ; write first byte
        ldir                                    ; and copy to the rest of them
        inc     hl                              ; HL => next line of gbuf
        ret
ddmWrite8FFBytes:
        ld      b,8                             ; B = 8 bytes to write
ddmWrite8FFLoop:
        ld      (hl),$FF                        ; write $FF byte
        inc     hl                              ; update gbuf ptr
        djnz    ddmWrite8FFLoop
        ret

.end
