;===============================================================;
;                                                               ;
; Alien Breed 5                                                 ;
; Intex routines                                                ;
;                                                               ;
;===============================================================;

;------------------------------------------------
; loadIntexConsole - start the intex console routines
;
; input:    none
; output:   none
;------------------------------------------------
loadIntexConsole:
        call    saveVideo                       ; save video buffer
        ld      hl,strIntexLoad                 ; HL => loading text data
        call    displayPage
        ld      b,40                            ; B = wait count
        call    wait

;------------------------------------------------
; intexMainMenu - prepare for the intex main menu
;
; input:    none
; output:   HL => main menu data
;           DE => jump table
;------------------------------------------------
intexMainMenu:
        ld      hl,intexMainMenuData
        ld      de,intexMainJumpTable
        ld      a,$B7                           ; A = "or a"
        jr      doIntexMenu

;------------------------------------------------
; intexWeaponMenu - prepare for the intex weapon menu
;
; input:    none
; output:   HL => weapon menu data
;           DE => jump table
;------------------------------------------------
intexWeaponMenu:
        ld      hl,intexWeaponMenuData
        ld      de,intexWeaponJumpTable
        ld      a,$37                           ; A = "scf"
        jr      doIntexMenu

;------------------------------------------------
; intexToolMenu - prepare for the intex tool menu
;
; input:    none
; output:   HL => tool menu data
;           DE => jump table
;------------------------------------------------
intexToolMenu:
        ld      hl,intexToolMenuData
        ld      de,intexToolJumpTable
        ld      a,$37                           ; A = "scf"
; fall through to menu routine

;------------------------------------------------
; doIntexMenu - execute an intex menu
;
; input:    HL => menu data
;           DE => jump table
; output:   none
;------------------------------------------------
doIntexMenu:
        ld      (__intexShowCredits),a          ; set either "or a" or "scf"
        ld      (__intexJumpTable),de           ; save jump table location
        ld      a,(hl)                          ; A = number of options for this menu
        ld      (__intexOptions),a              ; save it (self modifying code)
        inc     hl                              ; HL => text data
        call    displayPage                     ; display all text data
        ld      (__intexCursorTable),hl         ; save ptr to cursor position table
        xor     a
        ld      (__intexMenuItem),a             ; reset to top menu item
#ifndef APP83P
        nop                                     ; either "or a" or "scf" (will be set at start of this routine)
__intexShowCredits      = $-1
#else
        call    __intexShowCredits
#endif
        jr      nc,intexMenuLoop                ; if NC, skip showing credit limit
        ld      hl,strCreditLimit
        call    displayPageLoop                 ; type out "CREDIT LIMIT:" without clearing gbuf
        ld      hl,(credits)
        ld      de,56*256+50
        call    setShowHL                       ; show credits

intexMenuLoop:
        call    drawIntexCursor                 ; draw cursor
        call    libFastCopy                     ; update gbuf to lcd
        call    drawIntexCursor                 ; erase cursor
        call    waitKey                         ; wait for a key; A = key pressed
        ld      hl,__intexMenuItem              ; HL => selected menu item
        cp      GK_UP                           ; [UP]?
        jr      z,intexUp                       ; if so, move up
        cp      GK_DOWN                         ; [DOWN]?
        jr      z,intexDown                     ; if so, move down
        cp      GK_2ND                          ; [2nd]?
        jr      nz,intexMenuLoop                ; if not, loop again
        ld      hl,(__intexMenuItem)            ; HL = menu item
#ifdef APP83P
        ld      h,0
#endif
        add     hl,hl                           ; *2
#ifndef APP83P
        ld      de,$0000                        ; DE => start of jump table (set from earlier)
__intexJumpTable        = $-2
#else
        ld      de,(__intexJumpTable)
#endif
        add     hl,de                           ; HL => jump table entry
        bcall(_ldhlind)                         ; HL = routine
        jp      (hl)                            ; jump to appropriate routine

intexUp:
        ld      a,(hl)                          ; A = selected menu item
        or      a                               ; already at the top?
        jr      z,intexMenuLoop                 ; if so, can't go up, just return to loop
        dec     (hl)                            ; dec selected menu item
        jr      intexMenuLoop
intexDown:
#ifndef APP83P
        ld      a,$00                           ; A = number of menu options (set from earlier)
__intexOptions          = $-1
#else
        ld      a,(__intexOptions)
#endif
        dec     a                               ; A = last menu item
        cp      (hl)                            ; is this what player is currently on?
        jr      z,intexMenuLoop                 ; if so, can't go down, just return to loop
        inc     (hl)                            ; inc selected menu item
        jr      intexMenuLoop

;------------------------------------------------
; intexRadarService - show the level map
;
; input:    none
; output:   none
;------------------------------------------------
intexRadarService:
        ld      hl,intexMainMenu                ; where to ret to after showing map
        push    hl                              ; push it to stack
showMap:
        bcall(_grbufclr)
        ld      de,(tileDataPtr)                ; DE => start of level tile data
        ld      hl,gbuf                         ; HL => start of gbuf
        ld      a,(levelHeight)
        ld      b,a                             ; B = number rows to draw
showMapRow:
        push    bc                              ; save row counter
        push    hl                              ; save gbuf ptr
        ld      a,(levelWidth)
        ld      b,a                             ; B = number of pixels to draw in this row
        ld      c,%10000000                     ; C = pixel drawing tool
showMapPixel:
        ld      a,(de)                          ; A = tile
        inc     de                              ; DE => next tile
        cp      TILE_SOLID                      ; is it a solid tile?
        jr      c,afterDrawMapPixel             ; if not, don't draw a pixel
        cp      TILE_DOOR                       ; is it a normal door?
        jr      nc,afterDrawMapPixel            ; if so, don't draw a pixel
        ld      a,c                             ; A = pixel drawing tool
        or      (hl)                            ; A = new gbuf byte with new pixel OR'd on
        ld      (hl),a                          ; save it to gbuf
afterDrawMapPixel:
        ld      a,c                             ; A = pixel drawing tool
        rrca                                    ; rotate pixel to the right, set carry if it overflows
        ld      c,a                             ; C = pixel drawing tool with new pixel position
        jr      nc,afterNextMapByte             ; if pixel bit didn't overflow, skip the gbuf inc
        inc     hl                              ; HL => next byte on gbuf
afterNextMapByte:
        djnz    showMapPixel                    ; loop to show another pixel until row is finished
        push    de                              ; save level tile data ptr
        call    libFastCopy                     ; copy gbuf to lcd after each byte to give a "printing" effect
        ld      b,3
        call    wait
        pop     de                              ; restore level tile data ptr
        pop     hl                              ; restore gbuf ptr (to start of row we just finished)
        ld      bc,12
        add     hl,bc                           ; HL => next gbuf row
        pop     bc                              ; restore row counter
        djnz    showMapRow                      ; loop to show another row until all rows are done
showMapFlashLoop:
; use a bit of a hack of other routines to get position to draw marker
        ld      de,(playerX)
        inc     de
        inc     de
        inc     de                              ; DE = x coord of centre of player
        ld      hl,(playerY)
        inc     hl
        inc     hl
        inc     hl                              ; HL = y coord of centre of player
        call    getTile                         ; HL => tile location in level data of where player is standing
        call    getBlockXYFromHL                ; A = x (block value); L = y (block value)
        dec     a                               ; A = x-1
        dec     l                               ; L = y-1 (3x3 marker)
        ld      de,sprMapMarker                 ; DE => sprite
        call    putSprite                       ; XOR sprite to gbuf
        call    libFastCopy                     ; update gbuf to lcd
        ld      b,40                            ; B = delay value
showMapFlashDelay:
        ei
        halt
        push    bc                              ; save delay value
        bcall(_getcsc)                          ; A = any key press
        pop     bc
        cp      GK_2ND                          ; [2nd] pressed?
        ret     z                               ; if so, ret from routine
        djnz    showMapFlashDelay               ; continue with delay until B=0
        jr      showMapFlashLoop                ; XOR the marker & repeat

;------------------------------------------------
; intexMissionObjective - show the mission objectives from the intex console
;
; input:    none
; output:   none
;------------------------------------------------
intexMissionObjective:
        call    showObjectives                  ; run the routine to show the mission objectives
        call    wait2nd                         ; wait for [2nd]
        jp      intexMainMenu                   ; return to main menu

;------------------------------------------------
; intexEntertainment - show the intex entertainment lols tribute to ABSE
;
; input:    none
; output:   none
;------------------------------------------------
intexEntertainment:
        ld      hl,strIntexEntertainment        ; HL => string data
        call    displayPage                     ; display it
        call    waitKey
        jp      intexMainMenu                   ; return to main menu

;------------------------------------------------
; intexInfoBase - show the info base
;
; input:    none
; output:   none
;------------------------------------------------
intexInfoBase:
        ld      hl,intexInfoBaseData
        ld      b,(hl)                          ; B = number of pages of info base data
        inc     hl                              ; HL => actual text data
infoBaseLoop:
        push    bc                              ; save counter
        call    displayPage
        pop     bc                              ; restore counter
infoBaseKeyLoop:
        push    bc                              ; save counter
        push    hl                              ; save data ptr
        call    waitKey
        pop     hl                              ; restore data ptr
        pop     bc                              ; restore counter
        cp      GK_MODE                         ; [MODE]?
        jr      z,infoBaseDone                  ; if so, info base is done
        cp      GK_2ND                          ; [2nd]?
        jr      nz,infoBaseKeyLoop              ; if not, wait for another key
        djnz    infoBaseLoop                    ; loop to next page
infoBaseDone:
        jp      intexMainMenu                   ; return to main menu

;------------------------------------------------
; intexAbort - leave the intex console
;
; input:    none
; output:   none
;------------------------------------------------
intexAbort:
        ld      hl,strIntexAbort                ; HL => leaving intex text data
        call    displayPage
        ld      b,30                            ; B = wait count
        call    wait
        jp      restoreVideo                    ; restore video buffer & lead

;------------------------------------------------
; intexBuy@WEAPON@ - try to buy said weapon
;
; input:    none
; output:   none
;------------------------------------------------
intexBuyTwinLaser:
        ld      b,%00000010                     ; B = bit mask for weaponTable
        ld      de,CR_TWINLASER                 ; DE = cost
        jr      intexBuyWeapon
intexBuyFlameArc:
        ld      b,%00000100
        ld      de,CR_FLAMEARC
        jr      intexBuyWeapon
intexBuyPlasmaRifle:
        ld      b,%00001000
        ld      de,CR_PLASMARIFLE
        jr      intexBuyWeapon
intexBuyFlamethrower:
        ld      b,%00010000
        ld      de,CR_FLAMETHROWER
intexBuyWeapon:
        ld      a,(weaponTable)                 ; A = bit array of owned weapons
        and     b                               ; check if weapon already owned
        jr      nz,intexAlreadyOwned            ; if so, tell the player & return to main menu
        call    intexCheckCredits               ; check if enough credits available
        ld      hl,weaponTable                  ; HL => bit array of owned weapons
        ld      a,b                             ; A = bit mask to OR
        or      (hl)
        ld      (hl),a                          ; save new bit array of owned weapons
        jr      intexItemPurchased

;------------------------------------------------
; intexBuy@ITEM@ - try to buy said item
;
; input:    none
; output:   none
;------------------------------------------------
intexBuyKeyPack:
        ld      de,CR_KEYPACK                   ; DE = cost
        call    intexCheckCredits               ; check if enough credits available
        ld      hl,keys                         ; HL => key passes
        ld      a,5
        add     a,(hl)                          ; A = new number of key passes
        ld      (hl),a                          ; save it
        jr      intexItemPurchased

intexBuyAmmoClip:
        ld      de,CR_AMMOCLIP
        call    intexCheckCredits
        call    addAmmoClip                     ; use the same routine as when player picks up a clip
        jr      intexItemPurchased

intexBuyFirstAid:
        ld      de,CR_FIRSTAID
        call    intexCheckCredits
        ld      hl,health                       ; HL => health
        ld      a,MAX_HEALTH/2
        add     a,(hl)                          ; A = new health
        ld      (hl),a                          ; save it
        cp      MAX_HEALTH                      ; check for overflow
        jr      c,intexItemPurchased            ; if not, no need to cap it
        ld      (hl),MAX_HEALTH                 ; set max health
        jr      intexItemPurchased

intexBuyRadarScanner:
        ld      b,%00100000                     ; B = bit mask for weaponTable (radar scanner is stored as a 6th weapon)
        ld      de,CR_RADARSCANNER
        jr      intexBuyWeapon

intexBuyExtraLife:
        ld      de,CR_EXTRALIFE
        call    intexCheckCredits
        ld      hl,lives
        inc     (hl)                            ; inc lives
; fall through to intexPurchased

;------------------------------------------------
; intexPurchased - display a msg telling player that they purchased the item
;
; input:    none
; output:   none
;------------------------------------------------
intexItemPurchased:
        ld      hl,strIntexPurchased            ; HL => string data
        jr      intexMsgWaitMenu                ; jump into the middle of intexAlreadyOwned

;------------------------------------------------
; intexCheckCredits - check if enough credits are available to buy an item
;
; input:    DE = cost of item
; output:   none
;------------------------------------------------
intexCheckCredits:
        ld      hl,(credits)
        or      a                               ; clear carry
        sbc     hl,de                           ; HL = any left over credits, or negative if not enough
        jr      c,intexNotEnoughCredits         ; if not enough, tell player
        ld      (credits),hl                    ; save left over credits amount
        ret                                     ; ret to "intexBuy@@@@@" routine
intexNotEnoughCredits:
        pop     hl                              ; pop ret value from routine call
        ld      hl,strIntexNotEnoughCredits     ; HL => string data
        jr      intexMsgWaitMenu                ; jump into the middle of intexAlreadyOwned

;------------------------------------------------
; intexAlreadyOwned - display a msg telling player that they already own an item
;
; input:    none
; output:   none
;------------------------------------------------
intexAlreadyOwned:
        ld      hl,strIntexAlreadyOwned         ; HL => string data
intexMsgWaitMenu:
        call    displayPage
        ld      b,50
        call    wait                            ; wait B times
        jp      intexMainMenu                   ; return to main menu

;------------------------------------------------
; drawIntexCursor - draw the menu cursor using XOR
;
; input:    none
; output:   none
;------------------------------------------------
drawIntexCursor:
#ifndef APP83P
        ld      de,$0000                        ; DE = menu item (set from earlier)
__intexMenuItem         = $-2
        ld      hl,$0000                        ; HL => cursor table (set from earlier)
__intexCursorTable      = $-2
#else
        ld      de,(__intexMenuItem)
        ld      d,0                             ; DE = (__intexMenuItem)
        ld      hl,(__intexCursorTable)
#endif
        add     hl,de
        ld      a,(hl)                          ; A = cursor x
        ld      e,a                             ; E = cursor x
        ld      a,(__intexMenuItem)
        inc     a                               ; A = menu item plus 1 (so that it can't = 0)
        ld      b,a
        ld      a,10+1-7                        ; A = cursor y for top option line minus 1
calcIntexCursorY:
        add     a,7                             ; A = next menu option line
        djnz    calcIntexCursorY
        ld      l,a                             ; L = y
        ld      a,e                             ; A = x
        ld      de,sprMenuItem                  ; DE => sprite
        jp      putSprite                       ; XOR sprite & leave (putSprite was hacked earlier to XOR instead of OR)

;------------------------------------------------
; displayChar - display a character to gbuf and update pencol & penrow accordingly
;
; input:    HL => character to display, NULL means line feed
; output:   HL => next character
;------------------------------------------------
displayChar:
        ld      a,(hl)                          ; A = character
        inc     hl                              ; HL => next character
        push    hl                              ; save character pointer
        bcall(_vputmap)                         ; display character and update pencol & penrow accordingly
        pop     hl                              ; restore character pointer
        ret

;------------------------------------------------
; displayPage - display a page of text using the displayChar routine and update it to the lcd
;
; input:    HL => page string data, each line NULL terminated, page is terminated by $FF
; output:   HL => data directly after this page's data
;------------------------------------------------
displayPage:
        res     apdrunning,(iy+apdflags)        ; disable apd
        push    hl                              ; save data ptr
        bcall(_getcsc)
        bcall(_getcsc)                          ; clear key press buffer
        bcall(_grbufclr)
        pop     hl                              ; restore data ptr
displayPageLoop:
        ld      d,(hl)                          ; D = new penrow
        inc     hl
        ld      e,(hl)                          ; E = new pencol
        inc     hl                              ; HL => next line string
        ld      (pencol),de                     ; set cursor for new line
displayLineLoop:
        ld      a,(hl)                          ; A = first character of line
        inc     a                               ; was it $FF
        jr      z,displayPageEndPage            ; if so, update ptr to next page data and ret
        dec     a                               ; was it $00
        jr      z,displayPageEndLine
        call    displayChar                     ; display char to gbuf
        push    hl                              ; save char ptr
        call    libFastCopy                     ; update lcd
        pop     hl                              ; restore char ptr
        jr      displayLineLoop

displayPageEndLine:
        inc     hl                              ; HL => next line data
        jr      displayPageLoop                 ; load new pen coords & repeat
displayPageEndPage:
        inc     hl                              ; HL => data directly after this page's data
        set     apdrunning,(iy+apdflags)        ; enable apd
        ret

.end
