;===============================================================;
;                                                               ;
; Alien Breed 5                                                 ;
; Enemy routines                                                ;
;                                                               ;
;===============================================================;

;------------------------------------------------
; clearAllEnemyTables - clear enemyTable, aholeTable
;
; input:    none
; output:   none
;------------------------------------------------
clearAllEnemyTables:
        ld      hl,enemyTable                   ; HL => first byte to clear
        ld      de,enemyTable+1                 ; DE => where to copy cleared bytes to
        ld      bc,NUM_ENEMIES*ENEMY_SIZE-1     ; BC = number of cleared bytes to copy
        ld      (hl),0                          ; clear first byte
        ldir                                    ; clear all subsequent bytes
        ld      hl,aholeTable                   ; ditto for aholeTable
        ld      de,aholeTable+1
        ld      bc,NUM_AHOLES*AHOLE_SIZE-1
        ld      (hl),0
        ldir
        ret

;------------------------------------------------
; checkEnemyTrigger - check a tile to see if it's an enemy trigger
;
; input:    HL => tile in level data
; output:   none
;------------------------------------------------
checkEnemyTrigger:
        ld      a,(hl)                          ; A = tile
        cp      TILE_OPENALIENHOLE              ; check if it's an open hole
        jp      z,loadAlienHole                 ; if so, try to load it
        ex      de,hl                           ; DE => tile in level data
        sub     TILE_ENEMYTRIGGER               ; check first enemy trigger
        jr      z,newEnemy1                     ; if so, try to use it
        dec     a                               ; check second enemy trigger
        ret     nz                              ; if not, simply leave
newEnemy2:
        ld      hl,enemy2                       ; HL => enemy type/level
        jr      newEnemy
newEnemy1:
        ld      hl,enemy1
; fall through to newEnemy

;------------------------------------------------
; newEnemy - try to create a new enemy
;
; input:    HL => enemy type/level
;           DE => trigger tile in level data
; output:   none
;------------------------------------------------
newEnemy:
        push    ix
        push    de                              ; save ptr to tile in level data
; try to find an empty entry in enemyTable
        ld      ix,enemyTable                   ; IX => start of enemyTable
        ld      b,NUM_ENEMIES                   ; B = number of entries to check
        ld      de,ENEMY_SIZE                   ; DE = size of enemyTable entry
findEmptyEnemyEntry:
        ld      a,(ix+0)                        ; A = enemy type
        or      a                               ; 0?
        jr      z,enemyEntryFound               ; if so, empty entry found
        add     ix,de                           ; IX => next entry
        djnz    findEmptyEnemyEntry             ; loop until whole table checked
        pop     de                              ; clear stack
        jr      endNewEnemy                     ; no empty entry found, so skip to end and leave
enemyEntryFound:
        ld      a,(hl)                          ; A = enemy data
        and     %00011111                       ; A = enemy type (0->..)
        inc     a                               ; A = enemy type (1->..)
        ld      (ix+0),a                        ; save it
        ld      b,a                             ; B = enemy type
        ld      (ix+1),%10000000                ; set bit 7 to indicate that a new direction needs to be chosen
        ex      (sp),hl                         ; (SP) => enemy data; HL => tile in level data
        call    getExactXYFromHL                ; DE = x; HL = y
        ld      (ix+2),e
        ld      (ix+3),d                        ; save x
        ld      (ix+4),l
        ld      (ix+5),h                        ; save y
        pop     hl                              ; HL => enemy data
        ld      a,(hl)
        rlca
        rlca
        rlca
        and     %00000111                       ; A = enemy level
        ld      c,a                             ; C = enemy level
        ld      a,b                             ; A = enemy type (B is still preserved from earlier)
        dec     a                               ; A = enemy type (0->..)
        add     a,a
        add     a,a
        add     a,a                             ; *8
        add     a,c                             ; A = (enemy type*8) + enemy level, aka offset for enemyLevelTable
        ld      e,a
        ld      d,0                             ; DE = offset for enemyLevelTable
        ld      hl,enemyLevelTable
        add     hl,de
        ld      a,(difficulty)
        or      a                               ; check if NORMAL or TOUGH difficulty
        ld      a,(hl)                          ; A = ini health for enemy
        jr      z,newEnemyAfterTough            ; if NORMAL, don't double ini health
        add     a,a
newEnemyAfterTough:
        ld      (ix+7),a                        ; save it
        ld      l,b                             ; L = enemy type (B is still preserved from earlier)
        dec     l
        ld      h,0                             ; HL = enemy type (0->..)
        add     hl,hl
        add     hl,hl
        add     hl,hl                           ; *8
        ld      b,8
        call    libRandom                       ; A = 0-7 (random)
        ld      e,a
        ld      d,0
        add     hl,de                           ; HL = (enemy type * 8) + random 0-7
        ld      de,enemySpeedTable
        add     hl,de
        ld      a,(hl)                          ; A = enemy speed (picked randomly from enemySpeedTable)
#ifndef APP83P
        nop                                     ; will be set to either "or a" or "scf"
__enemySpeedCheat       = $-1
#else
        call    __enemySpeedCheat
#endif
        jr      nc,afterEnemySpeedCheat
        ld      a,1                             ; if enemy speed cheat is on, enemy always moves at full speed
afterEnemySpeedCheat:
        ld      b,a                             ; B = enemy speed
        add     a,a
        add     a,a                             ; A = enemy speed shifted left twice
        or      b                               ; A = enemy speed stored twice, in bits 0/1 & bits 2/3
        ld      (ix+8),a                        ; save it
endNewEnemy:
        pop     ix
        ret

;------------------------------------------------
; loadAlienHole - try load set up data for an alien hole
;
; input:    HL => tile in level data where alien hole is located
; output:   none
;------------------------------------------------
loadAlienHole:
        push    ix                              ; IX must be preserved in case one of the horizontal scrolling routines led here
        ld      b,NUM_AHOLES                    ; B = number of entries in aholeTable
        ld      ix,aholeTable
        ld      de,AHOLE_SIZE
findEmptyAholeEntry:
        ld      a,(ix+0)
        or      a                               ; is this entry empty?
        jr      z,aholeEntryFound
        add     ix,de                           ; IX => next entry
        djnz    findEmptyAholeEntry
        pop     ix
        ret                                     ; if no empty entry, leave
aholeEntryFound:
        ld      a,(hl)                          ; A = tile
        cp      TILE_OPENALIENHOLE              ; is it an already open hole?
        ld      a,INI_OPENALIENHOLE             ; A = initial value for counter in the event of an already open hole
        jr      z,finishLoadAlienHole           ; if hole is open, go ahead
        ld      a,INI_UNOPENALIENHOLE           ; A = initial value for counter in the even of a hole that isn't open yet
finishLoadAlienHole:
        ld      (ix+0),a                        ; initialise counter
        ld      (ix+1),l
        ld      (ix+2),h                        ; save ptr to tile in level data
        call    getExactXYFromHL                ; DE = x; HL = y
        ld      (ix+3),e
        ld      (ix+4),d                        ; save x
        ld      (ix+5),l
        ld      (ix+6),h                        ; save y
        pop     ix
        ret

;------------------------------------------------
; updateAlienHoles - update all active alien holes
;
; input:    none
; output:   none
;------------------------------------------------
updateAlienHoles:
        ld      hl,frame
        bit     0,(hl)                          ; check if even frame
        ret     nz                              ; if not, don't update this frame
        ld      b,NUM_AHOLES                    ; B = number of entries in aholeTable
        ld      ix,aholeTable                   ; IX => start of alien hole data
updateAholeLoop:
        push    bc                              ; save counter
        ld      a,(ix+0)
        or      a                               ; is this alien hole active
        jr      z,endUpdateAholeLoop            ; if not, skip
        call    loadScreenCollideObject1
        inc     ix                              ; IX+2 => coord data for alien hole (bit of a hack to use the next routine)
        call    loadEnemyCollideObject2
        dec     ix                              ; return IX to start of entry
        call    checkCollision                  ; check if alien hole is still on screen
        jr      nc,clearAlienHole               ; if not, deactivate the alien hole for now
        bit     7,(ix+0)                        ; see if counter is a negative value
        jr      nz,updateUnopenAlienHole        ; if so, hole isn't open yet, so update that part
        dec     (ix+0)                          ; decrement counter
        jr      nz,endUpdateAholeLoop           ; if it hasn't reached zero, nothing else to do this frame
; otherwise, if it's reached zero, spawn an alien and reset counter
        ld      (ix+0),INI_OPENALIENHOLE        ; reset counter
        ld      hl,enemy1                       ; HL => enemy type/level (alien hole spawns first enemy type)
        ld      e,(ix+1)
        ld      d,(ix+2)                        ; DE => tile in level data
        call    newEnemy                        ; load a new enemy
        jr      endUpdateAholeLoop
clearAlienHole:
        ld      (ix+0),0                        ; turn off alien hole
        jr      endUpdateAholeLoop
updateUnopenAlienHole:
        inc     (ix+0)                          ; increment counter
        jr      nz,endUpdateAholeLoop           ; if it hasn't reached zero, nothing else to do this frame
; otherwise, if it's reached zero, break the hole open and reset the counter for an open hole
        ld      (ix+0),INI_OPENALIENHOLE        ; reset counter
        ld      l,(ix+1)
        ld      h,(ix+2)                        ; HL => alien hole in level data
        ld      a,TILE_OPENALIENHOLE            ; A = tile to set
        push    ix                              ; save data ptr
        call    putTile                         ; put the tile & display it
        pop     ix                              ; restore data ptr
endUpdateAholeLoop:
        ld      de,AHOLE_SIZE
        add     ix,de                           ; IX => next entry
        pop     bc                              ; restore counter
        djnz    updateAholeLoop
        ret

;------------------------------------------------
; drawEnemies - draw enemies from enemyTable
;
; input:    none
; output:   none
;------------------------------------------------
drawEnemies:
        ld      ix,enemyTable                   ; IX => first entry
        ld      b,NUM_ENEMIES                   ; B = num entries to check/draw
drawEnemiesLoop:
        push    bc                              ; save loop counter
        ld      a,(ix+0)                        ; A = enemy type
        or      a                               ; 0?
        jr      z,endDrawEnemiesLoop            ; if so, no enemy in this entry
        dec     a
        ld      l,a
        ld      h,0
        ld      e,l
        ld      d,h                             ; DE & HL both = enemy type - 1
        add     hl,hl                           ; *2
        ld      bc,enemySpriteTable
        add     hl,bc
        bcall(_ldhlind)                         ; HL => first sprite for this enemy type
        ex      de,hl                           ; DE => first sprite; HL = enemy type - 1
        ld      bc,enemyCounterMask
        add     hl,bc
        ld      a,(hl)                          ; A = mask to apply to enemy counter
        and     (ix+6)                          ; A = enemy counter masked, which becomes offset factor for animated sprite
        ld      l,a
        ld      h,0
        add     hl,hl
        add     hl,hl
        add     hl,hl
        add     hl,hl
        add     hl,hl                           ; *32 = offset to animated sprite
        add     hl,de                           ; HL => sprite set to show, now to calc direction
        ld      a,(ix+1)                        ; A = enemy direction
        and     %01111111                       ; ignore bit 7 (it's a flag)
        add     a,a
        add     a,a
        add     a,a                             ; *8
        ld      e,a
        ld      d,0
        add     hl,de                           ; HL => sprite to show
        push    hl                              ; save sprite ptr
        ld      de,(screenX)
        ld      l,(ix+2)
        ld      h,(ix+3)                        ; HL = enemy x
        or      a                               ; clear carry
        sbc     hl,de                           ; L = x coord to display at (H will be 0)
        ld      b,l                             ; B = x
        ld      de,(screenY)
        ld      l,(ix+4)
        ld      h,(ix+5)                        ; HL = enemy y
        or      a
        sbc     hl,de                           ; L = y coord to display at (H will be 0)
        ld      c,l                             ; C = y
        pop     hl                              ; HL => sprite
        push    ix                              ; save enemyTable ptr
        call    putClippedSprite                ; draw sprite
        pop     ix                              ; restore enemyTable ptr
endDrawEnemiesLoop:
        pop     bc                              ; restore loop counter
        ld      de,ENEMY_SIZE
        add     ix,de                           ; IX => next entry
        djnz    drawEnemiesLoop
        ret

;------------------------------------------------
; moveEnemies - move enemies
;
; input:    none
; output:   none
;------------------------------------------------
moveEnemies:
        call    loadScreenCollideObject1        ; will be used to check if enemies are still on screen
        ld      ix,enemyTable                   ; IX => start of enemy data
        ld      b,NUM_ENEMIES                   ; B = number enemies to check/move
moveEnemiesLoop:
        push    bc                              ; save counter
        ld      a,(ix+0)                        ; A = enemy type
        or      a                               ; 0?
        jp      z,endMoveEnemiesLoop            ; if so, no enemy in this entry
        call    loadEnemyCollideObject2         ; load enemy position into collideObject2
        call    checkCollision                  ; check if enemy is "colliding" with the screen, ie. check if the enemy is on screen or just outside screen
        jr      c,enemyOnScreen                 ; if so, continue with movement
        ld      (ix+0),0                        ; otherwise, disable this enemy entry
        jp      endMoveEnemiesLoop              ; and go to next entry
enemyOnScreen:
        ld      a,(ix+8)                        ; A = speed counter & speed counter max, in bits 0/1 & bits 2/3 respectively
        ld      c,a                             ; save it to A
        and     %00000011
        ld      b,a                             ; B = speed counter
        ld      a,c
        and     %00001100                       ; A = speed counter max in bits 2/3
        dec     b                               ; dec counter
        jp      nz,noMoveEnemy                  ; if NZ, don't move this frame
        ld      b,a                             ; B = speed counter max in bits 2/3
        srl     b
        srl     b                               ; B = speed counter max in bits 0/1
        or      b                               ; A = updated value of speed counter & max, in bits 0/1 & 2/3 respectively
        ld      (ix+8),a                        ; save it
; now work out enemy movement
        ld      a,(ix+1)                        ; A = enemy direction
        bit     7,a                             ; check if it's set or null
        jr      z,moveEnemy                     ; if it's set, try to move enemy in set direction
; otherwise, work out a new direction for enemy
        ld      hl,(playerX)
        ld      c,(ix+2)
        ld      b,(ix+3)                        ; BC = enemy x
        or      a
        sbc     hl,bc                           ; get difference between player & enemy x coords
        ld      (tempX),hl                      ; save x difference
        ex      de,hl                           ; DE = x diff
        ld      hl,(playerY)
        ld      c,(ix+4)
        ld      b,(ix+5)                        ; BC = enemy y
        or      a
        sbc     hl,bc                           ; get y coord difference
        ld      (tempY),hl                      ; save y difference
        call    absHL                           ; HL = abs(HL)
        ex      de,hl                           ; DE = abs(y diff); HL = x diff
        call    absHL                           ; HL = abs(x diff)
        or      a
        sbc     hl,de                           ; check which difference is greater
        jr      c,enemyYGreater                 ; if y difference greater, try to move on y axis
        ld      hl,(tempX)                      ; otherwise, try to move on x axis
        bit     7,h                             ; check if should move left or right
        jr      z,setEnemyDirRight
        ld      a,2                             ; A = left
        jr      moveEnemy
setEnemyDirRight:
        ld      a,3                             ; A = right
        jr      moveEnemy
enemyYGreater:
        ld      hl,(tempY)
        bit     7,h                             ; check if should move up or down
        jr      z,setEnemyDirDown
        xor     a                               ; A = up
        jr      moveEnemy
setEnemyDirDown:
        ld      a,1                             ; A = down

moveEnemy:
        ld      (ix+1),a                        ; save enemy direction
        ld      hl,enemyMoveDone                ; HL => where to ret to
        push    hl                              ; push address to stack
        or      a
        jr      z,moveEnemyUp
        dec     a
        jp      z,moveEnemyDown
        dec     a
        jp      z,moveEnemyLeft

moveEnemyRight:
        ld      l,(ix+2)
        ld      h,(ix+3)
        inc     hl                              ; HL = prospective new x
        ld      (__enemyXTest),hl               ; save it for collision test
        ld      de,7
        add     hl,de
        ex      de,hl                           ; DE = x coord of tile to check
moveEnemyX:
        ld      a,$37                           ; A = scf
        ld      (__enemyPlane),a                ; save instruction
        ld      l,(ix+4)
        ld      h,(ix+5)                        ; HL = enemy y
        ld      (__enemyYTest),hl               ; save it for collision test
        call    enemyCheckTile                  ; check if enemy can walk here
        ret     nc                              ; if not, leave
        ld      bc,7
        add     hl,bc                           ; HL = enemy y + 7 (bottom corner)
        call    enemyCheckTile
        ret     nc
        call    checkEnemyCollisions            ; make sure enemy isn't colliding with other enemies
        ret     nc
        ld      a,(__enemyXTest)                ; A = lsb of new x coord

enemyMoveSuccess:
        inc     (ix+6)                          ; inc enemy counter
        ld      hl,(__enemyXTest)               ; HL = new enemy x
        ld      (ix+2),l
        ld      (ix+3),h                        ; save it
        ld      hl,(__enemyYTest)
        ld      (ix+4),l
        ld      (ix+5),h
        and     %00000111                       ; check if enemy has moved on to an even tile (A was either lsb of new x or y coord depending on the plane that has changed)
        scf                                     ; tell the moveEnemyDone routine that the move was a success
        ret     nz                              ; if not even tile, nothing more to do
        ld      a,(ix+1)
        or      %10000000                       ; set bit 7 to indicate that next frame the enemy should choose a new direction to move
        ld      (ix+1),a
        scf                                     ; again with the scf (see a few lines above)
        ret

moveEnemyUp:
        ld      l,(ix+4)
        ld      h,(ix+5)
        dec     hl
        ld      (__enemyYTest),hl
moveEnemyY:
        ld      a,$B7                           ; A = or a
        ld      (__enemyPlane),a                ; save instruction
        ld      e,(ix+2)
        ld      d,(ix+3)
        ld      (__enemyXTest),de
        call    enemyCheckTile
        ret     nc
        ex      de,hl                           ; DE = y, HL = x
        ld      bc,7
        add     hl,bc
        ex      de,hl                           ; DE = x+7, HL = y
        call    enemyCheckTile
        ret     nc
        call    checkEnemyCollisions
        ret     nc
        ld      a,(__enemyYTest)                ; A = lsb of new y coord
        jr      enemyMoveSuccess

moveEnemyDown:
        ld      l,(ix+4)
        ld      h,(ix+5)
        inc     hl
        ld      (__enemyYTest),hl
        ld      de,7
        add     hl,de
        jr      moveEnemyY

moveEnemyLeft:
        ld      e,(ix+2)
        ld      d,(ix+3)
        dec     de
        ld      (__enemyXTest),de
        jp      moveEnemyX

enemyMoveDone:
        jr      c,endMoveEnemiesLoop            ; if move was successful, nothing else to do
        ld      hl,enemy2ndMoveDone             ; HL => where to ret to after 2nd move attempt
        push    hl                              ; save ret address to stack
#ifndef APP83P
        nop                                     ; will be either set to "or a" or "scf"
__enemyPlane            = $-1
#else
        call    __enemyPlane
#endif
        jr      nc,enemyMoveTryX                ; if nc, check the x plane for movement
        ld      hl,(playerY)
        ld      e,(ix+4)
        ld      d,(ix+5)
        or      a
        sbc     hl,de                           ; HL = y coord difference
        bit     7,h                             ; check if pos or neg
        ld      (ix+1),0                        ; set direction to up in case enemy is moving up
        jr      nz,moveEnemyUp
        ld      (ix+1),1
        jr      moveEnemyDown
enemyMoveTryX:
        ld      hl,(playerX)
        ld      e,(ix+2)
        ld      d,(ix+3)
        or      a
        sbc     hl,de
        bit     7,h
        ld      (ix+1),2
        jr      nz,moveEnemyLeft
        ld      (ix+1),3
        jp      moveEnemyRight

enemy2ndMoveDone:
        jr      c,endMoveEnemiesLoop
        ld      b,24
        call    libRandom                       ; get a random direction (or anything higher than 3 means wait next frame)
        cp      4
        jr      c,enemySetRandomDir
        ld      a,(ix+1)
        or      %10000000
enemySetRandomDir:
        ld      (ix+1),a                        ; save it to be tried next time

endMoveEnemiesLoop:
        pop     bc                              ; restore counter
        ld      de,ENEMY_SIZE
        add     ix,de                           ; IX => next enemy entry
        dec     b
        jp      nz,moveEnemiesLoop              ; essentially same as djnz, but to accommodate to longer jump distance
        ret

noMoveEnemy:
        or      b                               ; A = updated value of speed counter & max, in bits 0/1 & 2/3 respectively
        ld      (ix+8),a                        ; save it
        jr      endMoveEnemiesLoop              ; go to end of loop

;------------------------------------------------
; enemyCheckTile - check a tile to see if enemy can walk on it
;
; input:    DE = x
;           HL = y
; output:   CA = can walk on this tile
;           DE = x
;           HL = y
;------------------------------------------------
enemyCheckTile:
        push    de                              ; save x
        push    hl                              ; save y
        call    getTile                         ; A = tile
        cp      TILE_SOLID                      ; check if solid tile
        pop     hl                              ; HL = y
        pop     de                              ; DE = x
        ret

;------------------------------------------------
; enemyDead - kill an enemy and initiate an explosion animation
;
; input:    IX => entry in enemyTable
; output:   IX => entry in enemyTable
;------------------------------------------------
enemyDead:
        ld      hl,SCORE_ENEMY
        call    addScoreDiff                    ; add points for killing enemy
        ld      (ix+0),0                        ; clear enemy type
        ld      e,(ix+2)
        ld      d,(ix+3)                        ; DE = x
        ld      l,(ix+4)
        ld      h,(ix+5)                        ; HL = y
        ld      bc,0*256+4                      ; B = animation type; C = counter ini
        jp      newAnim                         ; set up new animation

.end
