;===============================================================;
;                                                               ;
; Alien Breed 5                                                 ;
; Collision checking routines                                   ;
;                                                               ;
;===============================================================;

;------------------------------------------------
; checkBulletEnemyCollisions - check for collisions between bullets & enemies
;
; input:    none
; output:   none
;------------------------------------------------
checkBulletEnemyCollisions:
        di                                      ; disable interrupts as IY will be used
        push    iy                              ; save IY (system flag ptr)
        ld      iy,bulletTable                  ; IY = bullet data
        ld      b,NUM_BULLETS                   ; B = number of bullets to check
cbecLoop:
        push    bc                              ; save bullet entry counter
        ld      a,(iy+0)
        or      a                               ; bullet in this entry?
        jr      z,endCBECLoop                   ; if not, go to next entry
        call    loadBulletCollideObject1        ; A = damage value; bullet loaded to collideObject1
        ld      (__cbecDamage),a                ; save damage value to code for later
        ld      ix,enemyTable                   ; IX => enemy data
        ld      b,NUM_ENEMIES                   ; B = number of enemies to check
cbecLoop2:
        push    bc                              ; save enemy entry counter
        ld      a,(ix+0)
        or      a                               ; enemy in this entry?
        jr      z,endCBECLoop2                  ; if not, go to next entry
        call    loadEnemyCollideObject2         ; load enemy into collideObject2
        call    checkCollision                  ; check for collision
        jr      nc,endCBECLoop2                 ; if no collision, go to next entry
        ld      a,(iy+0)                        ; A = bullet type
        cp      4                               ; is it plasma rifle or flamethrower bullet?
        jr      nc,cbecAfterEndBullet           ; if so, don't end the bullet
        push    ix                              ; save enemyTable ptr
        push    iy
        pop     ix                              ; IX => bulletTable ptr
        call    removeBulletCheckAnim           ; check if bullet requires an explosion animation & remove it (located in shoot.asm)
        pop     ix                              ; restore enemyTable ptr
cbecAfterEndBullet:
        ld      a,(ix+7)                        ; A = enemy health
#ifndef APP83P
        sub     $00                             ; value will be loaded earlier
__cbecDamage            = $-1
#else
        ld      de,(__cbecDamage)               ; E = (__cbecDamage)
        sub     e
#endif
        ld      (ix+7),a                        ; save new health value
        jr      nc,endCBECLoop2                 ; if health didn't go down past 0, go to next entry
        call    enemyDead
endCBECLoop2:
        pop     bc                              ; restore enemy entry counter
        ld      de,ENEMY_SIZE
        add     ix,de                           ; IX => next enemy entry
        djnz    cbecLoop2
endCBECLoop:
        pop     bc                              ; restore bullet entry counter
        ld      de,BULLET_SIZE
        add     iy,de                           ; IY => next bullet entry
        djnz    cbecLoop
        pop     iy                              ; restore system flag ptr
        ei                                      ; thus, interrupts can be enabled again
        ret

;------------------------------------------------
; checkBulletBossCollisions - check for collisions between bullets & boss
;
; input:    none
; output:   none
;------------------------------------------------
checkBulletBossCollisions:
        ld      hl,collideObject2
        ld      de,(__bossX)
        ld      (hl),e                          ; load x coord LSB
        inc     hl
        ld      (hl),d                          ; load x MSB
        inc     hl
        ld      (hl),16                         ; load width
        inc     hl
        ld      de,(__bossY)
        ld      (hl),e                          ; load y coord LSB
        inc     hl
        ld      (hl),d                          ; load y MSB
        inc     hl
        ld      (hl),24                         ; load height
        di                                      ; no interrupts so we can use IY
        push    iy                              ; save system flag ptr
        ld      iy,bulletTable                  ; IY => start of bullet data
        ld      b,NUM_BULLETS                   ; B = number of entries to check
cbbcLoop:
        push    bc                              ; save loop counter
        ld      a,(iy+0)
        or      a                               ; is there a bullet in this entry?
        jr      z,endCBBCLoop                   ; if not, skip this entry
        call    loadBulletCollideObject1        ; A = damage value; bullet loaded to collideObject1
        ld      (__cbbcDamage),a                ; save damage value to code for later
        call    checkCollision                  ; check for a collision
        jr      nc,endCBBCLoop                  ; if no collision, next entry
        ld      e,(iy+2)
        ld      d,(iy+3)                        ; DE = bullet x
        ld      l,(iy+4)
        ld      h,(iy+5)                        ; HL = bullet y
        ld      bc,0*256+4                      ; B = animation type (explosion); C = ini anim counter
        call    newAnim                         ; start an explosion animation
        ld      (iy+0),0                        ; clear entry
#ifndef APP83P
        ld      hl,$0000                        ; boss health is initiated here during iniBoss and stored here thereafter
__bossHealth            = $-2
        ld      de,$0000                        ; LSB was loaded just before; MSB is always $00
__cbbcDamage            = $-2
#else
        ld      hl,(__bossHealth)
        ld      de,(__cbbcDamage)
        ld      d,0                             ; DE = (__cbbcDamage)
#endif
        or      a                               ; clear carry
        sbc     hl,de                           ; HL = boss health after taking damage
        ld      (__bossHealth),hl               ; save it
        jp      c,bossDead                      ; if it went below 0, boss is dead; no more collision checking req'd
endCBBCLoop:
        pop     bc                              ; restore loop counter
        ld      de,BULLET_SIZE
        add     iy,de                           ; IY => next entry
        djnz    cbbcLoop
        pop     iy                              ; restore system flag ptr
        ei                                      ; enable interrupts again
        ret

;------------------------------------------------
; checkPlayerEnemyCollisions - check for collisions between player & enemies
;
; input:    none
; output:   none
;------------------------------------------------
checkPlayerEnemyCollisions:
        call    loadPlayerCollideObject1
        ld      ix,enemyTable                   ; IX => start of enemy data
        ld      b,NUM_ENEMIES                   ; B = number of enemies to check
cpecLoop:
        push    bc                              ; save counter
        ld      a,(ix+0)
        or      a                               ; enemy in this entry?
        jr      z,endCPECLoop                   ; if not, skip this entry
        call    loadEnemyCollideObject2
        call    checkCollision                  ; check for collision
        jr      nc,endCPECLoop                  ; if no collision, skip this entry
        ld      hl,playerInv
        ld      a,(hl)
        inc     hl
        or      (hl)                            ; check if player dead or invincible
        jr      nz,cpecAfterHurtPlayer          ; if one of the above, skip hurting the player
        ld      l,(ix+0)
        dec     l
        ld      h,0                             ; HL = enemy type (0->..)
        ld      de,enemyDamageTable
        add     hl,de                           ; HL => enemy damage
        ld      l,(hl)                          ; L = damage to take on player
#ifndef APP83P
        nop                                     ; will be set to either "or a" or "scf"
__insaneModeCheat       = $-1
#else
        call    __insaneModeCheat
#endif
        jr      nc,cpecAfterInsane
        sla     l
        sla     l                               ; x4 damage for INSANE
        jr      cpecAfterTough
cpecAfterInsane:
        ld      a,(difficulty)
        or      a                               ; check difficulty level
        jr      z,cpecAfterTough                ; if NORMAL, don't double damage
        sla     l                               ; double the damage for TOUGH
cpecAfterTough:
        ld      a,(health)
        sub     l                               ; A = new player health after being hit by enemy
        ld      (health),a                      ; save it
        ld      a,1
        ld      (playerHurt),a                  ; set flag for player getting hurt this frame
cpecAfterHurtPlayer:
        call    enemyDead                       ; kill enemy and start an explosion animation
endCPECLoop:
        pop     bc                              ; restore counter
        ld      de,ENEMY_SIZE   
        add     ix,de                           ; IX => next entry in enemyTable
        djnz    cpecLoop
        ret

;------------------------------------------------
; checkPlayerBossCollisions - check for collisions between player & boss
;
; input:    none
; output:   none
; notes:    must be run directly after checkBulletBossCollisions so that boss is still loaded in collideObject2
;------------------------------------------------
checkPlayerBossCollisions:
        ld      hl,playerInv
        ld      a,(hl)
        inc     hl
        or      (hl)                            ; A = (playerInv) or'd with (playerDead)
        ret     nz                              ; leave if they're not both zero
        call    loadPlayerCollideObject1
        call    checkCollision
        ret     nc                              ; if not colliding, leave
        ld      a,-1
        ld      (health),a                      ; set player health to -1 (player is killed)
        neg                                     ; A = 1
        ld      (playerHurt),a                  ; set flag to say player was hurt this frame
        ret

;------------------------------------------------
; checkEnemyCollisions - check for collisions between enemies
;
; input:    (__enemyXTest) = x coord of test enemy
;           (__enemyYTest) = y coord of test enemy
; output:   CA = enemy not colliding
;------------------------------------------------
checkEnemyCollisions:
        push    ix                              ; IX needs to be saved for outside routines
        ld      ix,enemyTable                   ; IX = enemy data
        ld      bc,NUM_ENEMIES*256+0            ; B = number of entries to check; C = 0 (collision counter)
cecLoop:
        ld      a,(ix+0)
        or      a                               ; 0?
        jr      z,endCECLoop                    ; if so, no enemy in this entry
#ifndef APP83P
        ld      hl,$0000                        ; x input
__enemyXTest            = $-2
#else
        ld      hl,(__enemyXTest)
#endif
        ld      e,(ix+2)
        ld      d,(ix+3)                        ; DE = enemy x
        or      a
        sbc     hl,de
        ld      a,l                             ; A = x difference between enemies
        jr      nc,cecXAfterNeg                 ; if NC, the difference is already a positive value
        neg                                     ; negate to change difference to a positive value
cecXAfterNeg:
        cp      8                               ; is x difference greater or equal to 8?
        jr      nc,endCECLoop                   ; if so, go to next enemy
#ifndef APP83P
        ld      hl,$0000                        ; y input
__enemyYTest            = $-2
#else
        ld      hl,(__enemyYTest)
#endif
        ld      e,(ix+4)
        ld      d,(ix+5)                        ; DE = enemy y
        or      a
        sbc     hl,de
        ld      a,l                             ; A = y difference between enemies
        jr      nc,cecYAfterNeg                 ; if NC, the difference is already a positive value
        neg                                     ; negatte to change difference to a positive value
cecYAfterNeg:
        cp      8                               ; is y difference greater or equal to 8?
        jr      nc,endCECLoop                   ; if so, go to next enemy
        inc     c                               ; enemies are colliding, to inc collision counter
endCECLoop:
        ld      de,ENEMY_SIZE
        add     ix,de                           ; IX => next enemy entry
        djnz    cecLoop                         ; loop
        ld      a,c                             ; A = collision counter
        cp      2                               ; were there 2 or more collisions? (there will always be 1, as enemy will be colliding with itself)
        pop     ix                              ; restore IX for outside routines
        ret

;------------------------------------------------
; loadBulletCollideObject1 - load a bullet to collideObject1 and get it's damage value
;
; input:    IY => entry in bulletTable
; output:   A = damage value
;           IY => entry in bullet table
;------------------------------------------------
loadBulletCollideObject1:
        ld      l,(iy+0)
        dec     l
        ld      c,l                             ; C = bullet type (0->..)
        ld      h,0                             ; HL = bullet type (0->..)
        ld      b,0                             ; BC = bullet type (0->..) for later
        add     hl,hl
        add     hl,hl
        add     hl,hl                           ; *8
        ld      de,bulletSizeTable
        add     hl,de                           ; HL => size data for bullet type
        ld      a,(iy+1)
        add     a,a                             ; A = direction*2
        ld      e,a
        ld      d,0
        add     hl,de                           ; HL => bullet width
        ld      de,collideObject1
        ld      a,(iy+2)
        ld      (de),a                          ; save x LSB
        inc     de
        ld      a,(iy+3)
        ld      (de),a                          ; save x MSB
        inc     de
        ld      a,(hl)
        ld      (de),a                          ; save width
        inc     de
        inc     hl                              ; DE => bullet height
        ld      a,(iy+4)
        ld      (de),a                          ; save y LSB
        inc     de
        ld      a,(iy+5)
        ld      (de),a                          ; save y MSB
        inc     de
        ld      a,(hl)
        ld      (de),a                          ; save height
        ld      hl,bulletDamageTable
        add     hl,bc                           ; HL => bullet damage (BC was bullet type from earlier)
        ld      a,(hl)                          ; A = bullet damage
        ret

;------------------------------------------------
; loadPlayerCollideObject1 - load player to collideObject1
;
; input:    none
; output:   none
;------------------------------------------------
loadPlayerCollideObject1:
        ld      hl,collideObject1
        ld      de,(playerX)
        ld      (hl),e                          ; load x LSB
        inc     hl
        ld      (hl),d                          ; load x MSB
        inc     hl
        ld      (hl),8                          ; load width
        inc     hl
        ld      de,(playerY)
        ld      (hl),e                          ; load y LSB
        inc     hl
        ld      (hl),d                          ; load y MSB
        inc     hl
        ld      (hl),8                          ; load height
        ret

;------------------------------------------------
; loadScreenCollideObject1 - load screen to collideObject1
;
; input:    none
; output:   none
;------------------------------------------------
loadScreenCollideObject1:
        ld      hl,(screenX)
        ld      de,24
        or      a
        sbc     hl,de                           ; HL = screen x less DE
        jr      nc,loadScreenCo1X
        ld      hl,0                            ; if HL is less than 0, set it to 0
loadScreenCo1X:
        ld      (collideObject1),hl             ; load x
        ld      a,96+(24*2)
        ld      (collideObject1+2),a            ; load width
        ld      hl,(screenY)
        or      a
        sbc     hl,de                           ; HL = screen y less DE
        jr      nc,loadScreenCo1Y
        ld      hl,0
loadScreenCo1Y:
        ld      (collideObject1+3),hl           ; load y
        ld      a,64+(24*2)
        ld      (collideObject1+5),a            ; load height
        ret

;------------------------------------------------
; loadEnemyCollideObject2 - load an enemy into collideObject2
;
; input:    IX => entry in enemyTable
; output:   IX => entry in enemyTable
;------------------------------------------------
loadEnemyCollideObject2:
        ld      hl,collideObject2
        ld      a,(ix+2)
        ld      (hl),a                          ; save x LSB
        inc     hl
        ld      a,(ix+3)
        ld      (hl),a                          ; save x MSB
        inc     hl
        ld      (hl),8                          ; save width
        inc     hl
        ld      a,(ix+4)
        ld      (hl),a                          ; save y LSB
        inc     hl
        ld      a,(ix+5)
        ld      (hl),a                          ; save y MSB
        inc     hl
        ld      (hl),8                          ; save height
        ret

;------------------------------------------------
; checkCollision - check for collisions between two objects
;   modified version of a similar routine by Patrick Davidson
;
; input:    (collideObject1) = object 1 [x{2}, width, y{2}, height]
;           (collideObject2) = object 1 [x{2}, width, y{2}, height]
; output:   CA = 1 if they are colliding
;------------------------------------------------
checkCollision:
        ld      hl,(collideObject1)             ; HL = obj1 x
        ld      de,(collideObject2)             ; DE = obj2 x
        or      a                               ; clear carry
        sbc     hl,de
        jr      c,checkCollision1               ; if negative, obj1 is to the left of obj2, so check difference against obj1 width
        ld      de,(collideObject2+2)
        ld      d,0                             ; DE = obj2 width; HL = difference; CA = 0
        sbc     hl,de                           ; check if difference is greater than obj2 width
        ret     nc                              ; if so, no collision, return with CA = 0
        jr      checkCollision2
checkCollision1:
        call    absHL                           ; HL = abs(HL)
        inc     hl                              ; HL = difference (made positive via 2's complement)
        ld      de,(collideObject1+2)
        ld      d,0                             ; DE = obj1 width
        or      a                               ; clear carry
        sbc     hl,de                           ; check if difference is greater than obj1 width
        ret     nc                              ; if so, no collision, return with CA = 0
checkCollision2:
        ld      hl,(collideObject1+3)           ; HL = obj1 y
        ld      de,(collideObject2+3)           ; DE = obj2 y
        or      a                               ; clear carry
        sbc     hl,de
        jr      c,checkCollision3               ; if negative, obj1 is above obj2, so check difference against obj1 height
        ld      de,(collideObject2+5)
        ld      d,0                             ; DE = obj2 height; HL = difference; CA = 0
        sbc     hl,de                           ; check if difference is greater than ob2 height
        ret                                     ; ret regardless with the carry flag result
checkCollision3:
        call    absHL                           ; HL = abs(HL)
        inc     hl                              ; HL = difference (made positive via 2's complement)
        ld      de,(collideObject1+5)
        ld      d,0                             ; DE = obj1 height
        or      a                               ; clear carry
        sbc     hl,de                           ; check if difference is greater than obj1 height
        ret                                     ; ret regardless with the carry flag result

.end
