/************************************************************************/
/************************************************************************/
/*																		*/
/*						12 x 12 Tile and Sprite Editor					*/
/*																		*/
/************************************************************************/
/************************************************************************/

#include <stdio.h>						/* Generic includes */
#include <stdlib.h>
#include <string.h>
#include <graphics.h>
#include <dos.h>
#include <dir.h>

typedef struct _CharDef					/* Structure defining one character */
{
	int	Pixels[12];						/* Array of 12x12 pixels */
	int IsSprite;						/* Is Sprite Flag */
} CHARDEF;

#define	XPOS	(32)    				/* Cell Position */
#define YPOS	(18)
#define XSIZE	(28)
#define YSIZE	(13)

/************************************************************************/
/*																		*/
/*						Dump Character definition						*/
/*																		*/
/************************************************************************/

/*	If pixels are ABC													*/
/*				  DEF	then output $AB,$CD,$EF							*/

void CharDump(FILE *fOut,CHARDEF *c)
{
	int x;
	fprintf(fOut,"        .db  ");		/* Byte data */
	for (x = 0;x < 12;x = x + 2)		/* Two lines at a time, see above */
	{
		if (x > 0) fprintf(fOut,",");	/* Comma if continuation */
		fprintf(fOut,"$%02x,",(c->Pixels[x] >> 4));
		fprintf(fOut,"$%01x%01x,",(c->Pixels[x] & 0x0F),(c->Pixels[x+1] >> 8));
		fprintf(fOut,"$%02x",(c->Pixels[x+1] & 0xFF));
	}
	fprintf(fOut,"\n");
}

/************************************************************************/
/*																		*/
/*						Refresh cell type button						*/
/*																		*/
/************************************************************************/

void RefreshType(CHARDEF *c)
{
	char *s = (c->IsSprite) ? "Sprite":"Tile";
	setfillstyle(SOLID_FILL,LIGHTGRAY);bar(XPOS,180,XPOS+96,192);
	setcolor(BLACK);rectangle(XPOS,180,XPOS+96,192);
	outtextxy(XPOS+48-textwidth(s)/2,183,s);
}

/************************************************************************/
/*																		*/
/*							Refresh a given cell						*/
/*																		*/
/************************************************************************/

void RefreshCell(CHARDEF *c,int x,int y)
{
	setfillstyle(SOLID_FILL,BLACK);
	if (c->Pixels[y] & (0x800 >> x)) setfillstyle(SOLID_FILL,YELLOW);
	bar(x*XSIZE+XPOS,y*YSIZE+YPOS,x*XSIZE+XPOS+XSIZE-1,y*YSIZE+YPOS+YSIZE-1);
	setcolor(BLACK);
	rectangle(x*XSIZE+XPOS,y*YSIZE+YPOS,x*XSIZE+XPOS+XSIZE-1,y*YSIZE+YPOS+YSIZE-1);
}

/************************************************************************/
/*																		*/
/*						   Erase a Sprite Record						*/
/*																		*/
/************************************************************************/

void ClearChar(CHARDEF *c)
{
	int x;
	for (x = 0;x < 12;x++)	c->Pixels[x] = 0;
	c->IsSprite = 0;
}

/************************************************************************/
/*																		*/
/*							Draw a sprite image							*/
/*																		*/
/************************************************************************/

void DrawImage(int xc,int yc,CHARDEF *c)
{
	int x,y,p;
	for (x = 0;x < 12;x++)
		for (y = 0;y < 12;y++)
		{
			p = (c->Pixels[y] & (0x800 >> x)) ? YELLOW:BLACK;
			putpixel(xc+x*2,yc+y,p);
			putpixel(xc+x*2+1,yc+y,p);
		}
}

/************************************************************************/
/*																		*/
/*			  Rotate a sprite/tile image 90 degrees right				*/
/*																		*/
/************************************************************************/

void RotateImage(CHARDEF *Src,CHARDEF *Tgt)
{
	int x,y;
	for (x = 0;x < 12;x++) Tgt->Pixels[x] = 0;
	for (x = 0;x < 12;x++)
		for (y = 0;y < 12;y++)
		{
			if (Src->Pixels[y] & (0x800 >> x))
			{
				Tgt->Pixels[x] |= (0x800 >> (11-y));
			}
		}
}

/************************************************************************/
/*																		*/
/*						Refresh the sprite images						*/
/*																		*/
/************************************************************************/

void RefreshImages(CHARDEF *c)
{
	CHARDEF r1,r2;
	int x = XPOS+XSIZE*12+32;
	DrawImage(x,YPOS,c);
	RotateImage(c,&r1);DrawImage(x,YPOS+32,&r1);
	RotateImage(&r1,&r2);DrawImage(x+40,YPOS+32,&r2);
	RotateImage(&r2,&r1);DrawImage(x+80,YPOS+32,&r1);
}

/************************************************************************/
/*																		*/
/*					    Draw cursor at given position					*/
/*																		*/
/************************************************************************/

void DrawCursor(int x,int y)
{
	char Demo[256];
	int n = imagesize(0,0,8,8);
	if (x > 620) return;
	getimage(630,0,637,7,Demo);
	putimage(x-3,y-3,Demo,XOR_PUT);
}

/************************************************************************/
/*																		*/
/*		  Get a mouse click, given the current cursor position			*/
/*																		*/
/************************************************************************/

void GetMouse(int *x,int *y)
{
	union REGS r;
	DrawCursor(*x,*y);
	do
	{
		r.x.ax = 0x03;
		int86(0x33,&r,&r);
	} while(r.x.bx != 0);
	do
	{
		r.x.ax = 0x03;
		int86(0x33,&r,&r);
		DrawCursor(*x,*y);DrawCursor(r.x.cx,r.x.dx);
		*x = r.x.cx;*y = r.x.dx;
		delay(80);
	}
	while (r.x.bx == 0);
	DrawCursor(*x,*y);
}

void main(int argc,char *argv[])
{
	int x,y,x1,y1;
	CHARDEF Chr,r1,r2;
	char s[64],Drive[4],Dir[128],Name[9],Ext[4];
	FILE *f;

	if (argc != 2) return;
	x = VGA;y = VGALO;initgraph(&x,&y,".");
	setfillstyle(SOLID_FILL,BLUE);bar(0,0,640,200);
	setfillstyle(SOLID_FILL,YELLOW);bar(0,0,640,9);
	sprintf(s,"%s - 12 x 12 Sprite and Tile Editor",argv[1]);
	setcolor(BLACK);outtextxy(20,2,s);
	rectangle(3,2,14,8);setfillstyle(SOLID_FILL,BLACK);
	bar(6,4,11,6);for (y = 2;y < 9;y+=2) line(textwidth(s)+24,y,620,y);
	setcolor(YELLOW);rectangle(0,0,639,199);
	ClearChar(&Chr);
	f = fopen(argv[1],"rb");
	if (f != NULL)
	{
		fread(&Chr,sizeof(CHARDEF),1,f);
		fclose(f);
	}
	for (x = 0;x < 12;x++)
		for (y = 0;y < 12;y++)
				RefreshCell(&Chr,x,y);
	RefreshType(&Chr);RefreshImages(&Chr);
	x = 320;y = 100;
	while (GetMouse(&x,&y),y >= 10)
	{
		if (x > XPOS && y > YPOS && x < XPOS+12*XSIZE && y < YPOS+12*YSIZE)
		{
			x1 = (x-XPOS)/XSIZE;y1 = (y-YPOS)/YSIZE;
			Chr.Pixels[y1] ^= (0x800 >> x1);
			RefreshCell(&Chr,x1,y1);RefreshImages(&Chr);
		}
		if (y >= 180)
		{
			Chr.IsSprite = (Chr.IsSprite == 0);
			RefreshType(&Chr);
		}
	}
	f = fopen(argv[1],"wb");
	fwrite(&Chr,sizeof(CHARDEF),1,f);
	fclose(f);
	closegraph();

	fnsplit(argv[1],Drive,Dir,Name,Ext);
	strlwr(Name);
	fnmerge(s,Drive,Dir,Name,".inc");
	f = fopen(s,"w");
	fprintf(f,"%s_%s:\n",(Chr.IsSprite) ? "sprite":"tile",Name);
	CharDump(f,&Chr);
	if (Chr.IsSprite)
	{
		RotateImage(&Chr,&r1);CharDump(f,&r1);
		RotateImage(&r1,&r2);CharDump(f,&r2);
		RotateImage(&r2,&r1);CharDump(f,&r1);
	}
	fclose(f);
}