; =========================================================
; Module: ProgramEditor
; =========================================================
; Description:
;     Module to detect and edit programs.
;     Why not use Ion's built-in routine, ionDetect, I hear
;     you ask?
;     Well, this library offers extended functionality, and
;     rather than shoe-horn on top of Ion's functionality
;     I have instead opted to go for a system that more
;     closely follows the TIOS, which provides more useful
;     information than Ion does.
; =========================================================

.module ProgramEditor

; Used to store the last program's name (as OP1 can be
; destroyed by other functions).
.var ubyte[9], Name

; ---------------------------------------------------------
; SetupSearch -> Start searching from the beginning.
; ---------------------------------------------------------
; Destroys: af, bc, de, hl.
; ---------------------------------------------------------
SetupSearch
	ld hl,Name
	ld (hl),ProtProgObj
	inc hl
	ld bc,8
	bcall(_MemClear)	
	ret

; ---------------------------------------------------------
; FindNextLoop -> Find the next program alphabetically.
; ---------------------------------------------------------
; Inputs:   Name = current name.
; Outputs:  c on failure, nc on success.
;           Found program name stored in Name.
; Destroys: af, bc, de, hl.
; Remarks:  Will loop around after finding the last program
;           name.
; ---------------------------------------------------------
FindNextLoop
	call FindNext
	ret nc ; One was found!
	xor a
	ld (Name[1]),a
	; Run-on...

; ---------------------------------------------------------
; FindNext -> Find the next program alphabetically.
; ---------------------------------------------------------
; Inputs:   Name = current name.
; Outputs:  c on failure, nc on success.
;           Found program name stored in Name.
; Destroys: af, bc, de, hl.
; ---------------------------------------------------------
FindNext
	ld hl,Name
	rst rMOV9TOOP1
	bcall(_FindAlphaUp)
	ld hl,OP1
	ld bc,9
	ld de,Name
	ldir
	ret

; ---------------------------------------------------------
; GetInfo -> Get information about the current program.
; ---------------------------------------------------------
; Inputs:   Name = program name.
; Outputs:  c on failure, nc on success.
;           hl -> VAT entry.
;           de -> data offset.
;           b == 0 = variable stored in RAM.
;           b != 0 = ROM page number.
; Destroys: af, c.
; ---------------------------------------------------------
GetInfo
	ld hl,Name
	rst rMOV9TOOP1
	bcall(_ChkFindSym)
	ret

; ---------------------------------------------------------
; FindNextHeaderedLoop -> Find the next program which has a
;                         particular header.
; ---------------------------------------------------------
; Inputs:   Name = current name.
;           hl -> header.
; Outputs:  c on failure, nc on success.
;           Found program name stored in Name.
; Destroys: af, bc, de, hl.
; Remarks:  Will also search Flash ROM for variable. Will
;           loop around after the last program name.
; ---------------------------------------------------------
FindNextHeaderedLoop
	push hl
	call FindNextHeadered
	pop hl
	ret nc ; Found one!
	xor a
	ld (Name[1]),a
	; Run-on...

; ---------------------------------------------------------
; FindNextHeadered -> Find the next program which has a
;                     particular header.
; ---------------------------------------------------------
; Inputs:   Name = current name.
;           hl -> header.
; Outputs:  c on failure, nc on success.
;           Found program name stored in Name.
; Destroys: af, bc, de, hl.
; Remarks:  Will also search Flash ROM for variable.
; ---------------------------------------------------------
FindNextHeadered
	push hl
	call FindNext
	pop hl
	ret c ; Couldn't find any more programs.
	
	push hl
	call OpenReadOnly
	pop hl

	; Couldn't open (for some reason).
	jr c,FindNextHeadered 
	
	inc de
	inc de
	
-	ld a,(hl)
	or a
	jr z,FoundHeadered
	ld a,(de)
	cp (hl)
	jr nz,FindNextHeadered
	inc hl
	inc de
	jr {-}

FoundHeadered
	or a ; Clear carry.
	ret
	
; ---------------------------------------------------------
; GetBytePaged -> Reads a byte from Flash ROM.
; ---------------------------------------------------------
; Inputs:   hl = offset on ROM page to read.
;           b  = page number.
; Outputs:  a = byte read.
; Destroys: f.
; Remarks:  hl and b move on to point at the next byte.
; ---------------------------------------------------------
GetBytePaged
	push bc
	bcall(_LoadCIndPaged)
	ld a,c
	pop bc
	inc hl
	bit 7,h
	ret z
	inc b
	res 7,h
	set 6,h
	ret

; ---------------------------------------------------------
; OpenReadOnly -> Open a program to read its data.
; ---------------------------------------------------------
; Inputs:   Name = program name.
; Outputs:  c on failure, nc on success.
;           de -> program data (size bytes).
; Destroys: af, bc, de, hl.
; Remarks:  Program will be automatically copied to RAM
;           if need be (NOT unarchived).
; ---------------------------------------------------------
OpenReadOnly
	call GetInfo
	ret c
	
	ld a,b
	or a
	jr z,IsInRam

	; Variable is in ROM.
	; First up - do we have enough space to address it in RAM?
	
	ld h,d
	ld l,e

	ld d,10
-	call GetBytePaged
	dec d
	jr nz,{-}
	
	; Skip over name:
	
	ld d,a
-	call GetBytePaged
	dec d
	jr nz,{-}
	
	push hl
	bcall(_LoadDEIndPaged)
	push bc
	push de
	
	; de = size
	ld h,d
	ld l,e
	bcall(_EnoughMem)
	
	pop de
	pop bc
	pop hl
	
	ret c ; Not enough RAM.

	ld a,b	
	ld b,d
	ld c,e
	
	ld de,(tempMem)
	bcall(_FlashToRam)
	
	ld de,(tempMem)	

IsInRam
	or a
	ret

; ---------------------------------------------------------
; OpenReadWrite -> Open a program for read/write access.
; ---------------------------------------------------------
; Inputs:   Name = program name.
; Outputs:  c on failure, nc on success.
;           nz if data had to be unarchived.
;           de -> program data (size bytes).
; Destroys: af, bc, de, hl.
; Remarks:  Program will be unarchived if need be. Use
;           OpenReadOnly if you only need to read the file,
;           as it won't have to unarchive the file!
; ---------------------------------------------------------
OpenReadWrite
	call GetInfo
	ret c
	ld a,b
	or a
	
	jr nz,IsArchived
	call OpenReadOnly ; Reopen, but as read only.
	ret c
	xor a
	ret

IsArchived
	; Try opening as readonly.
	call OpenReadOnly
	ret c ; Couldn't open as read only!
	
	ex de,hl
	ld e,(hl)
	inc hl
	ld d,(hl)
	
	ld hl,32
	add hl,de

	bcall(_EnoughMem)
	ret c
	
	call GetInfo
	bcall(_Arc_Unarc)
	call OpenReadOnly
	ret c
	xor a
	inc a
	ret
.endmodule