; =========================================================
; Module: Infrared
; =========================================================
; Description:
;     Infrared communications library.
; Remarks:
;     A lot of the following documentation refers to 40kHz
;     cycles. Depending on the value of CarrierFrequency,
;     substitute in 38kHz as appropriate.
; =========================================================

.module Infrared

CarrierFrequency = 38
.if !(CarrierFrequency == 38 || CarrierFrequency == 40)
.fail "Carrier frequency can only be 38kHz or 40kHz."
.endif

; ---------------------------------------------------------
; Play -> Plays back a recorded IR signal.
; ---------------------------------------------------------
; Inputs:   hl -> Stream to play back.
; Destroys: af, bc, de, hl.
; Remarks:  Disables interrupts.
; Format:   First byte: how many events.
;           Subsequent data: Events in form <word>, <word>
;           where first word is how long to transmit a
;           40kHz signal and second word is how long to
;           delay with it switched off. Both expressed in
;           40kHz cycles (so 40000 = 1 second).
; ---------------------------------------------------------
Play
	di
	ld b,(hl)
	inc hl
PlayLoop
	push bc       ; 11

	ld c,(hl)     ; 7 +
	inc hl        ; 6 +
	ld b,(hl)     ; 7 +
	inc hl        ; 6 = 26
	
	.if CarrierFrequency == 38
	inc bc        ; 6
	.endif

	call PulseOut ; 17 + 4 + 10 = 31
	
	ld c,(hl)     ; 7 +
	inc hl        ; 6 +
	ld b,(hl)     ; 7 +
	inc hl        ; 6 = 26
	
	
	; The "silence" delay is the easier of the two.
	; Therefore, all the fixed-length nonsense from
	; the 40kHz carrier output gets lumped into this
	; delay too! :)
	; 6MHz / 40kHz = 150 CPU cycles / carrier cycle.
	; 6MHz / 38kHz = 158 CPU cycles / carrier cycle.
	; Overhead = 11 + 26 + 31 + 26 + 10 + 13 = 117.
	; Let's fit that 117 into one of the 150 we'd
	; otherwise need to delay by, so waste 33 CCs.
	
	dec bc        ; 6 +
	inc b         ; 4 +
	nop           ; 4 +
	              ; 7 + 12 (exit from Silence) = 33

	; ...just so you know, I really hate doing this.
	; Anyway, we now need to waste 150 clock cycles
	; for each BC.

Silence
	; Loop overhead gives us
	; 4 + 12 + 1 + 10 = 27
	; 4 + 7 + 4 + 12  = 27
	; We need to delay for 150, so: 150 - 27 = 123.
	; (If only it were as simple as 123).
	
	; Set up overhead: 11 + 7 + 8 + 10 = 36.
	; 123 - 36 = 87 / 13 = 6.69..
	push bc       ; 11
	ld b,7        ; 7
-	djnz {-}      ; 13 * (B - 1) + 8
	pop bc        ; 10
	
	; 6 * 13 would give us 78; 87 - 78 = 9 missing.
	xor a         ; 4
	nop           ; 4
	jp nz,Silence ; 1
	
	.if CarrierFrequency == 38
	nop
	nop
	.endif
	
	dec c         ; 4
	jr nz,{+}     ; 12/7
	dec b         ; 4
	jr nz,Silence ; 12/7 <- 7  for "33" delay.
	jr Finished   ; 12   <- 12 for "33" delay.

+       ; 4 + 12 = 16 clock cycles skipped.
        ; However, 11 - 1 = 10 gained.
        ; We need to delay for 11.
        jp z,Silence  ; 1
        jp Silence    ; 10

Finished
	
	pop bc        ; 10
	djnz PlayLoop ; 13
	ret

; ---------------------------------------------------------
; PulseOut -> Outputs a 40kHz signal.
; ---------------------------------------------------------
; Inputs:   bc -> Length of signal in cycles (multiply mS
;                 by 40 to get the actual lenghth).
; ---------------------------------------------------------
PulseOut
	inc b          ; 4
--	push bc        ; 11
	ld a,%11010010 ; 7
	out (bport),a  ; 11
	
	ld b,3         ; 7
-	nop            ; 4
	djnz {-}       ; 12/7
	nop            ; 4

	.if CarrierFrequency == 38
	nop
	.endif
	
	ld a,%11010000 ; 7
	out (bport),a  ; 11
	
	.if CarrierFrequency == 38
	nop
	.endif
	
	pop bc         ; 10
	dec c          ; 4
	jr nz,NoDecB   ; 12/7
	dec b          ; 4
	nop            ; 4
	ld a,0         ; 7
	jp nz,{--}     ; 10/1
	ret            ; 10
NoDecB
	nop            ; 4
	nop            ; 4
	jr {--}        ; 12

; ---------------------------------------------------------
; Record -> Record an IR signal.
; ---------------------------------------------------------
; Inputs:   hl -> location to save recorded signal.
;           b   = maximum number of samples.
; Outputs:  z on success, nz on failure.
; Destroys: af, bc, de, hl.
; Remarks:  Disables interrupts. Will use up 4 bytes per
;           sample, plus one extra byte to store the length
;           of the recoding.
; ---------------------------------------------------------
Record
	di
	push hl
	
	inc hl
	
	ld d,0
	ld e,b
	sla e

	; Taking into consideration the setup and loop overhead,
	; we should dock BC one 40kHz cycle and waste up to 150.

RecLoop
	call PulseIn    ; 17+29
	jr nz,TimedOut  ; 12/7 (Normal: 7; Exit: 12).
	dec bc          ; 6
	ld (hl),c       ; 7
	inc hl          ; 6
	ld (hl),b       ; 7
	inc hl          ; 6

	; How much do we need to delay?
	; 17+29+7+7+6+7+6+4+4+12+6=105
	; 105-99 = 45
	; (45-7-8)/13 = 2.31..
	
	ld b,3          ; 7
-	djnz {-}        ; 13 * (B - 1) + 8

	; 7+13*(3-1)+8 = 41
	
	; We're 4 short;
	nop

	; Back to normal...

	inc d	        ; 4
	dec e           ; 4
	jr nz,RecLoop   ; 12/7 (Normal: 12; Exit: 7).

TimedOut
	pop hl
	srl d
	ld (hl),d
	xor a
	ret
	
; ---------------------------------------------------------
; PulseIn -> Measure the length of a pulse.
; ---------------------------------------------------------
; Inputs:   d  = pulse type to count (high or low).
; Outputs:  bc = length of pulse in 40kHz cycles.
;           z on success, nz on timeout.
; Remarks:  Has an overhead of 29 clock cycles.
; ---------------------------------------------------------
PulseIn
	ld bc,0       ; 10

--	in a,(bport)  ; 11
	push bc       ; 11
	
	
	; In here is the controllable delay.
	; Loop overhead gives us:
	; 11+11+10+4+4+7+6+4+4+12 = 73.
	; We need to pad out to 150 = 77 to waste.
	; 77/13 = 5.92..
	
	ld b,5        ; 7
-	djnz {-}      ; 13 * (B - 1) + 8

	; 13*(6-1)+8 = 73, so four short!

	nop           ; 4
	
	.if CarrierFrequency == 38
	nop \ nop
	.endif

	; Here ends the controllable delay.
	
	pop bc        ; 10
	xor d         ; 4
	rrca          ; 4
	jr c,{+}      ; 12/7 (Normal: 7; Exit: 12).
	inc bc        ; 6
	ld a,b        ; 4
	or c          ; 4
	jr nz,{--}    ; 12/7 (Normal: 12; Exit: 7).
	
	inc b         ; 4
	ret           ; 10

+	xor a         ; 4
	ret           ; 10

; ---------------------------------------------------------
; Decompress -> Decompress a recorded IR signal.
; ---------------------------------------------------------
; Inputs:   hl -> pointer to data.
;           ix -> pointer to protocol descriptor.
;           de -> where to decompress to.
; Format:   <byte>         - flags.
;                            0 = has start bit.
;           <byte>         - data length (bits).
;           <word>         - total signal length.
;           <word>, <word> - low bit (on, off).
;           <word>, <word> - high bit (on, off).
;           <word>, <word> - start bit (on, off).
; ---------------------------------------------------------
Decompress
	push de
	
	; Store length of 'raw' stream;
	ld a,(ix+0)
	and 1
	add a,(ix+1)
	ld (de),a
	inc de
	
	bit 0,(ix+0)
	jr z,NoStartBit

	push ix
	ld b,4

-	ld a,(ix+12)
	ld (de),a
	inc ix
	inc de
	djnz {-}
	pop ix
NoStartBit

	ld b,(ix+1) ; Length in bits.
	ld c,%10000000
DecompressLoop
	push bc
	push ix

	ld b,4
	ld a,(hl)
	and c
	jr nz,Output1

-	ld a,(ix+6)
	ld (de),a
	inc ix
	inc de
	djnz {-}
	jr Outputted0
	
Output1
-	ld a,(ix+8)
	ld (de),a
	inc ix
	inc de
	djnz {-}	
Outputted0

	pop ix
	pop bc
	
	; Move onto next byte?
	rrc c
	jp nc,{+}
	inc hl
+
	djnz DecompressLoop
	
	; Finally; count up how long the entire stream is, then pad.
	ld l,(ix+2)
	ld h,(ix+3)
	
	pop ix
	push de
	push hl
	
	ld a,(ix+0)
	add a,a
	dec a
	ld b,a
	
	ld hl,0
-	ld e,(ix+1)
	ld d,(ix+2)
	add hl,de
	inc ix
	inc ix
	djnz {-}
	
	ex de,hl
	; de = current length.
	pop hl
	; hl = required length.
	
	or a
	sbc hl,de
	; hl = how much to pad.
	
	pop de
	ex de,hl
	
	dec hl
	ld (hl),d
	dec hl
	ld (hl),e	
	ret

; ---------------------------------------------------------
; ReadBits -> Read a bitstream and store it.
; ---------------------------------------------------------
; Inputs:   hl -> scratch RAM area.
;           ix -> pointer to protocol descriptor.
;           de -> where to output bitstream.
; Outputs:  z on success, nz on failure.
; ---------------------------------------------------------
ReadBits
	di
	
	.if Debug
		ld d,1
		call PulseIn
		ld a,$FF
		ld (de),a
		xor a
		ret
	.endif
	
	push hl
	ld a,(ix+0)
	and 1
	add a,(ix+1)
	ld b,a
	inc b
	push de

	ld d,1
	call PulseIn
	jr z,{+}
	pop de
	pop hl
	ret
+
	
	call Record
	pop de
	pop hl
	ret nz
	; Deliberate run-on function

; ---------------------------------------------------------
; Compress -> Compress a recorded IR signal.
; ---------------------------------------------------------
; Inputs:   hl -> pointer to data.
;           ix -> pointer to protocol descriptor.
;           de -> where to compress to.
; Outputs:  z on success, nz on failure.
; ---------------------------------------------------------
Compress
	ld a,(ix+0)
	and 1
	add a,(ix+1)
	cp (hl)
	ret nc ; Not enough data (nz).
	
	inc hl
	
	
	; Did we record a start bit?
	bit 0,(ix+0)
	jr z,{+}
	; No start bit...
	inc hl
	inc hl
	inc hl
	inc hl
+

	push de
	ld a,(ix+1) ; Total number of bits
	add a,7
	srl a
	srl a
	srl a
	ld b,a
	xor a
-	ld (de),a
	inc de
	djnz {-}	
	pop de
	
	ld b,(ix+1)    ; Total <on>, <off> signals.
	ld c,%10000000

CompressLoop

	push bc

	; For each bit we need to work out whether it's closer
	; to a '0' or a '1' signal.
	; This could be interesting, no?
	
	call GetClosestBit
	

	; Done!
	pop bc

	; Write the bit as required...
	jr c,{+}
	ld a,(de)
	or c
	ld (de),a
+
	
	
	; Move onto the next bit:
	rrc c
	jr nc,{+}
	inc de
+	
	djnz CompressLoop
	
	xor a
	ret

; Which bit is closest?
; In:  hl->four bytes, <on time>, <off time>
;      ix->protocol descriptor.
; Out: c : is a 0, nc : is a 1.
;      hl->next four bytes.
GetClosestBit
	.tvar uword[2], Differences
	ld bc,0
	ld (Differences[0]),bc
	ld (Differences[1]),bc

	push ix
	
	push hl
	
	call GetDifference
	ld (Differences[0]),bc
	call GetDifference
	push hl
	ld hl,(Differences[0])
	add hl,bc
	ld (Differences[0]),hl
	pop hl
	
	pop hl
	call GetDifference
	ld (Differences[1]),bc
	call GetDifference
	push hl
	ld hl,(Differences[1])
	add hl,bc
	ld (Differences[1]),hl
	pop hl
	
	pop ix	
	
	push de
	push hl
	ld hl,(Differences[0])
	ld de,(Differences[1])
	or a
	sbc hl,de
	pop hl
	pop de
	ret

; (Internal use only).
GetDifference
	push de

	ld e,(hl)
	inc hl
	ld d,(hl)
	inc hl
	
	push hl
	
	ld l,(ix+4)
	ld h,(ix+5)
				
	inc ix
	inc ix
	or a
	sbc hl,de
	call AbsHL
	ld b,h
	ld c,l

	pop hl
	
	pop de
	ret

AbsHL
	bit 7,h
	ret z
	push af
	ld a,h \ cpl \ ld h,a
	ld a,l \ cpl \ ld l,a	
	inc hl
	pop af
	ret
	
; ---------------------------------------------------------
; DecipherProtocol -> Work out an IR remote's protocol.
; ---------------------------------------------------------
; Inputs:   ix -> Pointer to 16 bytes to store data.
;           hl -> scratch RAM (up to 513 bytes).
; Outputs:  z on success, nz on failure.
; Destroys: af, bc, de, hl.
; Remarks:  Disables interrupts.
; ---------------------------------------------------------
DecipherProtocol

	.if Debug
		ld d,1
		call PulseIn
		ld hl,Dummy
		push ix
		pop de
		ld bc,16
		ldir
		xor a		
		ret
		Dummy
		.db 1 \ .db 43 \ .dw 4230 \ .dw 20,20 \ .dw 20,66 \ .dw 360,180
	.endif

	.tvar uword, ScratchRamLocation
	.tvar uword, TotalSignalLength
	di
	ld (ScratchRamLocation),hl
	
	ld d,1
	call PulseIn
	ret nz
	
	; First and foremost - take a recording that we can analyze.
	ld b,127
	call Record
	
	; Clear the protocol descriptor:
	push ix
	pop hl
	ld b,16
-	ld (hl),0
	inc hl
	djnz {-}
	
	; Check we have a decent amount of input data.
	ld hl,(ScratchRamLocation)
	ld a,(hl)
	cp 8
	ret c
	
	; Assume there is a start bit of some description.
	ld (ix+0),%00000001
	
	; Grab the start bit:
	push ix
	pop hl
	ld de,12
	add hl,de
	ld de,(ScratchRamLocation)
	inc de
	ex de,hl
	ld bc,4
	ldir
	
	; Now we can count up how many data bits there are...
	
	ld hl,(ScratchRamLocation)
	ld b,(hl)
	
	; Skip the start bit:
	dec b
	ld de,5 +2 ; +2 as we only want to count the *delays*
	add hl,de
		
-	push bc
	
	inc (ix+1)
	
	ld c,(hl) \ inc hl
	ld b,(hl)
	ld de,3
	add hl,de

	push hl
	call CheckInterWordDelay
	pop hl
	
	pop bc
	jr c,FinishedBitCount

	djnz {-}
	
FinishedBitCount
	
	; Get total length of signal;
	ld hl,0
	ld (TotalSignalLength),hl
	
	ld hl,(ScratchRamLocation)
	inc hl
	ld a,(ix+0)
	and 1
	add a,(ix+1)
	ld b,a
	sla b
-	ld e,(hl) \ inc hl
	ld d,(hl) \ inc hl
	push hl
	ld hl,(TotalSignalLength)
	add hl,de
	ld (TotalSignalLength),hl
	pop hl
	djnz {-}
	
	ld hl,(TotalSignalLength)
	ld (ix+2),l
	ld (ix+3),h

	; Last of all, we need to calculate the timings of the 0 and 1 bits.

	ld hl,(ScratchRamLocation)
	ld de,5
	add hl,de

	ld b,(ix+1)	
	dec b ; Start bit
	dec b ; Last bit (trails off into nothingness).
	
	push bc
	
	push ix
	pop de

	inc de
	inc de
	inc de
	inc de

	ld bc,8
	ldir
	
	pop bc
	
	dec b ; First two bits
	dec b ; preloaded...
	
	push bc
	push hl

TimeBitsLoop
	push bc
	push ix
		push hl
		call GetClosestBit
		pop hl	
	
		jr c,{+}
		.rept 4
		inc ix
		.loop
	+
	
		ld e,(hl) \ inc hl
		ld d,(hl) \ inc hl
		ld (ix+4),e
		ld (ix+5),d

		ld e,(hl) \ inc hl
		ld d,(hl) \ inc hl
		ld (ix+6),e
		ld (ix+7),d
	pop ix
	pop bc
	djnz TimeBitsLoop
	
	; Final stage: average out results.
	
	pop hl
	pop bc
	
AverageBitsLoop	
	push bc
	push ix
	
		push hl
		call GetClosestBit
		pop hl
		jr c,{+}
		.rept 4
		inc ix
		.loop
	+
		ld e,(hl) \ inc hl
		ld d,(hl) \ inc hl
		push hl
		ld l,(ix+4)
		ld h,(ix+5)
		add hl,de
		srl h \ rr l
		ld (ix+4),l
		ld (ix+5),h
		pop hl
	
		ld e,(hl) \ inc hl
		ld d,(hl) \ inc hl
		push hl
		ld l,(ix+6)
		ld h,(ix+7)
		add hl,de
		srl h \ rr l
		ld (ix+6),l
		ld (ix+7),h
		pop hl
	
	pop ix	
	pop bc
	djnz AverageBitsLoop
	
	xor a
	ret

CheckInterWordDelay
	ld l,(ix+12)
	ld h,(ix+13)
	ld e,(ix+14)
	ld d,(ix+15)
	add hl,de
	or a
	sbc hl,bc
	ret

.echoln "Infrared: {0} bytes.", $-Play
.endmodule