; =========================================================
; Module: Dialogs
; =========================================================
; Description:
;     Module to display pop-up dialogs. These only draw to
;     to the LCD, not plotSScreen, so you can easily use
;     these to save the display, display a dialog, then
;     restore.
; =========================================================

.module Dialogs
; ---------------------------------------------------------
; Draw -> Draws a simple dialog.
; ---------------------------------------------------------
; Inputs:   (l,h) = top left corner coordinate.
;           (e,d) = bottom right corner coordinate.
; Destroys: af, bc, de, hl.
; Remarks:  Writes to saveSScreen.
; ---------------------------------------------------------
Draw
	push hl
	push de
	
	call FillRect
	
	pop de
	pop hl
	push hl
	push de
	
	inc h
	inc l
	inc d
	inc e
	
	call FillRect	
	
	pop de
	pop hl

	inc h
	inc l
	dec d
	dec e

	call ClearRect

	
	ret
	
; ---------------------------------------------------------
; DrawWithTitle -> Draws a dialog with a titlebar.
; ---------------------------------------------------------
; Inputs:   (l,h) = top left corner coordinate.
;           (e,d) = bottom right corner coordinate.
;           bc -> titlebar caption.
; Destroys: af, bc, de, hl.
; Remarks:  Writes to saveSScreen.
; ---------------------------------------------------------
DrawWithTitle
	push bc
	
	push hl
	push de
	call Draw
	pop de
	pop hl
	
	;(l,h)->(e,d)
	ld d,h
	ld a,h
	sub 6
	ld h,a
	push hl
	
	res plotLoc,(iy+plotFlags)
	set bufferOnly,(iy+plotFlag3)

	push hl
	push de
	call FillRect
	pop de
	pop hl
	
	inc e
	ld l,e
	inc h
	inc d
	call FillRect
	
	
	pop hl
	inc l
	ld (penCol),hl
	
	pop hl
	
	res plotLoc,(iy+plotFlags)
	set textInverse,(iy+textFlags)
	res textEraseBelow,(iy+textFlags)
	set textWrite,(iy+sGrFlags)	
	bcall(_VPutS)
	res textInverse,(iy+textFlags)

	ret
	
; ---------------------------------------------------------
; DrawMessageFromTable -> Draws message from a table.
; ---------------------------------------------------------
; Inputs:   hl -> table.
;           a = message index.
; Destroys: af, bc, de, hl.
; Remarks:  Writes to saveSScreen.
; ---------------------------------------------------------
DrawMessageFromTable
	or a
	jr z,FoundMessage
	
	ld b,a

-	ld a,(hl)
	inc hl
	or a
	jr nz,{-}
	djnz {-}
	
FoundMessage
	ld a,(hl)
	dec a
	inc hl
	; Intentional run-on function.

; ---------------------------------------------------------
; DrawMessage -> Draws a message dialog.
; ---------------------------------------------------------
; Inputs:   hl -> message to display on the dialog.
;           a = message type.
; Destroys: af, bc, de, hl.
; Remarks:  Writes to saveSScreen.
; ---------------------------------------------------------
.enum MessageType, Information, Warning, Error
.tvar ubyte, CurrentMessage
.tvar ubyte, WidestString
.tvar ubyte, NumberOfLines
.tvar ubyte, LeftColumn
.tvar ubyte, MessageHeight
.tvar ubyte, TopRow
.tvar ubyte, DialogTop
.tvar ubyte, DialogBottom
.tvar ubyte, TextTopPadding
DrawMessage
	push hl
	
	ld (CurrentMessage),a
	
	xor a
	ld (WidestString),a
	ld (NumberOfLines),a
	ld (TextTopPadding),a
	
	; +2 pixels above.
	; +2 pixels below.
	; +7 titlebar.
	; +1 bottom border.
	; -1 pixel.
	ld a,11
	ld (MessageHeight),a

TryNextLine
	ld a,(NumberOfLines)
	inc a
	ld (NumberOfLines),a
	ld a,(MessageHeight)
	add a,6
	ld (MessageHeight),a
	ld b,0
	ld de,saveSScreen+1
-	ld a,(hl)
	
	or a
	jr z,{+}
	cp "\n"
	jr z,{+}
	
	inc b
	ld (de),a
	inc hl
	inc de
	jr {-}
	
+	ld a,b
	ld (saveSScreen),a
	
	push hl
	ld hl,saveSScreen
	ld a,(hl)
	or a
	jr z,{+}
	bcall(_SStringLength)
+
	pop hl

	; a = width of the error message (in pixels).
	ld b,a
	ld a,(WidestString)
	cp b
	jr nc,{+}
	ld a,b
	ld (WidestString),a
+
	
	ld a,(hl)
	inc hl
	or a
	jr nz,TryNextLine
	
	; Now we know the height, sort out the dialog box positioning:
	ld a,(MessageHeight)
	cp 27
	jr nc,{+}
	
	sub 27
	neg
	srl a
	ld (TextTopPadding),a
	
	ld a,27
+
	ld b,a
	ld a,63
	sub b
	srl a
	ld (DialogTop),a
	add a,b
	ld (DialogBottom),a	
	
	ld a,(WidestString)
	
	add a,17
	ld b,a
	ld a,97
	sub b
	srl a
	; a = offset.
	ld (LeftColumn),a
	ld (penCol),a
	ld hl,(penCol)
	push hl

	sub 3
	ld l,a
	add a,b
	add a,4
	ld e,a
	ld a,(DialogTop)
	add a,6
	ld h,a
	ld a,(DialogBottom)
	dec a
	ld d,a
	
	ld bc,ErrorText
	
	ld a,(CurrentMessage)
	or a
	jr z,{+}

	push de
	ld d,a	
-	ld a,(bc)
	inc bc
	or a
	jr nz,{-}
	dec d
	jr nz,{-}
FoundString
	pop de
	
+	
	call DrawWithTitle

	res plotLoc,(iy+plotFlags)
	set bufferOnly,(iy+plotFlag3)
		
	pop hl
	ld (penCol),hl
	ld a,(DialogTop)
	add a,8
	ld (penRow),a
	
	ld a,(CurrentMessage)
	rrca
	rrca
	rrca
	ld l,a
	ld h,0
	ld de,Icons
	add hl,de
	ld de,(penCol)	
	inc d
	bcall(_DisplayImage)


	pop hl
	ld de,saveSScreen
	
-	ld a,(hl)
	cp "\n"
	jr nz,{+}
	xor a
+	ld (de),a
	inc de
	ld a,(hl)
	inc hl
	or a
	jr nz,{-}
	
	ld hl,saveSScreen
	ld a,(NumberOfLines)
	ld b,a
	
	ld a,(TextTopPadding)
	ld c,a
	ld a,(penRow)
	add a,c
	ld (penRow),a
	
	res plotLoc,(iy+plotFlags)
	res textEraseBelow,(iy+textFlags)
	set textWrite,(iy+sGrFlags)	
	
DrawNextLine
	push bc
	
	; Set up penCol
	ld a,(LeftColumn)
	add a,17
	ld (penCol),a

	ld a,(hl)
	cp "\t"
	jr nz,NotHorizLine
	; Horizontal line!
	push hl

	
	ld a,(penRow)
	add a,3
	ld e,a
	ld a,(penCol)
	call ionGetPixel
	ld c,a
	

	ld a,(WidestString)
	dec a
	ld b,a

	ld a,(penCol)
	ld d,%10101010
	bit 0,a
	jr z,{+}
	rrc d
+

	
-	ld a,c
	and d
	or (hl)
	ld (hl),a
	rrc c
	jr nc,NotRunOffByte
	inc hl
NotRunOffByte
	djnz {-}


	pop hl
	inc hl
	inc hl
	jr DoneHorizontalLine
NotHorizLine


	bcall(_VPutS)
DoneHorizontalLine
	ld a,(penRow)
	add a,6
	ld (penRow),a
	pop bc
	djnz DrawNextLine
	ret

Icons
	.db 15, 15 \ .incbmp "Resources/Information.gif"
	.db 15, 15 \ .incbmp "Resources/Warning.gif"
	.db 15, 15 \ .incbmp "Resources/Error.gif"
	.db 15, 15 \ .incbmp "Resources/Transmission.gif"

ErrorText
	.db "Information",0
	.db "Warning",0
	.db "Error",0
	.db "Infrared",0
	

WaitKey
	bcall(_GetCSC)
-	bcall(_GetCSC)
	cp skEnter
	ret z
	cp skClear
	ret z
	cp sk2nd
	ret z
	jr {-}

.var ubyte[5], SavedFlags
SaveFlags
	ld a,(iy+plotFlags)
	ld (SavedFlags[0]),a
	ld a,(iy+plotFlag3)
	ld (SavedFlags[1]),a
	ld a,(iy+sGrFlags)
	ld (SavedFlags[2]),a
	ld a,(iy+textFlags)
	ld (SavedFlags[3]),a
	ld a,(iy+apiFlg4)
	ld (SavedFlags[4]),a
	ret
ReloadFlags
	ld a,(SavedFlags[0])
	ld (iy+plotFlags),a
	ld a,(SavedFlags[1])
	ld (iy+plotFlag3),a
	ld a,(SavedFlags[2])
	ld (iy+sGrFlags),a
	ld a,(SavedFlags[3])
	ld (iy+textFlags),a
	ld a,(SavedFlags[4])
	ld (iy+apiFlg4),a
	ret


SetUpRect
.tvar uword[2], Corners
.tvar ubyte[2], LineCaps
.tvar uword[2], CornerAddresses
.tvar ubyte, RectHeight

	; (l,h)->(e,d)
	ld (Corners[0]),hl
	ld (Corners[1]),de

	ld a,e
	ld e,h
	call ionGetPixel
	ld (CornerAddresses[1]),hl

	ld b,a
-	or b
	sla b
	jr nc,{-}
	
	ld (LineCaps[1]),a
	
	ld de,(Corners[0])
	ld a,e
	ld e,d
	call ionGetPixel
	ld (CornerAddresses[0]),hl	
	
	ld b,a
-	or b
	srl b
	jr nc,{-}
	
	ld (LineCaps[0]),a
	
	; Handle the special case where both caps end up in the SAME byte.
	
	or a
	ld hl,(CornerAddresses[0])
	ld de,(CornerAddresses[1])
	sbc hl,de
	jr nz,{+}
	ld a,(LineCaps[0])
	ld b,a
	ld a,(LineCaps[1])
	and b
	ld (LineCaps[0]),a
	ld (LineCaps[1]),a
+
	
	
	ld a,(Corners[0]+1)
	ld b,a
	ld a,(Corners[1]+1)
	sub b
	inc a
	ld (RectHeight),a
	ld b,a
	
	ret
	
	
.for x,0,1

.if x == 0
ClearRect
.else
FillRect
.endif
	call SetupRect
	push bc
	
	ld de,12
	ld hl,(CornerAddresses[0])
	ld a,(LineCaps[0])
	ld c,a
	call FillRectCap{x}
	
	pop bc
	push bc
	
	ld hl,(CornerAddresses[1])	
	ld a,(LineCaps[1])
	ld c,a
	call FillRectCap{x}

	pop bc
	ld a,-x
	
	jr FillRectMiddle
	

FillRectCap{x}
-	ld a,c
	.if x == 0
		cpl
		and (hl)
	.else
		or (hl)
	.endif
	ld (hl),a
	add hl,de
	djnz {-}
	ret
.loop

FillRectMiddle
	or a
	ld hl,(CornerAddresses[1])
	ld de,(CornerAddresses[0])
	sbc hl,de
	ret z
	
	ld c,b

	ld b,l
	dec b
	ret z
	
	ld hl,12
	inc de
	ex de,hl
	
--
	push bc
	push hl

-	ld (hl),a
	inc hl
	djnz {-}
	
	pop hl
	pop bc
	
	add hl,de
	dec c
	jr nz,{--}
	ret
	

Dither
	ld hl,plotSScreen
	ld e,$55
	ld c,64
	
--	ld b,12
-	ld a,(hl)
	and e
	ld (hl),a
	inc hl
	djnz {-}
	ld a,e
	cpl
	ld e,a
	dec c
	jr nz,{--}
	ret

.echoln "Dialogs: {0} bytes.", $-Draw
.endmodule