/************************************************************************/
/************************************************************************/
/*																		*/
/*	Name :		SHIP.C													*/
/*	Project :	Elite type game in 'C'/Z80 Assembly						*/
/*	Author :	Paul Robson												*/
/*	Created :	3rd May 2001											*/
/*	Function :	Code for drawing ships etc., their movement and attacks	*/
/*	Changes :															*/
/*																		*/
/************************************************************************/
/************************************************************************/

#include "anarchy.h"

static void _SHIPRetarget(OBJECT *o);
static void _SHIPMove(GAME *g,OBJECT *o,int dx,int dy);
static void _SHIPMoveDraw(GAME *g,OBJECT *o,int dx,int dy);

/************************************************************************/
/*																		*/
/*	  Draw an object of the given type at the given relative position	*/
/*																		*/
/************************************************************************/

void SHIPDraw(OBJECT *o)
{
	int x,y,z,x1,y1,w;
	if (abs(o->z) > 40) return;				/* Out of range */

	if (o->Picture >= 0)
	{
		x1 = o->x >> 2;y1 = o->y >> 1;		/* Draw on the scanner */
		if (y1 > 3) y1 = 3;
		if (y1 < -3) y1 = -3;
		if (x1 > 7) x1 = 7;
		if (x1 < -7) x1 = -7;
		IOPlot(88+x1,60+y1,COL_RED);
	}
	if (o->z < 0) return;					/* Don't draw stuff behind */

	x = o->x;y = o->y;z = o->z;				/* Translate coordinates */
	if (!SHIPTranslate(&x,&y,&z)) return;

	if (o->Firing && o->z < 40)				/* If object firing draw it */
		IOLine(x,y,x,56,COL_LASER);

	z = z >> 1;if (z < 0) z = 0;			/* FAKE DRAWING ROUTINE ! */
	if (o->Picture == 0)
	{
		IOLine(x-z,y-z,x-z,y+z,o->Colour);
		IOLine(x+z,y-z,x+z,y+z,o->Colour);
		IOLine(x-z,y,x+z,y,o->Colour);
	}
	else
	{
		IOLine(x,y-z,x-z,y+z,o->Colour);
		IOLine(x,y-z,x+z,y+z,o->Colour);
		IOLine(x-z,y+z,x+z,y+z,o->Colour);
	}
}

/************************************************************************/
/*																		*/
/*						Translate coordinates							*/
/*																		*/
/************************************************************************/

int SHIPTranslate(int *x,int *y,int *z)
{
	int w = 33-*z;
	if (w < 1) w = 1;
	*x = 4*(*x)*w/32;						/* Coordinate scale for distance */
	*y = 4*(*y)*w/32;
	*z = 8-(*z >> 2);						/* Object size */
	if (abs(*x)+*z > 48) return 0;			/* Off the screen */
	if (abs(*y)+*z > 32) return 0;
	*x += 48;*y += 32;						/* Centre it */
	return 1;
}

/************************************************************************/
/*																		*/
/*	  Draw an object of the given type at the given relative position	*/
/*																		*/
/************************************************************************/

void SHIPMoveDraw(GAME *g)
{
	int i;
	g->AnyLeft = 0;
	for (i = 0;i < g->ObjCount;i++)			/* Go through all ships */
		if (g->Obj[i].InUse)				/* Move & Draw if active */
		{
			_SHIPMoveDraw(g,&(g->Obj[i]),g->dx,g->dy);
			g->AnyLeft = 1;					/* & set active flag */
		}
}

static void _SHIPMoveDraw(GAME *g,OBJECT *o,int dx,int dy)
{
	int n;
	if (o->InUse == 0) return;				/* No ship */
	if (g->Firing &&						/* Hit the ship */
			abs(o->x) < 3 && abs(o->y) < 3 && o->z < 30 &&
			(g->Frame & 7) == 4)
	{
		n = (33-o->z) << g->Laser;			/* Damage done */
		n = n >> (abs(o->x)+abs(o->y)/2);	/* Adjust for distance */
		o->Energy -= n*2;					/* Lose Energy */
		o->Friendly = 0;					/* Not friendly */
	}
	if (g->Firing && o->Pack)				/* Don't join in ! */
						o->Friendly = 0;

	if (g->MissileInFlight)					/* Hit by missile */
		if (g->Missile.x == o->x &&
			g->Missile.y == o->y &&
			g->Missile.z == o->z)
							g->MissileInFlight = o->Energy = 0;

	if (o->Energy <= 0)						/* Deaded */
	{
		o->InUse = 0;       				/* Disable ship */
		g->ExplodeX = o->x;					/* Set up for explosion */
		g->ExplodeY = o->y;
		g->ExplodeZ = o->z;
		g->ExplodeCount = 1;
		g->Cash += o->Power << 1;			/* Get money for blowing em up */
		return;
	}

	_SHIPMove(g,o,dx,dy);					/* Move it */
	n = (2<<(o->FireRate))-1;				/* Firing mask */
	o->Firing = ((RANDBYTE&n) == 0) &&	/* Find out if firing */
								(o->Friendly == 0);
	if (o->Firing && o->z < 32)				/* If firing, damages player */
			g->Energy -= ((40-o->z) << o->Power) >> g->Shield;
	SHIPDraw(o);							/* Draw it */
}

/************************************************************************/
/*																		*/
/*					Move a point towards a target						*/
/*																		*/
/************************************************************************/

void SHIPChase(int *xp,int xt)
{
	if (*xp == xt) return;
	if (*xp < xt) (*xp)++;
	else (*xp)--;
}

/************************************************************************/
/*																		*/
/*							Retarget a ship								*/
/*																		*/
/************************************************************************/

static void _SHIPRetarget(OBJECT *o)
{
	if (o->Friendly)						/* If friendly, pass by */
	{
		if (o->z < 0) o->InUse = 0;			/* "Kill off" if behind */
		o->tx = o->x;						/* Set new target behind */
		o->ty = o->y;
		o->tz = -40;
	}
	else									/* Else pick random target */
	{
		o->tx = RANDBYTE%32-16;
		o->ty = RANDBYTE%32-16;
		o->tz = RANDBYTE%32+4;
	}
}

/************************************************************************/
/*																		*/
/*								Move a ship								*/
/*																		*/
/************************************************************************/

static void _SHIPMove(GAME *g,OBJECT *o,int dx,int dy)
{
	if (o->x == o->tx && o->y == o->ty)		/* If reached target, retarget */
								_SHIPRetarget(o);
	if (((g->Frame & 0xFFF) &				/* Own movement */
							o->Speed) == 0)
	{
		SHIPChase(&(o->x),o->tx);
		SHIPChase(&(o->y),o->ty);
		SHIPChase(&(o->z),o->tz);
	}
	o->x -= dx;o->y -= dy;					/* Player movement */
	o->tx -= dx;o->ty -= dy;
}

/************************************************************************/
/*																		*/
/*							Create a ship group							*/
/*																		*/
/************************************************************************/

void SHIPCreateGroup(GAME *g)
{
	int i;
	char Group;

	g->GroupsLeft--;						/* One fewer remaining */
	i = g->Anarchy * 3 / 2+1;				/* Max no of ships */
	do                 						/* Calculate ship count */
		g->ObjCount = (rand() & 15)+1;
	while (g->ObjCount > i || g->ObjCount > MAX_OBJECTS);

	Group = 'V';							/* Default : Villains */
	if ((rand() & 15) > 6+g->Anarchy)		/* Maybe not... */
	{
		Group = 'W';						/* Just wanderers */
		if ((rand() & 3) == 0) Group = 'C';	/* Maybe a convoy */
	}

	for (i = 0;i < g->ObjCount;i++)			/* Create them */
	{
		g->Obj[i].x = rand()%16-8;			/* Same for all ships */
		g->Obj[i].y = rand()%16-8;
		g->Obj[i].z = 70;
		g->Obj[i].tx = g->Obj[i].ty = g->Obj[i].tz = 0;
		g->Obj[i].InUse = 1;

		g->Obj[i].Friendly = (Group != 'V');
		g->Obj[i].Pack = (Group == 'C');

		SHIPCreate(g,&(g->Obj[i]));
		_SHIPRetarget(&(g->Obj[i]));
	}
}

/************************************************************************/
/*																		*/
/*						Create an appropriate ship						*/
/*																		*/
/************************************************************************/

void SHIPCreate(GAME *g,OBJECT *o)
{
	o->Colour = RANDBYTE%7+1;
	o->Picture = 0;
	o->Speed = 3;
	o->Power = 1+(g->Anarchy)/2;
	o->Energy = 20;
	o->FireRate = 4;
}

/************************************************************************/
/*																		*/
/*					 Do the explosion drawing							*/
/*																		*/
/************************************************************************/

void SHIPExplodeDraw(GAME *g)
{
	int n,x,y,z;
	if (g->ExplodeCount == 0) return;
	g->ExplodeCount = (g->ExplodeCount+1) & 15;
	n = g->ExplodeCount;
	x = g->ExplodeX;y = g->ExplodeY;z = g->ExplodeZ;
	if (SHIPTranslate(&x,&y,&z) == 0) return;
	n = n/2;
	IOPlot(x+n,y+n,COL_EXPLODE);
	IOPlot(x-n,y-n,COL_EXPLODE);
	IOPlot(x+n,y-n,COL_EXPLODE);
	IOPlot(x-n,y+n,COL_EXPLODE);
	IOPlot(x+n,y+0,COL_EXPLODE);
	IOPlot(x-n,y-0,COL_EXPLODE);
	IOPlot(x+0,y-n,COL_EXPLODE);
	IOPlot(x-0,y+n,COL_EXPLODE);
}