; ##########################################
;
;   Z80 graphics routines by Patai Gergely
; designed for use with the TI-83 calculator
;
;          Last update: 2003/02/15
;
; ##########################################
;
; # Some remarks #
;
; Note that these routines are not prepared for wrong input!
; I decided not to implement error handling, because my primary
; target was to create a library with fast graphing capabilities.
; As you can see, I haven't defined where the virtual screen
; should be. This way you can effectively combine these routines
; to create complex pictures without too much extra effort.
; Another thing to notice is the use of absolute jumps
; practically everywhere. I know that I waste one byte per
; jump, but a significant amount of CPU time is saved this way,
; leading to spectacular improvement in the speed of inner loops.
; Ideas for improvements and possible bugreports are welcome and
; will be credited if anyone decides to add one or two touches
; to this collection.
;
; You may freely use this package as long as you credit me.
;
; # Legal stuff #
;
; I cannot be held responsible for any damage caused by these
; pieces of code. I tested them with the utmost enthusiasm
; and found them to be completely bug-free on my calculator.
;
; # An overview of the routines #
;
; ClearScreen:
; input: HL=virtual screen address
; output: fills 768 bytes at HL with zeroes
;
; ###
;
; InvertScreen:
; input: HL=virtual screen address
; output: complements every byte of the virtual screen
;
; ###
;
; FlipScreen (a replica of ION FastCopy by Joe Wingbermuehle):
; input: HL=virtual screen address
; output: displays the virtual screen on the LCD (non-destructively)
;
; ###
;
; DrawLine:
; input: D=x coordinate of the first point (0-95)
;        E=y coordinate of the first point (0-63)
;        H=x coordinate of the second point (0-95)
;        L=y coordinate of the second point (0-63)
;        IX=address of the screen on which the line will be drawn
; output: draws an unclipped line from (D,E) to (H,L) on the screen
;         pointed by IX
;
; ###
;
; DrawClippedLine:
; input: D=x coordinate of the first point (visible between -48 and 47)
;        E=y coordinate of the first point (visible between -32 and 31)
;        H=x coordinate of the second point (visible between -48 and 47)
;        L=y coordinate of the second point (visible between -32 and 31)
;        IX=address of the screen on which the line will be drawn
; output: draws an unclipped line from (D,E) to (H,L) on the screen
;         pointed by IX
;
; ###
;
; ShadeBlack:
; input: BC=darkness (0-8; 0 is white, 8 is black)
;        HL=pointer to the virtual screen
; output: makes the image on the virtual screen brighter depending
;         on the value of BC (leaves intact if it equals 8) by
;         ANDing it with a shade mask. (Try it on a completely black
;         screen to see the patterns.) Might be useful...
;
; ###
;
; DrawPolygon (Uses ClearScreen, DrawLine and DrawHorizontal):
; input: The data at PolyData (equate it to a safe area!) as follows:
;        PD+0: word - pointer to work screen (768 bytes to be destroyed)
;        PD+2: word - pointer to destination screen (presumably your virtual screen)
;        PD+4: byte - the shade of the polygon from 0 (white) to 8 (black);
;                     values greater than eight will make various patterns (may vary)
;        PD+5: byte - the number of vertices (N) minus 1, e.g. 2 for a triangle (as N=3);
;                     a zero denotes a 257-gon
;        PD+6: words - N words (2*N bytes) containing the coordinates of the vertices;
;                      the high bytes (PD+odd) are the x coordinates (0-95) and
;                      the low bytes (PD+even) are the y coordinates (0-63)
; output: draws a convex polygon over the contents of the destination screen. As a
;         side effect, the contours of same polygon appear on the work screen
;         in front of a white background.
;
; ###
;
; DrawHorizontal: it should not be called separately. It's called from the polygon
;                 drawer, and it draws one scanline of the polygon using the data
;                 given at PolyData.
;
; ###
;
; ScrollRight:
; input: A=pixel counter
;        HL=pointer to the virtual screen
; output: scrolls the screen found at HL right by a number of pixels specified in A
;
; ###
;
; ScrollLeft:
; input: A=pixel counter
;        HL=pointer to the virtual screen
; output: scrolls the screen found at HL left by a number of pixels specified in A
;
; ###
;
; AlignSprite: 
; input: B=sprite Y size
;        C=sprite X size
;        DE=direct screen address of the sprite
;        HL=pointer to sprite data
; output: draws an aligned sprite at the specified address; useful for
;         displaying little picture elements. After execution, HL points
;         to the data immediately following the sprite, and DE points
;         to the byte immediately below its bottom left corner.
;
; ###
;
; AlignMaskedSprite: 
; input: B=sprite Y size
;        C=sprite X size
;        DE=direct screen address of the sprite
;        HL=pointer to sprite data
; output: draws an aligned masked sprite at the specified address;
;         the sprite bytes and the mask bytes are interleaved - each
;         sprite byte is preceded by its corresponding mask byte
;         (first the mask is ANDed, then the sprite is ORed to the
;         contents of the screen)
;
; ##############################################################################################

ClearScreen:		; Clearing the virtual screen at HL (~6500 cycles)
 ld (DL_VLinc+1),sp	; Backing up SP to a safe place
CS_hackSize1:
 ld de,768
 add hl,de
 ld sp,hl
 ld hl,0
CS_hackSize2:
 ld b,48
CS_loop:
 push hl		; 16 bytes are cleared in 8*11+13=101 cycles
 push hl		; That would be 16*21=336 with LDIR and 16*16+13=269 with LDIs
 push hl
 push hl
 push hl
 push hl
 push hl
 push hl
 djnz CS_loop
 ld sp,(DL_VLinc+1)
 ret

; ##############################################################################################

ClearLines:		; Clearing 12*BC bytes from HL (B must be 0)
 ld (DL_VLinc+1),sp	; Backing up SP to a safe place
 ld a,c
 sla c
 sla c
 add hl,bc
 add hl,bc
 add hl,bc
 ld sp,hl
 ld h,b
 ld l,b
 ld b,a
CL_loop:
 push hl
 push hl
 push hl
 push hl
 push hl
 push hl
 djnz CL_loop
 ld sp,(DL_VLinc+1)
 ret

; ##############################################################################################

FlipScreen:		; Copies the screen at HL to the LCD in ~52000 cycles (ION FastCopy)
 ld a,$80
 out ($10),a
 ld de,755
 add hl,de
 ld a,$20
 ld c,a
 inc hl
 dec hl
FS_column:
 ld b,64
 inc c
 ld de,-767
 out ($10),a
 add hl,de
 ld de,10
FS_inner:
 add hl,de
 inc hl
 inc hl
 inc de
 ld a,(hl)
 out ($11),a
 dec de
 djnz FS_inner
 ld a,c
 cp $2c
 jp nz,FS_column
 ret

; ##############################################################################################

DrawLine:		; This routine draws an unclipped line on an IX-pointed screen from (d,e) to (h,l)
 ld a,h			; Calculating delta X and swapping points if negative
 sub d			; (Lines are always drawn from left to right)
 jp nc,DL_okaydx
 ex de,hl
 neg
DL_okaydx:
 push af		; Saving DX (it will be popped into DE below)
 ld b,0			; Calculating the position of the first pixel to be drawn
 ld c,d			; IX+=D/8+E*12 (actually E*4+E*4+E*4)
 srl c
 srl c
 srl c
 add ix,bc
 ld c,e
 sla c
 sla c
 add ix,bc
 add ix,bc
 add ix,bc
 ld a,d			; Calculating the starting pixel mask
 ld c,$80
 and 7
 jp z,DL_okaymask
DL_calcmask:
 srl c
 dec a
 jp nz,DL_calcmask
DL_okaymask:
 ld a,l			; Calculating delta Y and negating the Y increment if necessary
 sub e			; This is the last instruction for which we need the original data
 ld hl,12
 jp nc,DL_okaydy
 ld hl,-12
 neg
DL_okaydy:
 pop de			; Recalling DX
 ld e,a			; D=DX, E=DY
 cp d
 jp c,DL_horizontal	; Line is rather horizontal than vertical
DL_vertical:		; This label is jumped to from DrawTrackLine only
 ld (DL_VLinc+1),hl	; Modifying y increment
 push ix		; Loading IX to HL for speed; we don't need the old value of HL any more
 pop hl
 ld b,e			; Pixel counter
 inc b
 srl a			; Setting up gradient counter (A=E/2)
 ld (DL_VrecallSP+1),sp	; Backing up SP to a safe place
DL_VLinc:
 ld sp,0		; This value is replaced by +/- 12
DL_Vloop:
 ex af,af'		; Saving A to its shadow register
 ld a,(hl)
 or c			; Writing pixel to current position
 ld (hl),a
 ex af,af'		; Recalling A (faster than push-pop, and there's no need for SP)
 add hl,sp		; Vertical movement
 sub d			; Handling gradient
 jp nc,DL_VnoSideStep
 rrc c			; Rotating mask
 jp nc,DL_VnoByte	; Handling byte boundary
 inc hl
DL_VnoByte:
 add a,e		; Handling gradient
DL_VnoSideStep:
 djnz DL_Vloop
DL_VrecallSP:
 ld sp,0
 ret
DL_horizontal:
 ld (DL_HLinc+1),hl	; Modifying y increment
 push ix		; Loading IX to HL for speed; we don't need the old value of HL any more
 pop hl
 ld b,d			; Pixel counter
 inc b
 ld a,d			; Setting up gradient counter
 srl a
 ld (DL_HrecallSP+1),sp	; Backing up SP to a safe place
DL_HLinc:
 ld sp,0		; This value is replaced by +/- 12
DL_Hloop:
 ex af,af'		; Saving A to its shadow register
 ld a,(hl)
 or c			; Writing pixel to current position
 ld (hl),a
 ex af,af'		; Recalling A
 rrc c			; Rotating mask
 jp nc,DL_HnoByte	; Handling byte boundary
 inc hl
DL_HnoByte:
 sub e			; Handling gradient
 jp nc,DL_HnoSideStep
 add hl,sp		; Vertical movement
 add a,d		; Handling gradient
DL_HnoSideStep:
 djnz DL_Hloop
DL_HrecallSP:
 ld sp,0
 ret

; ##############################################################################################

DrawPolygon:		; Draws a convex shaded polygon according to PolyData
 ld hl,PolyData+5	; Finding the maximum y value and storing it to B
 ld b,(hl)
 inc hl
 ld a,(hl)
 inc hl
 inc hl
DP_yMax:
 cp (hl)
 jr nc,DP_yMaxOK
 ld a,(hl)
DP_yMaxOK:
 inc hl
 inc hl
 djnz DP_yMax
 ld b,a
 ld hl,PolyData+5	; Finding the minimum y value and storing it to C
 ld c,(hl)
 inc hl
 ld a,(hl)
 inc hl
 inc hl
DP_yMin:
 cp (hl)
 jr c,DP_yMinOK
 ld a,(hl)
DP_yMinOK:
 inc hl
 inc hl
 dec c
 jr nz,DP_yMin
 ld c,a
 push bc
 ld hl,(PolyData)	; Clearing the relevant lines of the work screen
 ld e,c
 ld d,0
 sla e
 sla e
 add hl,de
 add hl,de
 add hl,de
 ld a,b
 sub c
 inc a
 ld c,a
 ld b,d
 call ClearLines
 ld hl,PolyData+5	; Drawing the edges
 ld b,(hl)
 inc hl
 ld e,(hl)
 inc hl
 ld d,(hl)
 inc hl
 push de
DP_lines:
 push bc
 ld c,(hl)
 inc hl
 ld b,(hl)
 inc hl
 ld ix,(PolyData)
 push hl
 push bc
 push bc
 pop hl
 call DrawLine
 pop de
 pop hl
 pop bc
 djnz DP_lines
 pop hl
 ld ix,(PolyData)
 call DrawLine
 pop bc
DP_fill:		; Filling the lines from C to B
 push bc
 ld hl,(PolyData)
 ld b,0
 call DrawHorizontal
 pop bc
 inc c
 ld a,c
 cp b
 jr c,DP_fill
 ret

; ##############################################################################################

DrawHorizontal:		; Draws a horizontal line between the first and the last set pixel
 ld hl,(PolyData)	; Line number is in BC (0-63)
 ld a,c
 ld (DL_VLinc+2),a
 sla c			; Address of first pixel of line to HL and that of last pixel to DE
 sla c
 add hl,bc
 add hl,bc
 add hl,bc
 push hl
 ld bc,11
 add hl,bc
 ex de,hl
 pop hl
 xor a			; Looking for left end (will be HL)
 ld b,11
DH_SeekLeft:
 cp (hl)
 jr nz,DH_LeftFound
 inc hl
 djnz DH_SeekLeft
 cp (hl)
 ret nz
DH_LeftFound:
 ex de,hl		; Looking for right end (will be DE)
 ld b,11
DH_SeekRight:
 cp (hl)
 jr nz,DH_RightFound
 dec hl
 djnz DH_SeekRight
DH_RightFound:
 ex de,hl
 ld b,a			; Choosing the AND masks for both ends
 ld c,a
 ld a,(hl)
 sla a
 jr c,DH_LeftMask
 inc b
 sla a
 jr c,DH_LeftMask
 inc b
 sla a
 jr c,DH_LeftMask
 inc b
 sla a
 jr c,DH_LeftMask
 inc b
 sla a
 jr c,DH_LeftMask
 inc b
 sla a
 jr c,DH_LeftMask
 inc b
 sla a
 jr c,DH_LeftMask
 inc b
DH_LeftMask:
 ld a,(de)
 sra a
 jr c,DH_RightMask
 inc c
 sra a
 jr c,DH_RightMask
 inc c
 sra a
 jr c,DH_RightMask
 inc c
 sra a
 jr c,DH_RightMask
 inc c
 sra a
 jr c,DH_RightMask
 inc c
 sra a
 jr c,DH_RightMask
 inc c
 sra a
 jr c,DH_RightMask
 inc c
DH_RightMask:
 ld ix,LeftMask		; We have the masks and the addresses, so we draw the line at last...
 push bc
 ld c,b
 ld b,0
 add ix,bc
 pop bc
 ld a,(ix)
 ld (DL_VLinc+1),a
 ld b,0
 ld ix,RightMask
 add ix,bc
 push hl
 ld bc,(PolyData+4)
 ld b,0
 ld hl,GrayShades
 add hl,bc
 add hl,bc
 inc hl
 ld c,0
 ld a,(DL_VLinc+2)
 rra
 sbc hl,bc
 ld a,(hl)
 pop hl
 ld bc,(PolyData)
 or a
 sbc hl,bc
 ex de,hl
 or a
 sbc hl,bc
 ex de,hl
 ld bc,(PolyData+2)
 add hl,bc
 ex de,hl
 add hl,bc
 ex de,hl
 ld b,a
 ld a,(DL_VLinc+1)
 ld d,a
 ld a,l			; An odd way of checking the end of the line...
 cp e
 jr z,DH_endline
 ld a,d
 and (hl)
 ld c,a
 ld a,d
 cpl
 and b
 or c
 ld (hl),a
 inc hl
 ld d,0
DH_loop:
 ld a,l			; An odd way of checking the end of the line again...
 cp e
 jr z,DH_endline
 ld (hl),b
 inc hl
 jr DH_loop
DH_endline:
 ld a,d
 or (ix)
 and (hl)
 ld c,a
 ld a,d
 or (ix)
 cpl
 and b
 or c
 ld (hl),a
 ret

; ##############################################################################################

ScrollUniversal:		; Moving whole bytes; screen is in HL, difference in BC
 push hl			; e.g.: BC=1 -> left by 8 pixels, BC=-24 -> down by 2 pixels,
 pop de				; BC=-3 -> right by 24 pixels, BC=36 -> up by 3 pixels etc.
 add hl,bc
 ld a,b
 ld bc,768
 rla
 ld a,$b0			; LDIR
 jp nc,SU_dirOK
 ld a,$b8			; LDDR
 ex de,hl
 add hl,bc
 ex de,hl
 add hl,bc
SU_dirOK:
 ld (SU_copy+1),a
SU_copy: 
 ldir
 ret

; ##############################################################################################

ScrollLeft:			; Scrolling the HL screen left by A pixels
 cp 1
 jp nz,ScrL_more
 ld b,64
 ld de,767
 add hl,de
ScrL_1line:
 rl (hl) \ dec hl \ rl (hl) \ dec hl \ rl (hl) \ dec hl \ rl (hl) \ dec hl
 rl (hl) \ dec hl \ rl (hl) \ dec hl \ rl (hl) \ dec hl \ rl (hl) \ dec hl
 rl (hl) \ dec hl \ rl (hl) \ dec hl \ rl (hl) \ dec hl \ rl (hl) \ dec hl
 djnz ScrL_1line
 ret
ScrL_more:
 ld (DL_VLinc+1),sp
 ld sp,hl
 ld (ScrL_pixels+1),a
 ld a,64
ScrL_line:			; One line from left to right: C' B' E' D' L' H' C B E D L H
 ex af,af'			; (apostrophes are interchanged by every EXX)
 pop bc \ pop de \ pop hl \ exx \ pop bc \ pop de \ pop hl
ScrL_pixels:
 ld a,0
ScrL_dot:
 sla h \ rl l \ rl d \ rl e \ rl b \ rl c \ exx
 rl h \ rl l \ rl d \ rl e \ rl b \ rl c \ exx
 dec a
 jp nz,ScrL_dot
 push hl \ push de \ push bc \ exx \ push hl \ push de \ push bc
 ld hl,12
 add hl,sp
 ld sp,hl
 ex af,af'
 dec a
 jp nz,ScrL_line
 ld sp,(DL_VLinc+1)
 ret

; ##############################################################################################

ScrollRight:			; Scrolling the HL screen right by A pixels
 cp 1
 jp nz,ScrR_more
 ld b,64
ScrR_1line:
 rr (hl) \ inc hl \ rr (hl) \ inc hl \ rr (hl) \ inc hl \ rr (hl) \ inc hl
 rr (hl) \ inc hl \ rr (hl) \ inc hl \ rr (hl) \ inc hl \ rr (hl) \ inc hl
 rr (hl) \ inc hl \ rr (hl) \ inc hl \ rr (hl) \ inc hl \ rr (hl) \ inc hl
 djnz ScrR_1line
 ret
ScrR_more:
 ld (DL_VLinc+1),sp
 ld sp,hl
 ld (ScrR_pixels+1),a
 ld a,64 
ScrR_line:
 ex af,af'
 pop bc \ pop de \ pop hl \ exx \ pop bc \ pop de \ pop hl
ScrR_pixels:
 ld a,0
ScrR_dot:
 exx \ srl c \ rr b \ rr e \ rr d \ rr l \ rr h
 exx \ rr c \ rr b \ rr e \ rr d \ rr l \ rr h
 dec a
 jp nz,ScrR_dot
 push hl \ push de \ push bc \ exx \ push hl \ push de \ push bc
 ld hl,12
 add hl,sp
 ld sp,hl
 ex af,af'
 dec a
 jp nz,ScrR_line
 ld sp,(DL_VLinc+1)
 ret

; ##############################################################################################

AlignSprite:		; Draws a sprite at HL with size B (vert), C (horz) to DE
 push bc
 ld b,0
 push de
 ldir
 pop de
 ld c,12
 ex de,hl
 add hl,bc
 ex de,hl
 pop bc
 djnz AlignSprite
 ret

; ##############################################################################################

AlignMaskedSprite:	; Draws a masked sprite at HL with size B (vert), C (horz) to DE
 push bc
 ld b,c
 push de
AMS_loop:
 ld a,(de)
 and (hl)
 inc hl
 or (hl)
 inc hl
 ld (de),a
 inc de
 djnz AMS_loop 
 pop de
 ld c,12
 ex de,hl
 add hl,bc
 ex de,hl
 pop bc
 djnz AlignMaskedSprite
 ret
