/*
 * Raw binary packing tool
 *
 * Copyright (c) 2010 Benjamin Moody <floppusmaximus@users.sf.net>
 *
 * This program is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "pack.h"
#include "utils.h"

static const char usage[] = "Usage: %s -a addr [binfile] [-o outfile]\n";

int main(int argc, char** argv)
{
  const char* infilename = NULL;
  const char* outfilename = NULL;
  unsigned int start = 0;
  FILE *inf, *outf;
  unsigned char* data;
  unsigned char* packed;
  unsigned int length, n, packed_size;
  int i;

  for (i = 1; i < argc; i++) {
    if (argv[i][0] != '-' || argv[i][1] == 0)
      infilename = argv[i];
    else {
      if (argv[i][1] == 'o') {
	if (argv[i][2])
	  outfilename = &argv[i][2];
	else {
	  outfilename = argv[++i];
	}
      }
      else if (argv[i][1] == 'a') {
	if (argv[i][2])
	  sscanf(&argv[i][2], "%i", &start);
	else
	  sscanf(argv[++i], "%i", &start);
      }
      else {
	fprintf(stderr, "%s: unknown option %s\n", argv[0], argv[i]);
	fprintf(stderr, usage, argv[0]);
	return 1;
      }
    }
  }

  if (!start) {
    fprintf(stderr, "%s: no starting address specified\n", argv[0]);
    fprintf(stderr, usage, argv[0]);
    return 1;
  }

  if (start < 0xC000 || start > 0xFFFF) {
    fprintf(stderr, "%s: starting address is not in RAM\n", argv[0]);
    return 1;
  }

  if (infilename && strcmp(infilename, "-")) {
    inf = fopen(infilename, "rb");
    if (!inf) {
      perror(infilename);
      return 1;
    }
  }
  else
    inf = stdin;

  length = 0;
  data = xmalloc(length + 1024);

  do {
    n = fread(data + length, 1, 1024, inf);
    length += n;
    data = xrealloc(data, length + 1024);
  } while (n == 1024);

  if (inf != stdin)
    fclose(inf);

  pack_asm_data(start, data, length, &packed, &packed_size);

  if (outfilename && strcmp(outfilename, "-")) {
    outf = fopen(outfilename, "wb");
    if (!outf) {
      perror(outfilename);
      xfree(data);
      xfree(packed);
      return 1;
    }
  }
  else
    outf = stdout;

  fwrite(packed, 1, packed_size, outf);

  if (outf != stdout)
    fclose(outf);

  xfree(data);
  xfree(packed);
  return 0;
}
