/*
 * Miniasm
 *
 * Copyright (c) 2010 Benjamin Moody <floppusmaximus@users.sf.net>
 *
 * This program is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#ifndef _MINIASM_H
#define _MINIASM_H

#ifdef __cplusplus
extern "C" {
#endif

typedef struct _asctx asctx;
typedef struct _asfile asfile;

/* Assembler flags */
enum {
  MINIASM_CACHE_ALL_FILES = 1,	/* Save the contents of input files in
				   memory.  (By default Miniasm caches
				   only those files which cannot be
				   rewound, e.g. input from a pipe.) */

  MINIASM_MEM_PAGED       = 2,	/* Use the TI paged memory model for
				   output data (bits 16-24 of the
				   address are a page number; bits
				   14-15 are ignored.)  For example,
				   code following ORG 0x14000 will be
				   placed at address 0x4000 in the
				   output buffer. */

  MINIASM_WRITE_LISTING   = 4,	/* Write a listing file. */

  MINIASM_WRITE_SYMBOLS   = 8	/* Write a symbol/export file. */
};


/**************** Miniasm API functions ****************/

/* Create a Miniasm context.  FLAGS is a set of flags as defined
   above.  DATA can be any arbitrary data; the pointer provided will
   be passed to the callback functions (see below.) */
asctx* miniasm_init(unsigned long flags, void* data);

/* Free resources associated with a Miniasm context. */
void miniasm_exit(asctx* ctx);

/* Assemble an input file.  CTX is a context obtained using
   miniasm_init().  FILENAME is the name of the file to assemble.

   If the file is assembled successfully, *BUFP will be set to the
   address of the output data buffer.  This is a pointer to an
   internal data structure; it must not be freed, and it is only valid
   until the next call to miniasm_assemble() or miniasm_exit() with
   the given context.  If errors prevent the assembly from completing,
   *BUFP may or may not contain any useful data.

   *STARTP will be set to the address at the start of the buffer, and
   *ENDP will be set to the address following the end of the buffer.
   *NPASSESP will be set to the number of passes taken through the
   input file.  *NERRORSP and *NWARNINGSP will be set to the number of
   errors and warnings, respectively, generated by the assembler.

   Any of BUFP, STARTP, ENDP, NPASSESP, NERRORSP and NWARNINGSP may be
   NULL.  Return value is nonzero if any errors occurred.
 */
int miniasm_assemble(asctx* ctx, const char* filename,
		     const unsigned char** bufp, long int* startp,
		     long int* endp, int* npassesp, int* nerrorsp,
		     int* nwarningsp);

/* Define a symbol that will be visible to the input program (which
   may then use its value in expressions, or may test for its presence
   using the IFDEF directive.)  NAME is the symbol's name
   (case-sensitive) and VALUE is its value. */
void miniasm_define(asctx* ctx, const char* name, long int value);

/* Check if the given symbol is defined.  Return 0 if it is not
   defined, 1 if it is.  If the symbol is defined and VALUE is
   non-null, set *VALUE to the value of the symbol. */
int miniasm_get_symbol(asctx* ctx, const char* name, long int* value);


/**************** I/O Callback Functions ****************/

/* These functions are not defined by the Miniasm library.  If you
   want to use Miniasm in your own programs, you will need to define
   these functions.  These functions are used for all assembler I/O;
   they are used to read input data (assembly source and binary data
   files) as well as to report errors and write to the listing file.

   The callbacks provided in simple.c, used by the standalone
   assembler, are a good starting point for writing your own
   callbacks; in most cases you will not need to modify them
   substantially.

   The DATA parameter to each of these functions is the value passed
   when calling miniasm_init().
 */

/* Open a text file named FILENAME.  This function returns a pointer
   to an asfile structure (which is opaque as far as Miniasm is
   concerned.)  Return NULL if unable to open the file. */
asfile* miniasm_open(const char* filename, asfile* parent,
		     int local, void* data);

/* Close a file opened with miniasm_open(). */
void miniasm_close(asfile* filep, void* data);

/* Read the next line from the input file; return a pointer to it.
   You are free to allocate memory however you like, but the string
   returned needs to be unique (it can't be overwritten by later calls
   to miniasm_gets().)  Miniasm will not modify the string.  If you're
   reading input lines into a static buffer, simply strdup() the
   result before returning it.  Return NULL if there's an error or we
   reach the end of the file. */
char* miniasm_gets(asfile* filep, void* data);

/* Free a string which was previously returned by miniasm_gets(). */
void miniasm_gets_free(char* str);

/* Rewind an input file (set it up so that the next call to
   miniasm_gets() will return the first line in the file.)  Return
   zero if this is possible, or nonzero if the file cannot be
   rewound. */
int miniasm_rewind(asfile* filep, void* data);

/* Get the size of a binary file named FILENAME.  If successful, set
   LENGTHP to its length, and return zero.  Otherwise, return nonzero.
   This function may be called several times for the same file. */
int miniasm_get_bin_size(const char* filename, asfile* parent, int local,
			 long int* lengthp, void* data);

/* Read N bytes from a binary file into the buffer BUF.  Return zero
   if successful, nonzero otherwise.  This function will be called
   only once per INCBIN statement. */
int miniasm_read_bin(const char* filename, asfile* parent, int local,
		     unsigned char* buf, long int n, void* data);

/* Display a message.  The message string does not have a newline
   appended; add a newline yourself if you need one. */
void miniasm_write_message(const char* string, void* data);

/* Clear the listing file (called at the start of each pass.) */
void miniasm_begin_listing(int passnum, int final, void* data);

/* Write output to the listing file.  LINENUM is the current line
   number, TEXT the text of the current line.  LINEADDR is the current
   program counter, and LINECOUNT is the number of code bytes that
   were output by this line.  If LINECOUNT is greater than zero,
   LINEBYTES points to the first such byte. */
void miniasm_write_listing(asfile* filep, int linenum, const char* text,
			   long int lineaddr, long int linecount,
			   const unsigned char* linebytes, void* data);

/* Write output to the symbol/export file(s).  NAME is the name of the
   symbol, VALUE its value, and EXPORTED is 1 if the symbol appears in
   a .export directive. */
void miniasm_write_symbol(const char* name, long int value,
			  int exported, void *data);

#ifdef __cplusplus
}
#endif
#endif
