/*
 * Convert binary file to TI-81 prg file
 *
 * Copyright (c) 2010 Benjamin Moody <floppusmaximus@users.sf.net>
 *
 * This program is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "prgfile.h"
#include "utils.h"

static int isvalid(unsigned int t)
{
  return (t == 0x07 || t == 0xEC
	  || (t >= 0x10 && t <= 0xE4 && t != 0xA2));
}

static const char usage[] = "Usage: %s [binfile] [-o prgfile] [-n name]\n";

int main(int argc, char** argv)
{
  const char* infilename = NULL;
  const char* outfilename = NULL;
  const char* progname = NULL;
  FILE *inf, *outf;
  unsigned char *data;
  unsigned int length, n;
  int i, status = 0;

  for (i = 1; i < argc; i++) {
    if (argv[i][0] != '-' || argv[i][1] == 0)
      infilename = argv[i];
    else {
      if (argv[i][1] == 'o') {
	if (argv[i][2])
	  outfilename = &argv[i][2];
	else {
	  outfilename = argv[++i];
	}
      }
      else if (argv[i][1] == 'n') {
	if (argv[i][2])
	  progname = &argv[i][2];
	else
	  progname = argv[++i];
      }
      else {
	fprintf(stderr, "%s: unknown option %s\n", argv[0], argv[i]);
	fprintf(stderr, usage, argv[0]);
	return 1;
      }
    }
  }

  if (infilename && strcmp(infilename, "-")) {
    inf = fopen(infilename, "rb");
    if (!inf) {
      perror(infilename);
      return 1;
    }
  }
  else {
    infilename = "standard input";
    inf = stdin;
  }

  length = 0;
  data = xmalloc(length + 1024);

  do {
    n = fread(data + length, 1, 1024, inf);
    length += n;
    data = xrealloc(data, length + 1024);
  } while (n == 1024);

  if (inf != stdin)
    fclose(inf);

  for (n = 0; n < length; n++) {
    if (!isvalid(data[n])) {
      fprintf(stderr, "%s: invalid token 0x%02X at offset 0x%04X\n",
	      infilename, data[n], n);
      status = 1;
    }
  }

  if (status) {
    xfree(data);
    return status;
  }

  if (outfilename && strcmp(outfilename, "-")) {
    outf = fopen(outfilename, "wb");
    if (!outf) {
      perror(outfilename);
      xfree(data);
      return 1;
    }
  }
  else
    outf = stdout;

  status = write_prg_file(outf, progname, data, length, 0, 0);

  if (outf != stdout)
    fclose(outf);

  xfree(data);
  return status;
}
