; -*- TI-Asm -*-

;================ Macros ================

; Macro to call a ROM routine
;
; Syntax:
;	ROM_CALL DispHL

ROM_CALL .macro x
	.if (_##x == (_##x & 38h))
	rst _##x
	.else
	call _##x
	.endif
	.endm

; Macro to copy videoMem to LCD
;
; Syntax:
;	UPDATE_LCD

 .ifdef TI81_VER_MAJOR
  .if TI81_VER_MAJOR >= 2

; call ROM routine to update display

UPDATE_LCD .macro
	ROM_CALL LCD_Copy
	.endm

  .else

; display is updated directly from RAM, no need to do anything

UPDATE_LCD .macro
	.endm

  .endif
 .else

; turning the run indicator on will cause the system ISR to update the
; screen periodically - this won't update immediately, and it will
; draw a black box, but it's better than nothing

UPDATE_LCD .macro
	set indicRun,(iy + indicFlags)
	.endm

 .endif

; Macro to copy plotSScreen to videoMem
;
; Syntax:
;	GR_BUF_CPY

GR_BUF_CPY .macro
	scf
	call GrBufCpyOrClr
	.endm

; Macro to clear plotSScreen
;
; Syntax:
;	GR_BUF_CLR

GR_BUF_CLR .macro
	or a
	call GrBufCpyOrClr
	.endm


;================ System Routines ================

;---- Equates for all (?) ROM versions ----

_CkOP1FP0	 .equ 0008h	; Set zero flag if OP1 = 0.0

_MovFPToOP1	 .equ 0010h	; Copy standard-precision (8-byte) FP
				; number at HL to OP1, and convert to
				; extended precision

_MovFPToOP2	 .equ 0018h	; Copy standard-precision (8-byte) FP
				; number at HL to OP2, and convert to
				; extended precision

_PutC		 .equ 0020h	; Display character A in large font
				; and update cursor position

_CursorOff	 .equ 0028h	; Turn cursor off

_Mov8B		 .equ 0030h	; Copy 8 bytes from HL to DE

;---- Equates for version 2.0V ----

 .ifdef TI81_VER_2_0V

_ApdSetup	 .equ 05E4h 	; Reset APD timer
_SetXXOP1	 .equ 09B8h	; Convert integer value A (0 to 99)
				; into an FP number in OP1
_SetXXOP2	 .equ 09BDh	; Convert integer value A (0 to 99)
				; into an FP number in OP2
_WToV		 .equ 09D1h	; Convert window to pixel coordinates:
				; given DE = address of X value and HL
				; = address of Y value (both standard-
				; precision FP numbers), return B =
				; pixel column and C = pixel row.
				; OP1-OP3 are used internally, so
				; don't use them for inputs.
_VToW		 .equ 09DDh	; Convert pixel to window coordinates:
				; given B = pixel column and C = pixel
				; row, store X value at address DE and
				; Y value at address HL (both in
				; standard-precision form.)  OP1-OP3
				; are used internally, so don't use
				; them for outputs.
_YIToF		 .equ 09E7h	; Given A = pixel row, compute Y value
				; and store at address HL
_XIToF		 .equ 09EFh	; Given A = pixel column, compute X
				; value and store at address HL
_XFToI		 .equ 0A07h	; Given DE = address of X value,
				; return A = pixel column
_YFToI		 .equ 0A0Fh	; Given DE = address of Y value,
				; return A = pixel row
_HideCursor	 .equ 0E46h	; Hide cursor (if currently shown)
_CursorOn	 .equ 0E8Eh	; Turn cursor on.  If graphDisp flag
				; is set, show a graph cursor, with
				; type given by (curGStyle) and
				; coordinates (curGX) and (curGY).  If
				; graphDisp flag is not set, show a
				; text cursor at coordinates (curCol)
				; and (curRow).
_ShowCursor	 .equ 0ECCh	; Show cursor (if enabled and not
				; currently shown)
_FormDisp	 .equ 0F3Bh	; Convert and display the value in OP1
				; (as in the BASIC Disp command.)  OP1
				; must be either a standard-precision
				; FP number or a matrix reference.
_FormERealNoPad	 .equ 102Ch	; Convert standard-precision number in
				; OP1 into a string.  (You usually
				; want to call OP1ExtToStd
				; beforehand.)  A = maximum length of
				; output string.  Ignores all format
				; settings; returns HL = address of
				; string.
_GetFormStrNoPad .equ 102Fh	; Return HL = address of first
				; non-space character in OP2
_DispTail	 .equ 114Ch	; Display full edit buffer contents
				; from current position to the end of
				; the buffer, and move cursor to the
				; end
_LdDEHLind	 .equ 116Dh	; DE := word at address HL
				; HL := word at address HL+2
_SaveShadow	 .equ 1176h	; Save cursor position to textShadCur
				; and current text flags to
				; textShadIns
_RstrShadow	 .equ 1187h	; Restore textShadCur, textShadIns,
				; and textShadow, and turn cursor on
_BufLeft	 .equ 11B4h	; Move to previous position in edit
				; buffer (if any); zero set if already
				; at start of buffer; otherwise return
				; A = token moved over
_BufRight	 .equ 11C6h	; Move to next position in edit buffer
				; (if any); zero set if already at end
				; of buffer; otherwise return A =
				; token moved over
_BufInsert	 .equ 11DCh	; Insert the token A in the edit
				; buffer at the current position;
				; return zero set if buffer full
_BufReplace	 .equ 11EEh	; Replace the token at the current
				; position (if any) with A; return
				; zero set if buffer full
_BufDelete	 .equ 1204h	; Delete the following token from the
				; edit buffer; return zero set if at
				; end of buffer
_BufToBtm	 .equ 1214h	; Move to the end of the edit buffer
_BufToTop	 .equ 122Fh	; Move to the start of the edit buffer
_BufPeek	 .equ 126Ah	; Set zero flag if at the end of the
				; edit buffer, otherwise return A =
				; next token
_IsAtTop	 .equ 1277h	; Set zero flag if at the start of the
				; edit buffer
_HLequalsDE	 .equ 127Eh	; Set zero flag if HL = DE
_CursorLeft	 .equ 1284h	; Move backwards one token in edit
				; buffer, and move text cursor left
				; accordingly; scroll and redisplay if
				; necessary
_CursorRight	 .equ 12A1h	; Move forwards one token in edit
				; buffer, and move text cursor right
				; accordingly; scroll and redisplay if
				; necessary
_CursorUp	 .equ 12C7h	; Move cursor to previous line in edit
				; buffer; scroll and redisplay if
				; necessary
_CursorDown	 .equ 12F5h	; Move cursor to next line in edit
				; buffer; scroll and redisplay if
				; necessary
_JError		 .equ 13BFh	; Throw error number A
_JErrorNo	 .equ 13C2h	; Throw error number (errNo)
_DarkLine	 .equ 2514h	; Draw dark line from (B, C) to (D, E)
_ILine		 .equ 2516h	; Draw line from (B, C) to (D, E):
				; H = 0 -> white line
				; H = 1 -> dark line
				; H = 2 -> invert line
_IPoint		 .equ 2586h	; Modify pixel at (B, C):
				; D = 0 -> turn pixel off
				; D = 1 -> turn pixel on
				; D = 2 -> invert pixel
				; D = 4 -> copy pixel from plotSScreen
				; to videoMem
_IOffset	 .equ 25CFh	; Convert pixel coords (B, C) into
				; buffer offset HL and bit mask A
_IBounds	 .equ 25F8h	; Check if pixel coords (B, C) are
				; valid (carry flag set if not)
_GrBufCpyOrClr	 .equ 2609h	; If carry flag set: copy plotSScreen
				; to videoMem
				; If carry flag clear: fill
				; plotSScreen with zeroes
_KbdScan	 .equ 2625h	; Scan for keypresses, update
				; kbdScanCode if necessary
_GetRawCSC	 .equ 2672h	; Scan for keys currently pressed; set
				; carry flag if multiple keys pressed;
				; A = 0 if no keys pressed; otherwise
				; A = scan code
_ReadKeyGroup	 .equ 26ACh	; Read keypad directly; A = inverted
				; bit mask of key groups to scan;
				; return A = inverted bit mask of keys
				; pressed in those groups
_GetCSC		 .equ 26C3h	; Check if a key has been pressed;
				; return A = scan code, or 0 for none
				; (scanning is done by ISR and
				; automatically handles debouncing,
				; multiple keys, and repeating)
_GetKey		 .equ 26D0h	; Wait for a keypress (handling 2nd
				; and Alpha), and return the key code
				; in A.  If On is pressed, onInterrupt
				; is set and A = 0.  Note: pressing
				; 2nd+Off aborts program and turns off
				; calc.
_CanAlphIns	 .equ 27E9h	; Clear alpha, alpha-lock, and
				; insertion flags
_CanAlpha	 .equ 27ECh	; Disable alpha and alpha-lock flags
_SetAlphaLock	 .equ 27F5h	; Set alpha and alpha-lock flags
_ResetMatrices	 .equ 2BDFh	; Reset the matrix variables [A], [B],
				; and [C] to all zeroes and set
				; dimensions to 6x6
_Intgr		 .equ 2BFFh	; OP1 := floor(OP1)
_Trunc		 .equ 2C0Eh	; OP1 := iPart(OP1)
_InvSub		 .equ 2C17h	; OP1 := OP2 - OP1
_Times2		 .equ 2C1Ch	; OP1 := OP1 * 2
_Plus1		 .equ 2C21h	; OP1 := OP1 + 1
_Minus1		 .equ 2C26h	; OP1 := OP1 - 1
_FPSub		 .equ 2C29h	; OP1 := OP1 - OP2
_FPAdd		 .equ 2C30h	; OP1 := OP1 + OP2
_DToR		 .equ 2C89h	; OP1 := OP1 * (pi / 180)
_RToD		 .equ 2C8Dh	; OP1 := OP1 * (180 / pi)
_Cube		 .equ 2C93h	; OP1 := OP1 ^ 3
_FPSquare	 .equ 2C98h	; OP1 := OP1 ^ 2
_FPMult		 .equ 2C9Bh	; OP1 := OP1 * OP2
_InvOP1S	 .equ 2D78h	; OP1 := -OP1
_Frac		 .equ 2D83h	; OP1 := fPart(OP1)
_DivBy2		 .equ 2DBAh	; OP1 := OP1 / 2
_FPRecip	 .equ 2DBFh	; OP1 := 1 / OP1
_FPDiv		 .equ 2DC5h	; OP1 := OP1 / OP2
_SqRoot		 .equ 2E3Ch	; OP1 := sqrt(OP1)
_RndGuard	 .equ 2EF6h	; Round OP1 to 10 mantissa digits
_Round		 .equ 2F0Dh	; Round OP1 to D digits after the
				; decimal point
_Factorial	 .equ 2F5Fh	; OP1 := (OP1)!
_LnX		 .equ 2FACh	; OP1 := ln(OP1)
_LogX		 .equ 2FB0h	; OP1 := log(OP1)
_EToX		 .equ 3182h	; OP1 := e ^ OP1
_TenX		 .equ 3187h	; OP1 := 10 ^ OP1
_CubeRoot	 .equ 3277h	; OP1 := cbrt(OP1)
_YToX		 .equ 328Ah	; OP1 := OP1 ^ OP2
_Sin		 .equ 3332h	; OP1 := sin(OP1), OP2 := cos(OP1)
_Cos		 .equ 3335h	; OP1 := cos(OP1)
_Tan		 .equ 3339h	; OP1 := tan(OP1)
_TanH		 .equ 36EEh	; OP1 := tanh(OP1)
_CosH		 .equ 36F2h	; OP1 := cosh(OP1)
_SinH		 .equ 36F6h	; OP1 := sinh(OP1)
_ACos		 .equ 3774h	; OP1 := cos^-1(OP1)
_ATan		 .equ 3778h	; OP1 := tan^-1(OP1)
_ATan2		 .equ 377Bh	; OP1 := theta s.t. R sin theta = OP1
				; and R cos theta = OP2
_ASin		 .equ 377Fh	; OP1 := sin^-1(OP1)
_ATanH		 .equ 394Eh	; OP1 := tanh^-1(OP1)
_ASinH		 .equ 3996h	; OP1 := sinh^-1(OP1)
_ACosH		 .equ 39A6h	; OP1 := cosh^-1(OP1)
_Combinations	 .equ 3A2Fh	; OP1 := OP1 "nCr" OP2
				; (OP1! / (OP1-OP2)! OP2!)
_Permutations	 .equ 3A7Fh	; OP1 := OP1 "nPr" OP2
				; (OP1! / (OP1-OP2)!)
_PToR		 .equ 3A89h	; Convert polar to rectangular coords
				; OP1 := OP1 * cos(OP2)
				; OP2 := OP1 * sin(OP2)
_RToP		 .equ 3AA7h	; Convert rectangular to polar coords
				; OP1 := sqrt(OP1 ^ 2 + OP2 ^ 2)
				; OP2 := atan2(OP2, OP1)
_CpOP1OP2	 .equ 3AD6h	; Compare OP1 to OP2: set zero flag if
				; numbers are equal, set carry flag if
				; OP1 < OP2
_CompStrsN	 .equ 3AF1h	; Compare B bytes at DE with HL
_OP2StdToExt	 .equ 3B21h	; Convert OP2 from standard (8-byte)
				; to extended (10-byte) precision
_OP1StdToExt	 .equ 3B40h	; Convert OP1 from standard to
				; extended precision
_OP1ExtToStd	 .equ 3B51h	; Convert OP1 from extended to
				; standard precision
_MovFPFrOP1	 .equ 3B87h	; Convert OP1 to standard precision
				; and copy 8 bytes to DE
_Mov8FrOP1	 .equ 3B8Ah	; Copy 8 bytes from OP1 to DE (note:
				; you usually want to use MovFPFrOP1
				; instead)
_OP3ToOP4	 .equ 3B8Fh	; OP4 := OP3
_OP1ToOP4	 .equ 3B94h	; OP4 := OP1
_OP2ToOP4	 .equ 3B99h	; OP4 := OP2
_Mov10ToOP4	 .equ 3B9Ch	; Copy 10 bytes from HL to OP4
_OP4ToOP2	 .equ 3BA1h	; OP2 := OP4
_OP3ToOP2	 .equ 3BA6h	; OP2 := OP3
_OP1ToOP3	 .equ 3BB5h	; OP3 := OP1
_Mov10FrOP1	 .equ 3BB8h	; Copy 10 bytes from OP1 to DE
_OP5ToOP2	 .equ 3BBDh	; OP2 := OP5
_OP1ToOP2	 .equ 3BC2h	; OP2 := OP1
_Mov10ToOP2	 .equ 3BC5h	; Copy 10 bytes from HL to OP2
_TempToOP2	 .equ 3BCAh	; Copy 10 bytes from scratchMem to OP2
_TempToOP1	 .equ 3BCFh	; Copy 10 bytes from scratchMem to OP1
_OP4ToOP1	 .equ 3BD4h	; OP1 := OP4
_OP5ToOP1	 .equ 3BD9h	; OP1 := OP5
_OP3ToOP1	 .equ 3BDEh	; OP1 := OP3
_OP4ToOP5	 .equ 3BE3h	; OP5 := OP4
_OP2ToOP5	 .equ 3BE8h	; OP5 := OP2
_Mov10ToOP5	 .equ 3BEBh	; Copy 10 bytes from HL to OP5
_OP2ToTemp	 .equ 3BF0h	; Copy 10 bytes from OP2 to scratchMem
_OP1ToTemp	 .equ 3BF5h	; Copy 10 bytes from OP1 to scratchMem
_Mov10ToTemp	 .equ 3BF8h	; Copy 10 bytes from HL to scratchMem
_OP1ToOP5	 .equ 3BFDh	; OP5 := OP1
_OP2ToOP1	 .equ 3C02h	; OP1 := OP2
_Mov10ToOP1	 .equ 3C05h	; Copy 10 bytes from HL to OP1
_Mov10B		 .equ 3C08h	; Copy 10 bytes from HL to DE
_Mov9B		 .equ 3C0Ah	; Copy 9 bytes from HL to DE
_Mov7B		 .equ 3C0Eh	; Copy 7 bytes from HL to DE
_Mov6B		 .equ 3C10h	; Copy 6 bytes from HL to DE
_Mov5B		 .equ 3C12h	; Copy 5 bytes from HL to DE
_Mov4B		 .equ 3C14h	; Copy 4 bytes from HL to DE
_Mov3B		 .equ 3C16h	; Copy 3 bytes from HL to DE
_Mov2B		 .equ 3C18h	; Copy 2 bytes from HL to DE
_Mov1B		 .equ 3C1Ah	; Copy 1 byte from HL to DE
_Mov8ToOP1	 .equ 3C1Dh	; Copy 8 bytes from HL to OP1 (note:
				; you usually want to use MovFPToOP1
				; instead)
_Mov8ToOP2	 .equ 3C22h	; Copy 8 bytes from HL to OP2 (note:
				; you usually want to use MovFPToOP2
				; instead)
_OP2Set3	 .equ 3C59h	; OP2 := 3.0
_OP1Set1	 .equ 3C61h	; OP1 := 1.0
_OP1Set2	 .equ 3C66h	; OP1 := 2.0
_OP2Set2	 .equ 3C6Bh	; OP2 := 2.0
_OP2Set1	 .equ 3C76h	; OP2 := 1.0
_OP1Set0	 .equ 3C91h	; OP1 := 0.0
_ZeroOP1	 .equ 3CA1h	; Set OP1 to all zeroes
_ZeroOP2	 .equ 3CA6h	; Set OP2 to all zeroes
_ZeroOP3	 .equ 3CABh	; Set OP3 to all zeroes
_ClrLp		 .equ 3CB2h	; Set B bytes at HL to zero
_ShRAcc		 .equ 3CB9h	; A := A / 16
_ShLAcc		 .equ 3CC1h	; A := A * 16
_OP1ExOP3	 .equ 3E68h	; Swap OP1 with OP3
_OP1ExOP4	 .equ 3E6Dh	; Swap OP1 with OP4
_OP1ExOP2	 .equ 3E72h	; Swap OP1 with OP2
_ExLp		 .equ 3E7Ah	; Swap B bytes at HL with bytes at DE
_CkOP2FP0	 .equ 3E89h	; Set zero flag if OP2 = 0
_CkOP1Int	 .equ 3E8Eh	; Set zero flag if OP1 is a positive integer
_CkOP1Odd	 .equ 3E95h	; Set zero flag if OP1 is even.
				; Doesn't work for non-integers.
_GetConOP1	 .equ 3F17h	; Load builtin constant to OP1
				; A = 0 -> 180 / pi
				; A = 1 -> pi / 2
				; A = 2 -> pi / 4
				; A = 3 -> log e
				; A = 4 -> pi
				; A = 5 -> pi / 180
				; A = 6 -> ln 10
_GetConOP2	 .equ 3F1Ch	; Load builtin constant to OP2
_CkOP2Pos	 .equ 3FB1h	; Set zero flag if OP2 >= 0
_CkOP1Pos	 .equ 3FB6h	; Set zero flag if OP1 >= 0
_ClrOP1S	 .equ 3FBCh	; OP1 := abs(OP1)
_DecO1Exp	 .equ 3FC3h	; OP1 := OP1 / 10 (carry set on underflow)
_IncO1Exp	 .equ 3FCBh	; OP1 := OP1 * 10 (error on overflow)
_MatSubtract	 .equ 4056h	; Subtract matrix OP2 from matrix OP1.
				; Store the result in scratchMem, and
				; return a matrix ref in OP1.
_MatAdd		 .equ 405Ah	; Add matrices OP1 and OP2.  Store the
				; result in scratchMem, and return a
				; matrix ref in OP1.
_MatMultiply	 .equ 40A6h	; Multiply matrices OP1 and OP2.
				; Store the result in scratchMem, and
				; return a matrix ref in OP1.
_MatTranspose	 .equ 40FBh	; Transpose matrix OP1.  Store the
				; result in scratchMem, and return a
				; matrix ref in OP1.
_CkPosInt	 .equ 4131h	; Set zero flag if OP1 is a positive
				; integer
_MatInvert	 .equ 424Ch	; Invert matrix OP1; store the result
				; in scratchMem, and return a matrix
				; ref in OP1.
_MatDeterminant	 .equ 424Fh	; Get determinant of matrix OP1; store
				; the result in OP1.
_MatRndGuard	 .equ 4428h	; Round entries of matrix OP1 to 10
				; mantissa digits; store the result in
				; scratchMem, and return a matrix ref
				; in OP1.
_MatRound	 .equ 442Ch	; Round entries of matrix OP1 to A
				; digits after the decimal point;
				; store the result in scratchMem, and
				; return a matrix ref in OP1.
_MatNegate	 .equ 4433h	; Negate matrix OP1; store the result
				; in scratchMem, and return a matrix
				; ref in OP1.
_MatTimesScalar	 .equ 4438h	; Multiply matrix OP1 by scalar OP2;
				; store the result in scratchMem, and
				; return a matrix ref in OP1.
_DispHL		 .equ 4829h	; Display HL in decimal, padded on
				; left with spaces (destroys OP1)
_FormReal	 .equ 4CB3h	; Convert standard-precision number in
				; OP1 into a string.  (You usually
				; want to call OP1ExtToStd
				; beforehand.)  A = length of desired
				; output string.  Uses (fmtMode) and
				; (fmtDigits).  Result is stored in
				; OP2, right-justified and padded with
				; spaces if needed (you can use
				; GetFormStrNoPad to skip the spaces
				; if necessary.)
_FormEReal	 .equ 4CB7h	; Convert standard-precision number in
				; OP1 into a string.  A = length of
				; desired output string.  Ignores all
				; format settings.  Result is stored
				; in OP2, right-justified and padded
				; with spaces.
_BrkInc		 .equ 564Fh	; If ON has been pressed, throw an
				; error; otherwise, increment (curPC)
				; and return A = next token
_IncFetch	 .equ 5656h	; Increment (curPC) and return A =
				; next token (A = 0 if at end of file)
_CurFetch	 .equ 565Dh	; Return A = token at (curPC) (A = 0
				; at end of file)
_NxtFetch	 .equ 566Bh	; Return A = token at (curPC) + 1,
				; without changing (curPC) (A = 0 at
				; end of file)
_StoVar		 .equ 5B50h	; Store a value to a real variable;
				; D = variable token, OP1 = value to
				; store (normally a standard-precision
				; float, but in the case of Ans, OP1
				; may be a matrix reference)
_RclVar		 .equ 5BB6h	; Retrieve the value of a real
				; variable; D = variable token; return
				; value in OP1 (as either a standard-
				; precision float, or, in the case of
				; Ans, possibly a matrix reference)
_StoAns		 .equ 5BE2h	; Store the value of OP1 to Ans
_RclAns		 .equ 5C06h	; Retrieve the value of Ans to OP1
_StoMatrix	 .equ 5C31h	; Store a matrix to a matrix variable;
				; A = variable token (tAns, tMatA,
				; tMatB, or tMatC); OP1 = matrix
				; reference
_RclMatrix	 .equ 5C72h	; Make OP1 a reference to a matrix
				; variable; A = variable token (tAns,
				; tMatA, tMatB, or tMatC)
_LdBCHLind	 .equ 5EE8h	; BC := word at address HL
				; HL := word at address HL+2
_LdHLind	 .equ 5EECh	; HL := word at address HL
_Random		 .equ 63DDh	; OP1 := random number
_StoRand	 .equ 6420h	; Seed random number generator with
				; value in OP1
_PutTokString	 .equ 677Ch	; Display token number A, using the
				; large font
_PutSysString	 .equ 6799h	; Display system message number A,
				; using the large font
_PutNthFlagStr	 .equ 67A4h	; Display string number A in a series
				; of strings starting at HL
_PutFlagStr	 .equ 67A7h	; Display string at HL using the large
				; font.  Only characters 0-127 are
				; allowed; last character in the
				; string is indicated by having bit 7
				; set.
_FindNthFlagStr	 .equ 67B8h	; Skip over A strings starting at HL
_PutMap		 .equ 680Ah	; Display character A in large font,
				; without moving cursor
_PutSmallChar	 .equ 689Dh	; Display small character A to screen
				; buffer at address HL.  textLeftSide
				; flag determines position.  Updates
				; textLeftSide and HL.
_PopHLDEBCAFRet	 .equ 000Bh	; Pop registers HL, DE, BC, AF in that
				; order, and return (you probably want
				; to JP to this, not CALL)
_PutSmallFlagStr .equ 696Bh	; Display string at address DE to
				; buffer at HL, subtracting 14h from
				; each char (to translate from large
				; to small font.)  Only intended to be
				; used for numbers, but you could
				; probably use this for other purposes
				; if you're careful.
_HomeUp		 .equ 6982h	; Set (curRow, curCol) to (0, 0)
_EraseEOL	 .equ 699Ch	; Erase text from cursor position to
				; end of line
_EraseEOW	 .equ 69BBh	; Erase text from cursor position to
				; bottom of screen
_NewLine	 .equ 69DBh	; Move to next line (does not erase)
_GetCurLoc	 .equ 6A60h	; Return HL = textShadow address
				; corresponding to current cursor
				; position
_Load_LFont	 .equ 6A75h	; Load font character A to scratchMem,
				; in packed form
_SaveTR		 .equ 6AD0h	; Enable top-right indicator (save top
				; right corner of screen to indicMem
				; and set trIndicInUse flag)
_RestoreTR	 .equ 6AF9h	; Disable top-right indicator (restore
				; indicMem to top right corner of
				; screen and clear trIndicInUse flag)
_PopHLDEBCRet	 .equ 6B55h	; Pop registers HL, DE, BC in that
				; order, and return (you probably want
				; to JP to this, not CALL)
_ClrWindow	 .equ 6AADh	; Clear screen and set cursor to (0, 0)
_ClrScrn	 .equ 6AB0h	; Clear screen; if textSave set, clear
				; textShadow as well
_ClrTxtShd	 .equ 6AC2h	; Clear textShadow
_GetTokLen	 .equ 6BCFh	; Given A = token, return BC = length
				; of the token string
_RunIndicOn	 .equ 6BE5h	; Enable run indicator
_RunIndicOff	 .equ 6BF1h	; Disable run indicator
_tokenStrings	 .equ 6C5Bh	; Start of token strings (used by
				; PutTokString and similar routines);
				; you can use FindNthFlagStr to look
				; up a specific token
_systemStrings	 .equ 6F80h	; Start of system message strings
				; (used by the PutSysString routine;
				; these include menu titles and
				; various other strings)
_LCD_Copy	 .equ 6C29h	; Copy contents of videoMem to LCD
_LCD_Busy	 .equ 6C55h	; Short delay to ensure LCD driver is
				; ready to accept commands
_GetStatCount	 .equ 71D7h	; Return A = DE = number of stat list
				; elements (call this followed by
				; GetStatXPtr to obtain the address of
				; the end of free memory)
_ConvOP1	 .equ 71DDh	; Convert OP1 (must be < 1000) into an
				; integer; result in DE and A = E
_GetStatXPtr	 .equ 7204h	; Return HL = address of {x}(A)
_GetStatYPtr	 .equ 7209h	; Return HL = address of {y}(A)
_IsEditEmpty	 .equ 7E2Ch	; Set zero flag if the edit buffer is
				; empty
_IsAtBtm	 .equ 7E37h	; Set zero flag if at the end of the
				; edit buffer

 .endif ; TI81_VER_2_0V


;---- Equates for version 1.8K (see above for descriptions) ----

 .ifdef TI81_VER_1_8K

_ApdSetup	 .equ 05ACh
_SetXXOP1	 .equ 0980h
_SetXXOP2	 .equ 0985h
_WToV		 .equ 0999h
_VToW		 .equ 09A5h
_YIToF		 .equ 09AFh
_XIToF		 .equ 09B7h
_XFToI		 .equ 09CFh
_YFToI		 .equ 09D7h
_HideCursor	 .equ 0E0Eh
_CursorOn	 .equ 0E53h
_ShowCursor	 .equ 0E91h
_FormDisp	 .equ 0EF9h
_FormERealNoPad	 .equ 0FEAh
_GetFormStrNoPad .equ 0FEDh
_DispTail	 .equ 110Ah
_LdDEHLind	 .equ 112Bh
_SaveShadow	 .equ 1134h
_RstrShadow	 .equ 1145h
_BufLeft	 .equ 1172h
_BufRight	 .equ 1184h
_BufInsert	 .equ 119Ah
_BufReplace	 .equ 11ACh
_BufDelete	 .equ 11C2h
_BufToBtm	 .equ 11D2h
_BufToTop	 .equ 11EDh
_BufPeek	 .equ 1228h
_IsAtTop	 .equ 1235h
_HLequalsDE	 .equ 123Ch
_CursorLeft	 .equ 1242h
_CursorRight	 .equ 125Fh
_CursorUp	 .equ 1285h
_CursorDown	 .equ 12B3h
_JError		 .equ 137Dh
_JErrorNo	 .equ 1380h
_DarkLine	 .equ 24D2h
_ILine		 .equ 24D4h
_IPoint		 .equ 2544h
_IOffset	 .equ 258Dh
_IBounds	 .equ 25B6h
_GrBufCpyOrClr	 .equ 25C7h
_KbdScan	 .equ 25E3h
_GetRawCSC	 .equ 2630h
_ReadKeyGroup	 .equ 266Ah
_GetCSC		 .equ 2681h
_GetKey		 .equ 268Eh
_CanAlphIns	 .equ 2798h
_CanAlpha	 .equ 279Bh
_SetAlphaLock	 .equ 27A4h
_ResetMatrices	 .equ 2B9Ch
_Intgr		 .equ 2BBCh
_Trunc		 .equ 2BCBh
_InvSub		 .equ 2BD4h
_Times2		 .equ 2BD9h
_Plus1		 .equ 2BDEh
_Minus1		 .equ 2BE3h
_FPSub		 .equ 2BE6h
_FPAdd		 .equ 2BEDh
_DToR		 .equ 2C46h
_RToD		 .equ 2C4Ah
_Cube		 .equ 2C50h
_FPSquare	 .equ 2C55h
_FPMult		 .equ 2C58h
_InvOP1S	 .equ 2D35h
_Frac		 .equ 2D40h
_DivBy2		 .equ 2D77h
_FPRecip	 .equ 2D7Ch
_FPDiv		 .equ 2D82h
_SqRoot		 .equ 2DF9h
_RndGuard	 .equ 2EB3h
_Round		 .equ 2ECAh
_Factorial	 .equ 2F1Ch
_LnX		 .equ 2F69h
_LogX		 .equ 2F6Dh
_EToX		 .equ 313Fh
_TenX		 .equ 3144h
_CubeRoot	 .equ 3234h
_YToX		 .equ 3247h
_Sin		 .equ 32EFh
_Cos		 .equ 32F2h
_Tan		 .equ 32F6h
_TanH		 .equ 36ABh
_CosH		 .equ 36AFh
_SinH		 .equ 36B3h
_ACos		 .equ 3731h
_ATan		 .equ 3735h
_ATan2		 .equ 3738h
_ASin		 .equ 373Ch
_ATanH		 .equ 390Bh
_ASinH		 .equ 3953h
_ACosH		 .equ 3963h
_Combinations	 .equ 39ECh
_Permutations	 .equ 3A3Ch
_PToR		 .equ 3A46h
_RToP		 .equ 3A64h
_CpOP1OP2	 .equ 3A93h
_CompStrsN	 .equ 3AAEh
_OP2StdToExt	 .equ 3ADEh
_OP1StdToExt	 .equ 3AFDh
_OP1ExtToStd	 .equ 3B0Eh
_MovFPFrOP1	 .equ 3B44h
_Mov8FrOP1	 .equ 3B47h
_OP3ToOP4	 .equ 3B4Ch
_OP1ToOP4	 .equ 3B51h
_OP2ToOP4	 .equ 3B56h
_Mov10ToOP4	 .equ 3B59h
_OP4ToOP2	 .equ 3B5Eh
_OP3ToOP2	 .equ 3B63h
_OP1ToOP3	 .equ 3B72h
_Mov10FrOP1	 .equ 3B75h
_OP5ToOP2	 .equ 3B7Ah
_OP1ToOP2	 .equ 3B7Fh
_Mov10ToOP2	 .equ 3B82h
_TempToOP2	 .equ 3B87h
_TempToOP1	 .equ 3B8Ch
_OP4ToOP1	 .equ 3B91h
_OP5ToOP1	 .equ 3B96h
_OP3ToOP1	 .equ 3B9Bh
_OP4ToOP5	 .equ 3BA0h
_OP2ToOP5	 .equ 3BA5h
_Mov10ToOP5	 .equ 3BA8h
_OP2ToTemp	 .equ 3BADh
_OP1ToTemp	 .equ 3BB2h
_Mov10ToTemp	 .equ 3BB5h
_OP1ToOP5	 .equ 3BBAh
_OP2ToOP1	 .equ 3BBFh
_Mov10ToOP1	 .equ 3BC2h
_Mov10B		 .equ 3BC5h
_Mov9B		 .equ 3BC7h
_Mov7B		 .equ 3BCBh
_Mov6B		 .equ 3BCDh
_Mov5B		 .equ 3BCFh
_Mov4B		 .equ 3BD1h
_Mov3B		 .equ 3BD3h
_Mov2B		 .equ 3BD5h
_Mov1B		 .equ 3BD7h
_Mov8ToOP1	 .equ 3BDAh
_Mov8ToOP2	 .equ 3BDFh
_OP2Set3	 .equ 3C16h
_OP1Set1	 .equ 3C1Eh
_OP1Set2	 .equ 3C23h
_OP2Set2	 .equ 3C28h
_OP2Set1	 .equ 3C33h
_OP1Set0	 .equ 3C4Eh
_ZeroOP1	 .equ 3C5Eh
_ZeroOP2	 .equ 3C63h
_ZeroOP3	 .equ 3C68h
_ClrLp		 .equ 3C6Fh
_ShRAcc		 .equ 3C76h
_ShLAcc		 .equ 3C7Eh
_OP1ExOP3	 .equ 3E25h
_OP1ExOP4	 .equ 3E2Ah
_OP1ExOP2	 .equ 3E2Fh
_ExLp		 .equ 3E37h
_CkOP2FP0	 .equ 3E46h
_CkOP1Int	 .equ 3E4Bh
_CkOP1Odd	 .equ 3E52h
_GetConOP1	 .equ 3ED4h
_GetConOP2	 .equ 3ED9h
_CkOP2Pos	 .equ 3F6Eh
_CkOP1Pos	 .equ 3F73h
_ClrOP1S	 .equ 3F79h
_DecO1Exp	 .equ 3F80h
_IncO1Exp	 .equ 3F88h
_MatSubtract	 .equ 4013h
_MatAdd		 .equ 4017h
_MatMultiply	 .equ 4063h
_MatTranspose	 .equ 40B8h
_CkPosInt	 .equ 40EEh
_MatInvert	 .equ 4209h
_MatDeterminant	 .equ 420Ch
_MatRndGuard	 .equ 43E5h
_MatRound	 .equ 43E9h
_MatNegate	 .equ 43F0h
_MatTimesScalar	 .equ 43F5h
_DispHL		 .equ 47E6h
_FormReal	 .equ 4C70h
_FormEReal	 .equ 4C74h
_BrkInc		 .equ 560Ch
_IncFetch	 .equ 5613h
_CurFetch	 .equ 561Ah
_NxtFetch	 .equ 5628h
_StoVar		 .equ 5B0Dh
_RclVar		 .equ 5B73h
_StoAns		 .equ 5B9Fh
_RclAns		 .equ 5BC3h
_StoMatrix	 .equ 5BEEh
_RclMatrix	 .equ 5C2Fh
_LdBCHLind	 .equ 5EA5h
_LdHLind	 .equ 5EA9h
_Random		 .equ 639Ah
_StoRand	 .equ 63DDh
_PutTokString	 .equ 6739h
_PutSysString	 .equ 6756h
_PutNthFlagStr	 .equ 6761h
_PutFlagStr	 .equ 6764h
_FindNthFlagStr	 .equ 6775h
_PutMap		 .equ 67C7h
_PutSmallChar	 .equ 685Ah
_PopHLDEBCAFRet	 .equ 68AAh
_PutSmallFlagStr .equ 692Ah
_HomeUp		 .equ 6941h
_EraseEOL	 .equ 695Bh
_EraseEOW	 .equ 697Ah
_NewLine	 .equ 699Ah
_GetCurLoc	 .equ 6A1Fh
_Load_LFont	 .equ 6A34h
_SaveTR		 .equ 6A8Fh
_RestoreTR	 .equ 6AB8h
_PopHLDEBCRet	 .equ 6B14h
_ClrWindow	 .equ 6A6Ch
_ClrScrn	 .equ 6A6Fh
_ClrTxtShd	 .equ 6A81h
_GetTokLen	 .equ 6B8Eh
_RunIndicOn	 .equ 6BA4h
_RunIndicOff	 .equ 6BAEh
_tokenStrings	 .equ 6BE7h
_systemStrings	 .equ 6F0Ch
_GetStatCount	 .equ 71CCh
_ConvOP1	 .equ 71D2h
_GetStatXPtr	 .equ 71F9h
_GetStatYPtr	 .equ 71FEh
_IsEditEmpty	 .equ 7E21h
_IsAtBtm	 .equ 7E2Ch

 .endif	; TI81_VER_1_8K


;---- Equates for version 1.6K (see above for descriptions) ----

 .ifdef TI81_VER_1_6K

_ApdSetup	 .equ 05A7h
_SetXXOP1	 .equ 0974h
_SetXXOP2	 .equ 0979h
_WToV		 .equ 098Dh
_VToW		 .equ 0999h
_YIToF		 .equ 09A3h
_XIToF		 .equ 09ABh
_XFToI		 .equ 09C3h
_YFToI		 .equ 09CBh
_HideCursor	 .equ 0E02h
_CursorOn	 .equ 0E47h
_ShowCursor	 .equ 0E85h
_FormDisp	 .equ 0EEDh
_FormERealNoPad	 .equ 0FDEh
_GetFormStrNoPad .equ 0FE1h
_DispTail	 .equ 10FEh
_LdDEHLind	 .equ 111Fh
_SaveShadow	 .equ 1128h
_RstrShadow	 .equ 1139h
_BufLeft	 .equ 1166h
_BufRight	 .equ 1178h
_BufInsert	 .equ 118Eh
_BufReplace	 .equ 11A0h
_BufDelete	 .equ 11B6h
_BufToBtm	 .equ 11C6h
_BufToTop	 .equ 11E1h
_BufPeek	 .equ 121Ch
_IsAtTop	 .equ 1229h
_HLequalsDE	 .equ 1230h
_CursorLeft	 .equ 1236h
_CursorRight	 .equ 1253h
_CursorUp	 .equ 1279h
_CursorDown	 .equ 12A7h
_JError		 .equ 1371h
_JErrorNo	 .equ 1374h
_DarkLine	 .equ 24C6h
_ILine		 .equ 24C8h
_IPoint		 .equ 2538h
_IOffset	 .equ 2581h
_IBounds	 .equ 25AAh
_GrBufCpyOrClr	 .equ 25BBh
_KbdScan	 .equ 25D7h
_GetRawCSC	 .equ 2624h
_ReadKeyGroup	 .equ 265Eh
_GetCSC		 .equ 2675h
_GetKey		 .equ 2682h
_CanAlphIns	 .equ 278Bh
_CanAlpha	 .equ 278Eh
_SetAlphaLock	 .equ 2797h
_ResetMatrices	 .equ 2B79h
_Intgr		 .equ 2B99h
_Trunc		 .equ 2BA8h
_InvSub		 .equ 2BB1h
_Times2		 .equ 2BB6h
_Plus1		 .equ 2BBBh
_Minus1		 .equ 2BC0h
_FPSub		 .equ 2BC3h
_FPAdd		 .equ 2BCAh
_DToR		 .equ 2C23h
_RToD		 .equ 2C27h
_Cube		 .equ 2C2Dh
_FPSquare	 .equ 2C32h
_FPMult		 .equ 2C35h
_InvOP1S	 .equ 2D12h
_Frac		 .equ 2D1Dh
_DivBy2		 .equ 2D54h
_FPRecip	 .equ 2D59h
_FPDiv		 .equ 2D5Fh
_SqRoot		 .equ 2DD6h
_RndGuard	 .equ 2E90h
_Round		 .equ 2EA7h
_Factorial	 .equ 2EF9h
_LnX		 .equ 2F46h
_LogX		 .equ 2F4Ah
_EToX		 .equ 311Ch
_TenX		 .equ 3121h
_CubeRoot	 .equ 3211h
_YToX		 .equ 3224h
_Sin		 .equ 32CCh
_Cos		 .equ 32CFh
_Tan		 .equ 32D3h
_TanH		 .equ 3688h
_CosH		 .equ 368Ch
_SinH		 .equ 3690h
_ACos		 .equ 370Eh
_ATan		 .equ 3712h
_ATan2		 .equ 3715h
_ASin		 .equ 3719h
_ATanH		 .equ 38E8h
_ASinH		 .equ 3930h
_ACosH		 .equ 3940h
_Combinations	 .equ 39C9h
_Permutations	 .equ 3A19h
_PToR		 .equ 3A23h
_RToP		 .equ 3A41h
_CpOP1OP2	 .equ 3A70h
_CompStrsN	 .equ 3A8Bh
_OP2StdToExt	 .equ 3ABBh
_OP1StdToExt	 .equ 3ADAh
_OP1ExtToStd	 .equ 3AEBh
_MovFPFrOP1	 .equ 3B21h
_Mov8FrOP1	 .equ 3B24h
_OP3ToOP4	 .equ 3B29h
_OP1ToOP4	 .equ 3B2Eh
_OP2ToOP4	 .equ 3B33h
_Mov10ToOP4	 .equ 3B36h
_OP4ToOP2	 .equ 3B3Bh
_OP3ToOP2	 .equ 3B40h
_OP1ToOP3	 .equ 3B4Fh
_Mov10FrOP1	 .equ 3B52h
_OP5ToOP2	 .equ 3B57h
_OP1ToOP2	 .equ 3B5Ch
_Mov10ToOP2	 .equ 3B5Fh
_TempToOP2	 .equ 3B64h
_TempToOP1	 .equ 3B69h
_OP4ToOP1	 .equ 3B6Eh
_OP5ToOP1	 .equ 3B73h
_OP3ToOP1	 .equ 3B78h
_OP4ToOP5	 .equ 3B7Dh
_OP2ToOP5	 .equ 3B82h
_Mov10ToOP5	 .equ 3B85h
_OP2ToTemp	 .equ 3B8Ah
_OP1ToTemp	 .equ 3B8Fh
_Mov10ToTemp	 .equ 3B92h
_OP1ToOP5	 .equ 3B97h
_OP2ToOP1	 .equ 3B9Ch
_Mov10ToOP1	 .equ 3B9Fh
_Mov10B		 .equ 3BA2h
_Mov9B		 .equ 3BA4h
_Mov7B		 .equ 3BA8h
_Mov6B		 .equ 3BAAh
_Mov5B		 .equ 3BACh
_Mov4B		 .equ 3BAEh
_Mov3B		 .equ 3BB0h
_Mov2B		 .equ 3BB2h
_Mov1B		 .equ 3BB4h
_Mov8ToOP1	 .equ 3BB7h
_Mov8ToOP2	 .equ 3BBCh
_OP2Set3	 .equ 3BF3h
_OP1Set1	 .equ 3BFBh
_OP1Set2	 .equ 3C00h
_OP2Set2	 .equ 3C05h
_OP2Set1	 .equ 3C10h
_OP1Set0	 .equ 3C2Bh
_ZeroOP1	 .equ 3C3Bh
_ZeroOP2	 .equ 3C40h
_ZeroOP3	 .equ 3C45h
_ClrLp		 .equ 3C4Ch
_ShRAcc		 .equ 3C53h
_ShLAcc		 .equ 3C5Bh
_OP1ExOP3	 .equ 3E02h
_OP1ExOP4	 .equ 3E07h
_OP1ExOP2	 .equ 3E0Ch
_ExLp		 .equ 3E14h
_CkOP2FP0	 .equ 3E23h
_CkOP1Int	 .equ 3E28h
_CkOP1Odd	 .equ 3E2Fh
_GetConOP1	 .equ 3EB1h
_GetConOP2	 .equ 3EB6h
_CkOP2Pos	 .equ 3F4Bh
_CkOP1Pos	 .equ 3F50h
_ClrOP1S	 .equ 3F56h
_DecO1Exp	 .equ 3F5Dh
_IncO1Exp	 .equ 3F65h
_MatSubtract	 .equ 3FF0h
_MatAdd		 .equ 3FF4h
_MatMultiply	 .equ 4040h
_MatTranspose	 .equ 4095h
_CkPosInt	 .equ 40CBh
_MatInvert	 .equ 41E6h
_MatDeterminant	 .equ 41E9h
_MatRndGuard	 .equ 43C2h
_MatRound	 .equ 43C6h
_MatNegate	 .equ 43CDh
_MatTimesScalar	 .equ 43D2h
_DispHL		 .equ 47C3h
_FormReal	 .equ 4C4Dh
_FormEReal	 .equ 4C51h
_BrkInc		 .equ 55E9h
_IncFetch	 .equ 55F0h
_CurFetch	 .equ 55F7h
_NxtFetch	 .equ 5605h
_StoVar		 .equ 5AEAh
_RclVar		 .equ 5B50h
_StoAns		 .equ 5B7Ch
_RclAns		 .equ 5BA0h
_StoMatrix	 .equ 5BCBh
_RclMatrix	 .equ 5C0Ch
_LdBCHLind	 .equ 5E82h
_LdHLind	 .equ 5E86h
_Random		 .equ 6377h
_StoRand	 .equ 63BAh
_PutTokString	 .equ 6716h
_PutSysString	 .equ 6733h
_PutNthFlagStr	 .equ 673Eh
_PutFlagStr	 .equ 6741h
_FindNthFlagStr	 .equ 6752h
_PutMap		 .equ 67A4h
_PutSmallChar	 .equ 6837h
_PopHLDEBCAFRet	 .equ 6887h
_PutSmallFlagStr .equ 6907h
_HomeUp		 .equ 691Eh
_EraseEOL	 .equ 6938h
_EraseEOW	 .equ 6957h
_NewLine	 .equ 6977h
_GetCurLoc	 .equ 69FCh
_Load_LFont	 .equ 6A11h
_SaveTR		 .equ 6A6Ch
_RestoreTR	 .equ 6A95h
_PopHLDEBCRet	 .equ 6AF1h
_ClrWindow	 .equ 6A49h
_ClrScrn	 .equ 6A4Ch
_ClrTxtShd	 .equ 6A5Eh
_GetTokLen	 .equ 6B6Bh
_RunIndicOn	 .equ 6B81h
_RunIndicOff	 .equ 6B8Bh
_tokenStrings	 .equ 6BC4h
_systemStrings	 .equ 6EE9h
_GetStatCount	 .equ 71AFh
_ConvOP1	 .equ 71B5h
_GetStatXPtr	 .equ 71DCh
_GetStatYPtr	 .equ 71E1h
_IsEditEmpty	 .equ 7E04h
_IsAtBtm	 .equ 7E0Fh

 .endif ; TI81_VER_1_6K

;---- Equates for version 1.1K (see above for descriptions) ----

 .ifdef TI81_VER_1_1K

_ApdSetup	 .equ 04C7h
_SetXXOP1	 .equ 0896h
_SetXXOP2	 .equ 089Bh
_WToV		 .equ 08AFh
_VToW		 .equ 08BBh
_YIToF		 .equ 08C5h
_XIToF		 .equ 08CDh
_XFToI		 .equ 08E5h
_YFToI		 .equ 08EDh
_HideCursor	 .equ 0D1Bh
_CursorOn	 .equ 0D60h
_ShowCursor	 .equ 0D9Eh
_FormDisp	 .equ 0E06h
_FormERealNoPad	 .equ 0EF7h
_GetFormStrNoPad .equ 0EFAh
_DispTail	 .equ 101Dh
_LdDEHLind	 .equ 103Eh
_SaveShadow	 .equ 1047h
_RstrShadow	 .equ 1058h
_BufLeft	 .equ 1085h
_BufRight	 .equ 1097h
_BufInsert	 .equ 10B0h
_BufReplace	 .equ 10C2h
_BufDelete	 .equ 10D9h
_BufToBtm	 .equ 10E9h
_BufToTop	 .equ 1104h
_BufPeek	 .equ 1140h
_IsAtTop	 .equ 114Dh
_HLequalsDE	 .equ 1154h
_CursorLeft	 .equ 115Ah
_CursorRight	 .equ 1178h
_CursorUp	 .equ 119Eh
_CursorDown	 .equ 11CCh
_JError		 .equ 1296h
_JErrorNo	 .equ 1299h
_DarkLine	 .equ 23E6h
_ILine		 .equ 23E8h
_IPoint		 .equ 2456h
_IOffset	 .equ 249Fh
_IBounds	 .equ 24C8h
_GrBufCpyOrClr	 .equ 24D9h
_KbdScan	 .equ 24FDh
_GetRawCSC	 .equ 254Ah
_ReadKeyGroup	 .equ 2584h
_GetCSC		 .equ 259Bh
_GetKey		 .equ 25A8h
_CanAlphIns	 .equ 26B1h
_CanAlpha	 .equ 26B4h
_SetAlphaLock	 .equ 26BDh
_ResetMatrices	 .equ 2AA8h
_Intgr		 .equ 2AC8h
_Trunc		 .equ 2AD7h
_InvSub		 .equ 2AE0h
_Times2		 .equ 2AE5h
_Plus1		 .equ 2AEAh
_Minus1		 .equ 2AEFh
_FPSub		 .equ 2AF2h
_FPAdd		 .equ 2AF9h
_DToR		 .equ 2B52h
_RToD		 .equ 2B56h
_Cube		 .equ 2B5Ch
_FPSquare	 .equ 2B61h
_FPMult		 .equ 2B64h
_InvOP1S	 .equ 2C44h
_Frac		 .equ 2C4Fh
_DivBy2		 .equ 2C86h
_FPRecip	 .equ 2C8Bh
_FPDiv		 .equ 2C91h
_SqRoot		 .equ 2D08h
_RndGuard	 .equ 2DC2h
_Round		 .equ 2DD9h
_Factorial	 .equ 2E2Bh
_LnX		 .equ 2E79h
_LogX		 .equ 2E7Dh
_EToX		 .equ 304Fh
_TenX		 .equ 3054h
_CubeRoot	 .equ 313Dh
_YToX		 .equ 3150h
_Sin		 .equ 31F8h
_Cos		 .equ 31FBh
_Tan		 .equ 31FFh
_TanH		 .equ 35B4h
_CosH		 .equ 35B8h
_SinH		 .equ 35BCh
_ACos		 .equ 363Ah
_ATan		 .equ 363Eh
_ATan2		 .equ 3641h
_ASin		 .equ 3645h
_ATanH		 .equ 3816h
_ASinH		 .equ 385Eh
_ACosH		 .equ 386Eh
_Combinations	 .equ 38F7h
_Permutations	 .equ 3947h
_PToR		 .equ 3951h
_RToP		 .equ 396Fh
_CpOP1OP2	 .equ 399Eh
_CompStrsN	 .equ 39B9h
_OP2StdToExt	 .equ 39E9h
_OP1StdToExt	 .equ 3A08h
_OP1ExtToStd	 .equ 3A19h
_MovFPFrOP1	 .equ 3A4Fh
_Mov8FrOP1	 .equ 3A52h
_OP3ToOP4	 .equ 3A57h
_OP1ToOP4	 .equ 3A5Ch
_OP2ToOP4	 .equ 3A61h
_Mov10ToOP4	 .equ 3A64h
_OP4ToOP2	 .equ 3A69h
_OP3ToOP2	 .equ 3A6Eh
_OP1ToOP3	 .equ 3A7Dh
_Mov10FrOP1	 .equ 3A80h
_OP5ToOP2	 .equ 3A85h
_OP1ToOP2	 .equ 3A8Ah
_Mov10ToOP2	 .equ 3A8Dh
_TempToOP2	 .equ 3A92h
_TempToOP1	 .equ 3A97h
_OP4ToOP1	 .equ 3A9Ch
_OP5ToOP1	 .equ 3AA1h
_OP3ToOP1	 .equ 3AA6h
_OP4ToOP5	 .equ 3AABh
_OP2ToOP5	 .equ 3AB0h
_Mov10ToOP5	 .equ 3AB3h
_OP2ToTemp	 .equ 3AB8h
_OP1ToTemp	 .equ 3ABDh
_Mov10ToTemp	 .equ 3AC0h
_OP1ToOP5	 .equ 3AC5h
_OP2ToOP1	 .equ 3ACAh
_Mov10ToOP1	 .equ 3ACDh
_Mov10B		 .equ 3AD0h
_Mov9B		 .equ 3AD2h
_Mov7B		 .equ 3AD6h
_Mov6B		 .equ 3AD8h
_Mov5B		 .equ 3ADAh
_Mov4B		 .equ 3ADCh
_Mov3B		 .equ 3ADEh
_Mov2B		 .equ 3AE0h
_Mov1B		 .equ 3AE2h
_Mov8ToOP1	 .equ 3AE5h
_Mov8ToOP2	 .equ 3AEAh
_OP2Set3	 .equ 3B21h
_OP1Set1	 .equ 3B29h
_OP1Set2	 .equ 3B2Eh
_OP2Set2	 .equ 3B33h
_OP2Set1	 .equ 3B3Eh
_OP1Set0	 .equ 3B59h
_ZeroOP1	 .equ 3B69h
_ZeroOP2	 .equ 3B6Eh
_ZeroOP3	 .equ 3B73h
_ClrLp		 .equ 3B7Ah
_ShRAcc		 .equ 3B81h
_ShLAcc		 .equ 3B89h
_OP1ExOP3	 .equ 3D30h
_OP1ExOP4	 .equ 3D35h
_OP1ExOP2	 .equ 3D3Ah
_ExLp		 .equ 3D42h
_CkOP2FP0	 .equ 3D51h
_CkOP1Int	 .equ 3D56h
_CkOP1Odd	 .equ 3D5Dh
_GetConOP1	 .equ 3DDFh
_GetConOP2	 .equ 3DE4h
_CkOP2Pos	 .equ 3E79h
_CkOP1Pos	 .equ 3E7Eh
_ClrOP1S	 .equ 3E84h
_DecO1Exp	 .equ 3E8Bh
_IncO1Exp	 .equ 3E93h
_MatSubtract	 .equ 3F1Eh
_MatAdd		 .equ 3F22h
_MatMultiply	 .equ 3F6Eh
_MatTranspose	 .equ 3FC3h
_CkPosInt	 .equ 3FF9h
_MatInvert	 .equ 4114h
_MatDeterminant	 .equ 4117h
_MatRndGuard	 .equ 4301h
_MatRound	 .equ 4305h
_MatNegate	 .equ 430Ch
_MatTimesScalar	 .equ 4311h
_DispHL		 .equ 4708h
_FormReal	 .equ 4B95h
_FormEReal	 .equ 4B99h
_BrkInc		 .equ 55D6h
_IncFetch	 .equ 55DDh
_CurFetch	 .equ 55E4h
_NxtFetch	 .equ 55F2h
_StoVar		 .equ 5AE1h
_RclVar		 .equ 5B47h
_StoAns		 .equ 5B73h
_RclAns		 .equ 5B97h
_StoMatrix	 .equ 5BC2h
_RclMatrix	 .equ 5C03h
_LdBCHLind	 .equ 5E79h
_LdHLind	 .equ 5E7Dh
_Random		 .equ 6379h
_StoRand	 .equ 63BCh
_PutTokString	 .equ 671Bh
_PutSysString	 .equ 6738h
_PutNthFlagStr	 .equ 6743h
_PutFlagStr	 .equ 6746h
_FindNthFlagStr	 .equ 6757h
_PutMap		 .equ 67AAh
_PutSmallChar	 .equ 683Dh
_PopHLDEBCAFRet	 .equ 688Dh
_PutSmallFlagStr .equ 690Eh
_HomeUp		 .equ 6925h
_EraseEOL	 .equ 693Ah
_EraseEOW	 .equ 6959h
_NewLine	 .equ 6979h
_GetCurLoc	 .equ 69FEh
_Load_LFont	 .equ 6A13h
_SaveTR		 .equ 6A6Eh
_RestoreTR	 .equ 6A97h
_PopHLDEBCRet	 .equ 6AF3h
_ClrWindow	 .equ 6A4Bh
_ClrScrn	 .equ 6A4Eh
_ClrTxtShd	 .equ 6A60h
_GetTokLen	 .equ 6B6Dh
_RunIndicOn	 .equ 6B83h
_RunIndicOff	 .equ 6B8Dh
_tokenStrings	 .equ 6BC8h
_systemStrings	 .equ 6EEDh
_GetStatCount	 .equ 71D5h
_ConvOP1	 .equ 71DBh
_GetStatXPtr	 .equ 7202h
_GetStatYPtr	 .equ 7207h
_IsEditEmpty	 .equ 7E29h
_IsAtBtm	 .equ 7E34h

 .endif ; TI81_VER_1_1K


;================ RAM Addresses ================

;---- Equates for all (?) ROM versions ----

videoMem	 .equ 0E000h	; 768 bytes: buffer of LCD contents.
				; Note: on OS 1.x, this buffer is used
				; to refresh the LCD directly.  On OS
				; 2.x, LCD_Copy must be called to
				; update the LCD (this will be done
				; automatically upon calling GetKey,
				; or periodically when the cursor or
				; run indicator is enabled.)

contrast	 .equ 0E300h	; Byte: current contrast level
flags		 .equ 0E301h	; 7 bytes: system flags (also
				; accessible via IY)
kbdScanCode	 .equ 0E308h	; Byte: current valid scan code
				; (buffer for GetCSC; arrow keys and
				; Del will auto-repeat)
kbdLGSC		 .equ 0E309h	; Byte: last single keypress seen at
				; interrupt time
kbdPSC		 .equ 0E30Ah	; Byte: currently pressed key (0 =
				; none; FF = multiple)
kbdWUR		 .equ 0E30Bh	; Byte: key repeating counter
kbdDebncCnt	 .equ 0E30Ch	; Byte: key debouncing counter
apdSubTimer	 .equ 0E30Dh	; Byte: APD counter (LSB)
apdTimer	 .equ 0E30Eh	; Byte: APD counter (MSB)
apdWarmUp	 .equ 0E30Fh	; Byte: Delay before halting CPU
indicCounter	 .equ 0E310h	; Byte: Run indicator counter
curTime		 .equ 0E311h	; Byte: Cursor counter
curRow		 .equ 0E312h	; Byte: Cursor row
curCol		 .equ 0E313h	; Byte: Cursor column
curUnder	 .equ 0E314h	; Byte: Character "underneath" cursor

OP1		 .equ 0E315h	; 10 bytes: floating-point register
OP2		 .equ 0E31Fh	; 10 bytes: floating-point register
OP3		 .equ 0E329h	; 10 bytes: floating-point register
OP4		 .equ 0E333h	; 10 bytes: floating-point register
OP5		 .equ 0E33Dh	; 10 bytes: floating-point register

errSP		 .equ 0E34Ah	; Word: stack pointer of active error
				; handler (0 = none; note that error
				; handlers are not nested as they are
				; on other calculators)

; addresses between E34F and F146, inclusive, differ between versions
; (see below)

fmtMode		 .equ 0F147h	; Byte: current number format mode
				; (0 = Norm, 1 = Sci, 3 = Eng)
fmtDigits	 .equ 0F148h	; Byte: number of digits of precision
				; displayed at home screen

errNo		 .equ 0F149h	; Byte: most recent error number

begPC		 .equ 0F1CCh	; Word: start of current BASIC
				; program/expression
curPC		 .equ 0F1CEh	; Word: address of token currently
				; being evaluated
endPC		 .equ 0F1D0h	; Word: end of current BASIC
				; program/expression

prgm0Name	 .equ 0F1D3h	; 8 bytes: prgm0 name (a string of
				; tokens, padded with spaces)
prgm1Name	 .equ 0F1DBh	; 8 bytes: prgm1 name
prgm2Name	 .equ 0F1E3h	; ...
prgm3Name	 .equ 0F1EBh
prgm4Name	 .equ 0F1F3h
prgm5Name	 .equ 0F1FBh
prgm6Name	 .equ 0F203h
prgm7Name	 .equ 0F20Bh
prgm8Name	 .equ 0F213h
prgm9Name	 .equ 0F21Bh
prgmAName	 .equ 0F223h
prgmBName	 .equ 0F22Bh
prgmCName	 .equ 0F233h
prgmDName	 .equ 0F23Bh
prgmEName	 .equ 0F243h
prgmFName	 .equ 0F24Bh
prgmGName	 .equ 0F253h
prgmHName	 .equ 0F25Bh
prgmIName	 .equ 0F263h
prgmJName	 .equ 0F26Bh
prgmKName	 .equ 0F273h
prgmLName	 .equ 0F27Bh
prgmMName	 .equ 0F283h
prgmNName	 .equ 0F28Bh
prgmOName	 .equ 0F293h
prgmPName	 .equ 0F29Bh
prgmQName	 .equ 0F2A3h
prgmRName	 .equ 0F2ABh
prgmSName	 .equ 0F2B3h
prgmTName	 .equ 0F2BBh
prgmUName	 .equ 0F2C3h
prgmVName	 .equ 0F2CBh
prgmWName	 .equ 0F2D3h
prgmXName	 .equ 0F2DBh
prgmYName	 .equ 0F2E3h
prgmZName	 .equ 0F2EBh
prgmThetaName	 .equ 0F2F3h

prgm0Start	 .equ 0F2FBh	; Word: address of start of prgm0
prgm1Start	 .equ 0F2FDh	; Word: address of start of prgm1 / end of prgm0
prgm2Start	 .equ 0F2FFh	; ...
prgm3Start	 .equ 0F301h
prgm4Start	 .equ 0F303h
prgm5Start	 .equ 0F305h
prgm6Start	 .equ 0F307h
prgm7Start	 .equ 0F309h
prgm8Start	 .equ 0F30Bh
prgm9Start	 .equ 0F30Dh
prgmAStart	 .equ 0F30Fh
prgmBStart	 .equ 0F311h
prgmCStart	 .equ 0F313h
prgmDStart	 .equ 0F315h
prgmEStart	 .equ 0F317h
prgmFStart	 .equ 0F319h
prgmGStart	 .equ 0F31Bh
prgmHStart	 .equ 0F31Dh
prgmIStart	 .equ 0F31Fh
prgmJStart	 .equ 0F321h
prgmKStart	 .equ 0F323h
prgmLStart	 .equ 0F325h
prgmMStart	 .equ 0F327h
prgmNStart	 .equ 0F329h
prgmOStart	 .equ 0F32Bh
prgmPStart	 .equ 0F32Dh
prgmQStart	 .equ 0F32Fh
prgmRStart	 .equ 0F331h
prgmSStart	 .equ 0F333h
prgmTStart	 .equ 0F335h
prgmUStart	 .equ 0F337h
prgmVStart	 .equ 0F339h
prgmWStart	 .equ 0F33Bh
prgmXStart	 .equ 0F33Dh
prgmYStart	 .equ 0F33Fh
prgmZStart	 .equ 0F341h
prgmThetaStart	 .equ 0F343h
prgmThetaEnd	 .equ 0F345h	; Word: address of end of prgmTheta

progMem		 .equ 0F347h	; Start of user program/list memory
progMemEnd	 .equ 0FCA6h	; End of user program/list memory

y1Start		 .equ 0FCB1h	; Word: address of start of Y1
y2Start		 .equ 0FCB3h	; Word: address of start of Y2 / end of Y1
y3Start		 .equ 0FCB5h	; Word: address of start of Y3 / end of Y2
y4Start		 .equ 0FCB7h	; Word: address of start of Y4 / end of Y3
x1tStart	 .equ 0FCB9h	; Word: address of start of X1T / end of Y4
y1tStart	 .equ 0FCBBh	; Word: address of start of Y1T / end of X1T
x2tStart	 .equ 0FCBDh	; Word: address of start of X2T / end of Y1T
y2tStart	 .equ 0FCBFh	; Word: address of start of Y2T / end of X2T
x3tStart	 .equ 0FCC1h	; Word: address of start of X3T / end of Y2T
y3tStart	 .equ 0FCC3h	; Word: address of start of Y3T / end of X3T
y3tEnd		 .equ 0FCC5h	; Word: address of end of Y3T

equMem		 .equ 0FCC7h	; Start of user equation memory
equMemEnd	 .equ 0FE2Eh	; End of user equation memory

onCheckSum	 .equ 0FE2Fh	; Word: checksum of RAM


;---- Equates for versions 1.6K to 2.0V ----

 .ifdef TI81_VER_MAJOR
  .if (TI81_VER_MAJOR >= 2) || (TI81_VER_MINOR > 1)

tempMatrix	 .equ 0E34Fh	; 288 bytes: used by parser for
				; intermediate matrix results; usually
				; free for assembly programs' use

scratchMem	 .equ 0E46Fh	; 288 bytes: temporary memory area
				; used as scratch space by many system
				; routines (notably, the first 5 bytes
				; are used by PutC/PutMap to store a
				; copy of the font bitmap)

editTop		 .equ 0E58Fh	; Word: address of start of current
				; edit buffer
editCursor	 .equ 0E591h	; Word: address of current cursor
				; position (end of buffer head / start
				; of gap)
editTail	 .equ 0E593h	; Word: address of end of buffer gap
				; (start of tail)
editBtm		 .equ 0E595h	; Word: address of end of current edit
				; buffer

cmdShadow	 .equ 0E69Dh	; 128 bytes: backup of home screen
cmdShadCur	 .equ 0E71Dh	; 2 bytes: backup of home screen
				; cursor position

indicMem	 .equ 0E729h	; 8 bytes: backup of top right corner
				; of screen

textShadow	 .equ 0E732h	; 128 bytes: backup of text on screen
textShadCur	 .equ 0E7B2h	; 2 bytes: backup of cursor position
textShadIns	 .equ 0E7B4h	; Byte: backup of textFlags

curGStyle	 .equ 0E7B5h	; Byte: graph cursor style (bits 0-2 =
				; pattern for blinking portion; bits
				; 4-7 = pattern for non-blinking
				; portion.)  Special cursor types are
				; 32h for "zoom box" mode and B2h for
				; "draw line" mode.
curGY		 .equ 0E7B6h	; Byte: Y coordinate of graph cursor
curGX		 .equ 0E7B7h	; Byte: X coordinate of graph cursor
curGY2		 .equ 0E7B8h	; Byte: Y coordinate of selected point
curGX2		 .equ 0E7B9h	; Byte: X coordinate of selected point

plotSScreen	 .equ 0E7C3h	; 768 bytes: current graph contents

Tmin		 .equ 0EAD3h	; 8 bytes (FP): Tmin value
Tmax		 .equ 0EADBh	; 8 bytes (FP): Tmax value
Tstep		 .equ 0EAE3h	; 8 bytes (FP): Tstep value
Xmin		 .equ 0EAEBh	; 8 bytes (FP): Xmin value - be
				; careful when modifying this or any
				; of the other window range variables;
				; it's probably easiest and safest to
				; use StoVar instead of modifying the
				; values directly
Xmax		 .equ 0EAF3h	; 8 bytes (FP): Xmax value
Xscl		 .equ 0EAFBh	; 8 bytes (FP): Xscl value
Ymin		 .equ 0EB03h	; 8 bytes (FP): Ymin value
Ymax		 .equ 0EB0Bh	; 8 bytes (FP): Ymax value
Yscl		 .equ 0EB13h	; 8 bytes (FP): Yscl value
Xres		 .equ 0EB1Bh	; 8 bytes (FP): Xres value

realA		 .equ 0EB55h	; 8 bytes (FP): real variable A
realB		 .equ 0EB5Dh	; 8 bytes (FP): real variable B
realC		 .equ 0EB65h	; ...
realD		 .equ 0EB6Dh
realE		 .equ 0EB75h
realF		 .equ 0EB7Dh
realG		 .equ 0EB85h
realH		 .equ 0EB8Dh
realI		 .equ 0EB95h
realJ		 .equ 0EB9Dh
realK		 .equ 0EBA5h
realL		 .equ 0EBADh
realM		 .equ 0EBB5h
realN		 .equ 0EBBDh
realO		 .equ 0EBC5h
realP		 .equ 0EBCDh
realQ		 .equ 0EBD5h
realR		 .equ 0EBDDh
realS		 .equ 0EBE5h
realT		 .equ 0EBEDh
realU		 .equ 0EBF5h
realV		 .equ 0EBFDh
realW		 .equ 0EC05h
realX		 .equ 0EC0Dh
realY		 .equ 0EC15h
realZ		 .equ 0EC1Dh
realTheta	 .equ 0EC25h	; 8 bytes (FP): real variable theta

matArow		 .equ 0EC2Dh	; Byte: number of rows of [A]
matAcol		 .equ 0EC2Eh	; Byte: number of columns of [A]
matA		 .equ 0EC2Fh	; 288 bytes: matrix [A] entries
matBrow		 .equ 0ED4Fh	; Byte: number of rows of [B]
matBcol		 .equ 0ED50h	; Byte: number of columns of [B]
matB		 .equ 0ED51h	; 288 bytes: matrix [B] entries
matCrow		 .equ 0EE71h	; Byte: number of rows of [C]
matCcol		 .equ 0EE72h	; Byte: number of columns of [C]
matC		 .equ 0EE73h	; 288 bytes: matrix [C] entries
AnsRow		 .equ 0EF93h	; Byte: number of rows of Ans, if Ans
				; is a matrix; zero if Ans is a real
				; number
AnsCol		 .equ 0EF94h	; Byte: number of columns of Ans, if
				; Ans is a matrix
Ans		 .equ 0EF95h	; 288 bytes: Ans value

SumX		 .equ 0F0B5h	; 8 bytes (FP): sum of x (Sigma-x)
SumXSqr		 .equ 0F0BDh	; 8 bytes (FP): sum of x^2 (Sigma-x^2)
SumXY		 .equ 0F0C5h	; 8 bytes (FP): sum of x*y (Sigma-xy)
SumY		 .equ 0F0CDh	; 8 bytes (FP): sum of y (Sigma-y)
SumYSqr		 .equ 0F0D5h	; 8 bytes (FP): sum of y^2 (Sigma-y^2)
StatN		 .equ 0F0DDh	; 8 bytes (FP): number of data points (n)
XMean		 .equ 0F0E5h	; 8 bytes (FP): mean x value (x-bar)
StdPX		 .equ 0F0EDh	; 8 bytes (FP): population s.d. (sigma-x)
StdX		 .equ 0F0F5h	; 8 bytes (FP): sample s.d. (Sx)
YMean		 .equ 0F0FDh	; 8 bytes (FP): mean y value (y-bar)
StdPY		 .equ 0F105h	; 8 bytes (FP): population s.d. (sigma-y)
StdY		 .equ 0F10Dh	; 8 bytes (FP): sample s.d. (Sy)
RegA		 .equ 0F115h	; 8 bytes (FP): regression coefficient (a)
RegB		 .equ 0F11Dh	; 8 bytes (FP): regression coefficient (b)
Corr		 .equ 0F125h	; 8 bytes (FP): correlation coeff. (r)
DimX		 .equ 0F12Dh	; 8 bytes (FP): number of entries in
				; stat lists {x} and {y} (Dim{x})


;---- Equates for version 1.1K (see above for descriptions) ----

  .else ; (TI81_VER_MAJOR < 2) && (TI81_VER_MINOR <= 1)

tempMatrix	 .equ 0EB1Fh
scratchMem	 .equ 0EC3Fh
editTop		 .equ 0EF13h
editCursor	 .equ 0EF15h
editTail	 .equ 0EF17h
editBtm		 .equ 0EF19h
cmdShadow	 .equ 0F021h
cmdShadCur	 .equ 0F0A1h
indicMem	 .equ 0F0ADh
textShadow	 .equ 0F0C4h
textShadCur	 .equ 0F144h
textShadIns	 .equ 0F146h
curGStyle	 .equ 0F0B5h
curGY		 .equ 0F0B6h
curGX		 .equ 0F0B7h
curGY2		 .equ 0F0B8h
curGX2		 .equ 0F0B9h
plotSScreen	 .equ 0E34Fh
Tmin		 .equ 0E65Fh
Tmax		 .equ 0E667h
Tstep		 .equ 0E66Fh
Xmin		 .equ 0E677h
Xmax		 .equ 0E67Fh
Xscl		 .equ 0E687h
Ymin		 .equ 0E68Fh
Ymax		 .equ 0E697h
Yscl		 .equ 0E69Fh
Xres		 .equ 0E6A7h
realA		 .equ 0E6E1h
realB		 .equ 0E6E9h
realC		 .equ 0E6F1h
realD		 .equ 0E6F9h
realE		 .equ 0E701h
realF		 .equ 0E709h
realG		 .equ 0E711h
realH		 .equ 0E719h
realI		 .equ 0E721h
realJ		 .equ 0E729h
realK		 .equ 0E731h
realL		 .equ 0E739h
realM		 .equ 0E741h
realN		 .equ 0E749h
realO		 .equ 0E751h
realP		 .equ 0E759h
realQ		 .equ 0E761h
realR		 .equ 0E769h
realS		 .equ 0E771h
realT		 .equ 0E779h
realU		 .equ 0E781h
realV		 .equ 0E789h
realW		 .equ 0E791h
realX		 .equ 0E799h
realY		 .equ 0E7A1h
realZ		 .equ 0E7A9h
realTheta	 .equ 0E7B1h
matArow		 .equ 0E7B9h
matAcol		 .equ 0E7BAh
matA		 .equ 0E7BBh
matBrow		 .equ 0E8DBh
matBcol		 .equ 0E8DCh
matB		 .equ 0E8DDh
matCrow		 .equ 0E9FDh
matCcol		 .equ 0E9FEh
matC		 .equ 0E9FFh
AnsRow		 .equ 0ED5Fh
AnsCol		 .equ 0ED60h
Ans		 .equ 0ED61h
SumX		 .equ 0EE81h
SumXSqr		 .equ 0EE89h
SumXY		 .equ 0EE91h
SumY		 .equ 0EE99h
SumYSqr		 .equ 0EEA1h
StatN		 .equ 0EEA9h
XMean		 .equ 0EEB1h
StdPX		 .equ 0EEB9h
StdX		 .equ 0EEC1h
YMean		 .equ 0EEC9h
StdPY		 .equ 0EED1h
StdY		 .equ 0EED9h
RegA		 .equ 0EEE1h
RegB		 .equ 0EEE9h
Corr		 .equ 0EEF1h
DimX		 .equ 0EEF9h

  .endif ; (TI81_VER_MAJOR < 2) && (TI81_VER_MINOR <= 1)
 .endif ; TI81_VER_MAJOR defined


;================ System Flags ================

;---- IY + 0 ----
trigFlags	.equ 0
apdFlags	.equ 0
parseFlags	.equ 0

trigDeg		.equ 0	; Trig functions use degrees
apdAble		.equ 1	; APD enabled
apdRunning	.equ 2	; APD counter running
parseNumOP1	.equ 5	; OP1 currently holds an intermediate result
			; for the parser

;---- IY + 1 ----
kbdFlags	.equ 1
shiftFlags	.equ 1

kbdSCR 		.equ 0	; Keypad has been scanned
kbdKeyPress 	.equ 1	; kbdScanCode set due to an actual keypress
			; (as opposed to implicit repeating)
shift2nd 	.equ 3	; 2nd mode active
shiftAlpha 	.equ 4	; Alpha mode active
shiftALock 	.equ 5	; Alpha locked on

;---- IY + 2 ----
grfModeFlags	.equ 2
plotFlags	.equ 2

grfModeParam	.equ 0	; Parametric graphing mode
grfModeDot	.equ 1	; Dot graphing mode
grfModeSimul	.equ 2	; Simultaneous graphing mode
grfModeGrid	.equ 3	; Draw grid
grfModePolar	.equ 4	; Show polar coordinates
plotLoc		.equ 6	; Drawing commands (e.g., IPoint, ILine) draw
			; only to videoMem; otherwise they draw to
			; both plotSScreen and videoMem

;---- IY + 3 ----
graphFlags	.equ 3

graphDraw	.equ 0	; Graph needs to be redrawn
graphProg	.equ 1	; Currently drawing graph
graphDisp	.equ 3	; Graph currently visible

;---- IY + 4 ----
textFlags	.equ 4

textLeftSide	.equ 0	; Draw small font text on left-hand side of
			; byte (i.e., most significant nibble)
textInverse	.equ 1	; Draw text in reverse video
textSave	.equ 2	; Save text in textShadow
textInsMode	.equ 3	; Insertion mode
textAutoScroll	.equ 4	; Scroll screen upon reaching bottom
textScrolled	.equ 5	; Screen has been scrolled

;---- IY + 5 ----
cmdFlags	.equ 5
trIndicFlags	.equ 5
onFlags		.equ 5
curFlags	.equ 5

cmdVirgin	.equ 1	; Nothing typed yet in edit
trIndicInUse	.equ 2	; Top right indicator active
onRunning	.equ 3	; Calculator turned on
onInterrupt	.equ 4	; On key pressed while running
curWord		.equ 5	; Mode-screen-style blinking-word cursor
curAble		.equ 6	; Cursor enabled
curOn		.equ 7	; Cursor currently visible

;---- IY + 6 ----
indicFlags	.equ 6

indicRun	.equ 0	; Run indicator enabled

;---- Other variables accessible via IY ----
kbdScanCode_off		.equ kbdScanCode - flags
kbdLGSC_off		.equ kbdLGSC - flags
kbdPSC_off		.equ kbdPSC - flags
kbdWUR_off		.equ kbdWUR - flags
kbdDebncCnt_off		.equ kbdDebncCnt - flags
apdSubTimer_off		.equ apdSubTimer - flags
apdTimer_off		.equ apdTimer - flags
apdWarmUp_off		.equ apdWarmUp - flags
indicCounter_off	.equ indicCounter - flags
curTime_off		.equ curTime - flags
curRow_off		.equ curRow - flags
curCol_off		.equ curCol - flags
curUnder_off		.equ curUnder - flags
OP1_off			.equ OP1 - flags
OP2_off			.equ OP2 - flags
OP3_off			.equ OP3 - flags
OP4_off			.equ OP4 - flags
OP5_off			.equ OP5 - flags


;================ Keypad Scan Codes ================

skDown		.equ 01h
skLeft		.equ 02h
skRight		.equ 03h
skUp		.equ 04h
skEnter		.equ 09h
skAdd		.equ 0Ah
skSub		.equ 0Bh
skMul		.equ 0Ch
skDiv		.equ 0Dh
skPower		.equ 0Eh
skClear		.equ 0Fh
skChs		.equ 11h
sk3		.equ 12h
sk6		.equ 13h
sk9		.equ 14h
skRParen	.equ 15h
skTan		.equ 16h
skVars		.equ 17h
skDecPnt	.equ 19h
sk2		.equ 1Ah
sk5		.equ 1Bh
sk8		.equ 1Ch
skLParen	.equ 1Dh
skCos		.equ 1Eh
skPrgm		.equ 1Fh
skMode		.equ 20h
sk0		.equ 21h
sk1		.equ 22h
sk4		.equ 23h
sk7		.equ 24h
skEE		.equ 25h
skSin		.equ 26h
skMatrix	.equ 27h
skGraphvar	.equ 28h
skStore		.equ 2Ah
skLn		.equ 2Bh
skLog		.equ 2Ch
skSquare	.equ 2Dh
skRecip		.equ 2Eh
skMath		.equ 2Fh
skAlpha		.equ 30h
skGraph		.equ 31h
skTrace		.equ 32h
skZoom		.equ 33h
skRange		.equ 34h
skYEqu		.equ 35h
sk2nd		.equ 36h
skIns		.equ 37h
skDel		.equ 38h


;================ GetKey Keypress Values ================
; (note that these are equal to the corresponding token values)

kRight		.equ 01h
kLeft		.equ 02h
kUp		.equ 03h
kDown		.equ 04h
kIns		.equ 05h
kDel		.equ 06h

k0		.equ 10h
k1		.equ 11h
k2		.equ 12h
k3		.equ 13h
k4		.equ 14h
k5		.equ 15h
k6		.equ 16h
k7		.equ 17h
k8		.equ 18h
k9		.equ 19h
kDecPt		.equ 1Ah
kEE		.equ 1Bh
kStore		.equ 1Ch
kRParen		.equ 1Dh
kComma		.equ 1Eh
kLParen		.equ 1Fh
kAdd		.equ 26h
kSub		.equ 27h
kMul		.equ 28h
kDiv		.equ 29h
kQuote		.equ 2Ch
kChs		.equ 30h
kAbs		.equ 31h
kSqrt		.equ 32h
kLn		.equ 34h
kExp		.equ 35h
kLog		.equ 36h
kALog		.equ 37h
kSin		.equ 38h
kASin		.equ 39h
kCos		.equ 3Ah
kACos		.equ 3Bh
kTan		.equ 3Ch
kATan		.equ 3Dh
kQuest		.equ 45h
kPower		.equ 46h
kRecip		.equ 47h
kSqr		.equ 48h
kSpace		.equ 56h
kAns		.equ 57h
kA		.equ 59h
kB		.equ 5Ah
kC		.equ 5Bh
kD		.equ 5Ch
kE		.equ 5Dh
kF		.equ 5Eh
kG		.equ 5Fh
kH		.equ 60h
kI		.equ 61h
kJ		.equ 62h
kK		.equ 63h
kL		.equ 64h
kM		.equ 65h
kN		.equ 66h
kO		.equ 67h
kP		.equ 68h
kQ		.equ 69h
kR		.equ 6Ah
kS		.equ 6Bh
kT		.equ 6Ch
kU		.equ 6Dh
kV		.equ 6Eh
kW		.equ 6Fh
kX		.equ 70h
kY		.equ 71h
kZ		.equ 72h
kTheta		.equ 73h
kListX		.equ 84h
kListY		.equ 85h
kMatA		.equ 86h
kMatB		.equ 87h
kMatC		.equ 88h
kPi		.equ 99h

kQuit		.equ 0E6h
kLastEnt	.equ 0EBh
kEnter		.equ 0ECh
kYequ		.equ 0EDh
kRange		.equ 0EEh
kTrace		.equ 0EFh
kGraph		.equ 0F0h
kClear		.equ 0F1h
kMode		.equ 0F2h
kZoom		.equ 0F3h
kDraw		.equ 0F4h
kMath		.equ 0F5h
kTest		.equ 0F6h
kYVars		.equ 0F7h
kVars		.equ 0F8h
kMatrix		.equ 0F9h
kStat		.equ 0FAh
kReset		.equ 0FBh
kPrgm		.equ 0FCh
kVarx		.equ 0FFh


;================ Token Values ================

tProg		.equ 07h	; Prgm
t0		.equ 10h	; 0
t1		.equ 11h	; 1
t2		.equ 12h	; 2
t3		.equ 13h	; 3
t4		.equ 14h	; 4
t5		.equ 15h	; 5
t6		.equ 16h	; 6
t7		.equ 17h	; 7
t8		.equ 18h	; 8
t9		.equ 19h	; 9
tDecPt		.equ 1Ah	; .
tEE		.equ 1Bh	; E
tStore		.equ 1Ch	; ->
tRParen		.equ 1Dh	; )
tComma		.equ 1Eh	; ,
tLParen		.equ 1Fh	; (
tEQ		.equ 20h	; =
tNE		.equ 21h	; !=
tLT		.equ 22h	; <
tLE		.equ 23h	; <=
tGT		.equ 24h	; >
tGE		.equ 25h	; >=
tAdd		.equ 26h	; +
tSub		.equ 27h	; -
tMul		.equ 28h	; *
tDiv		.equ 29h	; /
tnCr		.equ 2Ah	; _nCr_
tnPr		.equ 2Bh	; _nPr_
tQuote		.equ 2Ch	; "
tIPart		.equ 2Dh	; IPart_
tInt		.equ 2Eh	; Int_
tFPart		.equ 2Fh	; FPart_
tChs		.equ 30h	; (-)
tAbs		.equ 31h	; abs_
tSqrt		.equ 32h	; (square root)
tCubRt		.equ 33h	; (cube root)
tLn		.equ 34h	; ln_
tExp		.equ 35h	; e^
tLog		.equ 36h	; log_
tALog		.equ 37h	; 10^
tSin		.equ 38h	; sin_
tASin		.equ 39h	; sin^-1_
tCos		.equ 3Ah	; cos_
tACos		.equ 3Bh	; cos^-1_
tTan		.equ 3Ch	; tan_
tATan		.equ 3Dh	; tan^-1_
tSinH		.equ 3Eh	; sinh_
tASinH		.equ 3Fh	; sinh^-1_
tCosH		.equ 40h	; cosh_
tACosH		.equ 41h	; cosh^-1_
tTanH		.equ 42h	; tanh_
tATanH		.equ 43h	; tanh^-1_
tDet		.equ 44h	; det
tQuest		.equ 45h	; ?
tPower		.equ 46h	; ^
tRecip		.equ 47h	; ^-1
tSqr		.equ 48h	; ^2
tTrnspos	.equ 49h	; ^T
tCube		.equ 4Ah	; ^3
tFact		.equ 4Bh	; !
tFromRad	.equ 4Ch	; ^r
tFromDeg	.equ 4Dh	; ^o (degrees)
tRound		.equ 4Eh	; Round(
tRToP		.equ 4Fh	; R>P(
tPToR		.equ 50h	; P>R(
tRowSwap	.equ 51h	; RowSwap(
tRowPlus	.equ 52h	; Row+(
tmRow		.equ 53h	; *Row(
tmRowPlus	.equ 54h	; *Row+(
tNDeriv		.equ 55h	; NDeriv(
tSpace		.equ 56h	; _ (space)
tAns		.equ 57h	; Ans
tRand		.equ 58h	; Rand
tA		.equ 59h	; A
tB		.equ 5Ah	; B
tC		.equ 5Bh	; C
tD		.equ 5Ch	; D
tE		.equ 5Dh	; E
tF		.equ 5Eh	; F
tG		.equ 5Fh	; G
tH		.equ 60h	; H
tI		.equ 61h	; I
tJ		.equ 62h	; J
tK		.equ 63h	; K
tL		.equ 64h	; L
tM		.equ 65h	; M
tN		.equ 66h	; N
tO		.equ 67h	; O
tP		.equ 68h	; P
tQ		.equ 69h	; Q
tR		.equ 6Ah	; R
tS		.equ 6Bh	; S
tT		.equ 6Ch	; T
tU		.equ 6Dh	; U
tV		.equ 6Eh	; V
tW		.equ 6Fh	; W
tX		.equ 70h	; X
tY		.equ 71h	; Y
tZ		.equ 72h	; Z
tTheta		.equ 73h	; theta
tArow		.equ 74h	; Arow
tAcol		.equ 75h	; Acol
tBrow		.equ 76h	; Brow
tBcol		.equ 77h	; Bcol
tCrow		.equ 78h	; Crow
tCcol		.equ 79h	; Ccol
tTmin		.equ 7Ah	; Tmin
tTmax		.equ 7Bh	; Tmax
tTstep		.equ 7Ch	; Tstep
tXmin		.equ 7Dh	; Xmin
tXmax		.equ 7Eh	; Xmax
tXscl		.equ 7Fh	; Xscl
tYmin		.equ 80h	; Ymin
tYmax		.equ 81h	; Ymax
tYscl		.equ 82h	; Yscl
tXres		.equ 83h	; Xres
tListX		.equ 84h	; {x}(
tListY		.equ 85h	; {y}(
tMatA		.equ 86h	; [A]
tMatB		.equ 87h	; [B]
tMatC		.equ 88h	; [C]
tSumX		.equ 89h	; Sigma-x
tSumXSqr	.equ 8Ah	; Sigma-x^2
tSumXY		.equ 8Bh	; Sigma-xy
tSumY		.equ 8Ch	; Sigma-y
tSumYSqr	.equ 8Dh	; Sigma-y^2
tStatN		.equ 8Eh	; n
tXMean		.equ 8Fh	; x-bar
tStdPX		.equ 90h	; sigma-x
tStdX		.equ 91h	; Sx
tYMean		.equ 92h	; y-bar
tStdPY		.equ 93h	; sigma-y
tStdY		.equ 94h	; Sy
tRegB		.equ 95h	; b
tRegA		.equ 96h	; a
tCorr		.equ 97h	; r
tDimX		.equ 98h	; Dim{x}
tPi		.equ 99h	; pi
tOneVar		.equ 9Ah	; 1-Var
tLR		.equ 9Bh	; LinReg
tLRExp		.equ 9Ch	; ExpReg
tLRLn		.equ 9Dh	; LnReg
tLRPwr		.equ 9Eh	; PwrReg
tXSort		.equ 9Fh	; xSort
tYSort		.equ 0A0h	; ySort
tClrStat	.equ 0A1h	; ClrStat
tRegEq		.equ 0A2h	; RegEQ
tHist		.equ 0A3h	; Hist
txyLine		.equ 0A4h	; xyLine
tScatter	.equ 0A5h	; Scatter
tRad		.equ 0A6h	; Rad
tDeg		.equ 0A7h	; Deg
tNormF		.equ 0A8h	; Norm
tSci		.equ 0A9h	; Sci
tEng		.equ 0AAh	; Eng
tFloat		.equ 0ABh	; Float
tFix		.equ 0ACh	; Fix_
tStndrd		.equ 0ADh	; Function
tParam		.equ 0AEh	; Param
tDrawLine	.equ 0AFh	; Connected
tDrawDot	.equ 0B0h	; Dot
tSeq		.equ 0B1h	; Sequence
tSimulG		.equ 0B2h	; Simul
tGridOff	.equ 0B3h	; Grid Off
tGridOn		.equ 0B4h	; Grid On
tRectG		.equ 0B5h	; Rect
tPolarG		.equ 0B6h	; Polar
tDisp		.equ 0B7h	; Disp_
tInput		.equ 0B8h	; Input_
tPause		.equ 0B9h	; Pause
tEnd		.equ 0BAh	; End
tStop		.equ 0BBh	; Stop
tLbl		.equ 0BCh	; Lbl_
tGoto		.equ 0BDh	; Goto_
tIf		.equ 0BEh	; If_
tISG		.equ 0BFh	; IS>(
tDSL		.equ 0C0h	; DS<(
tY1		.equ 0C1h	; Y1
tY2		.equ 0C2h	; Y2
tY3		.equ 0C3h	; Y3
tY4		.equ 0C4h	; Y4
tX1T		.equ 0C5h	; X1t
tY1T		.equ 0C6h	; Y1t
tX2T		.equ 0C7h	; X2t
tY2T		.equ 0C8h	; Y2t
tX3T		.equ 0C9h	; X3t
tY3T		.equ 0CAh	; Y3t
tAllOn		.equ 0CBh	; All-On
tY1On		.equ 0CCh	; Y1-On
tY2On		.equ 0CDh	; Y2-On
tY3On		.equ 0CEh	; Y3-On
tY4On		.equ 0CFh	; Y4-On
tX1TOn		.equ 0D0h	; X1t-On
tX2TOn		.equ 0D1h	; X2t-On
tX3TOn		.equ 0D2h	; X3t-On
tAllOff		.equ 0D3h	; All-Off
tY1Off		.equ 0D4h	; Y1-Off
tY2Off		.equ 0D5h	; Y2-Off
tY3Off		.equ 0D6h	; Y3-Off
tY4Off		.equ 0D7h	; Y4-Off
tX1TOff		.equ 0D8h	; X1t-Off
tX2TOff		.equ 0D9h	; X2t-Off
tX3TOff		.equ 0DAh	; X3t-Off
tLine		.equ 0DBh	; Line(
tPtOn		.equ 0DCh	; PT-On(
tPtOff		.equ 0DDh	; PT-Off(
tPtChg		.equ 0DEh	; PT-Chg(
tDrawF		.equ 0DFh	; DrawF_
tShade		.equ 0E0h	; Shade(
tClrDraw	.equ 0E1h	; ClrDraw
tClLCD		.equ 0E2h	; ClrHome
tDispHome	.equ 0E3h	; DispHome
tDispG		.equ 0E4h	; DispGraph
tEnter		.equ 0ECh


;================ Parser Codes ================

OPEQ		.equ 10h	; =
OPNE		.equ 11h	; !=
OPLT		.equ 12h	; <
OPLE		.equ 13h	; <=
OPGT		.equ 14h	; >
OPGE		.equ 15h	; >=
OPAdd		.equ 30h	; +
OPSub		.equ 31h	; -
OPMul		.equ 40h	; *
OPDiv		.equ 41h	; /
OPnCr		.equ 50h	; nCr
OPnPr		.equ 51h	; nPr
OPMul1		.equ 60h	; weak implied mult
OPIPart		.equ 70h	; IPart
OPInt		.equ 71h	; Int
OPFPart		.equ 72h	; FPart
OPChs		.equ 73h	; (-)
OPAbs		.equ 74h	; abs
OPSqrt		.equ 75h	; square root
OPCubRt		.equ 76h	; cube root
OPLn		.equ 77h	; ln
OPExp		.equ 78h	; e^
OPLog		.equ 79h	; log
OPALog		.equ 7Ah	; 10^
OPSin		.equ 7Bh	; sin
OPASin		.equ 7Ch	; sin^-1
OPCos		.equ 7Dh	; cos
OPACos		.equ 7Eh	; cos^-1
OPTan		.equ 7Fh	; tan
OPATan		.equ 80h	; tan^-1
OPSinH		.equ 81h	; sinh
OPASinH		.equ 82h	; sinh^-1
OPCosH		.equ 83h	; cosh
OPACosH		.equ 84h	; cosh^-1
OPTanH		.equ 85h	; tanh
OPATanH		.equ 86h	; tanh^-1
OPDet		.equ 87h	; det
OPMul2		.equ 90h	; strong implied mult
OPPower		.equ 0A0h	; ^


;================ Large Font Characters ================

Lspace		.equ 01h	; space
Lstore		.equ 02h	; right arrow
LupArrow	.equ 03h	; up arrow
LdownArrow	.equ 04h	; down arrow
Lconvert	.equ 05h	; right "conversion" arrow
LLT		.equ 06h	; <
LLE		.equ 07h	; <=
LEQ		.equ 08h	; =
LNE		.equ 09h	; !=
LGT		.equ 0Ah	; >
LGE		.equ 0Bh	; >=
LinvEQ		.equ 0Ch	; inverted =
LsupEQ		.equ 0Dh	; superscript =
LplusSign	.equ 0Eh	; +
Ldash		.equ 0Fh	; -
Lasterisk	.equ 10h	; *
Lslash		.equ 11h	; /
Lcaret		.equ 12h	; ^
Lroot		.equ 13h	; square root
LlParen		.equ 14h	; (
LrParen		.equ 15h	; )
LlBrack		.equ 16h	; [
LrBrack		.equ 17h	; ]
LlBrace		.equ 18h	; {
LrBrace		.equ 19h	; }
Lquestion	.equ 1Ah	; ?
Lexclam		.equ 1Bh	; !
Lcolon		.equ 1Ch	; :
Lcomma		.equ 1Dh	; ,
Lellipsis	.equ 1Eh	; ...
Linverse	.equ 1Fh	; superscript -1
LsupDash	.equ 20h	; superscript -
LsupPeriod	.equ 21h	; superscript .
Lsup0		.equ 22h	; superscript 0
Lsup1		.equ 23h	; superscript 1
Lsup2		.equ 24h	; superscript 2
Lsup3		.equ 25h	; superscript 3
Lsup4		.equ 26h	; superscript 4
Lsup5		.equ 27h	; superscript 5
Lsup6		.equ 28h	; superscript 6
Lsup7		.equ 29h	; superscript 7
Lsup8		.equ 2Ah	; superscript 8
Lsup9		.equ 2Bh	; superscript 9
LsupE		.equ 2Ch	; superscript E
Lsub1		.equ 2Dh	; subscript 1
Lsub2		.equ 2Eh	; subscript 2
Lsub3		.equ 2Fh	; subscript 3
Lsub4		.equ 30h	; subscript 4
Lten		.equ 31h	; subscript 10
LsubT		.equ 32h	; subscript T
Lquote		.equ 33h	; "
Lneg		.equ 34h	; (-)
Lperiod		.equ 35h	; .
L0		.equ 36h	; 0
L1		.equ 37h	; 1
L2		.equ 38h	; 2
L3		.equ 39h	; 3
L4		.equ 3Ah	; 4
L5		.equ 3Bh	; 5
L6		.equ 3Ch	; 6
L7		.equ 3Dh	; 7
L8		.equ 3Eh	; 8
L9		.equ 3Fh	; 9
Lexponent	.equ 40h	; EE
Lradian		.equ 41h	; superscript r
Ldegree		.equ 42h	; degree sign
LsupTheta	.equ 43h	; superscript theta
LsupR		.equ 44h	; superscript R
LsupT		.equ 45h	; superscript T
LsupX		.equ 46h	; superscript X
LsupY		.equ 47h	; superscript Y
LxMean		.equ 48h	; x-bar
LyMean		.equ 49h	; y-bar
LcapSigma	.equ 4Ah	; Sigma
Lsigma		.equ 4Bh	; sigma
Lpi		.equ 4Ch	; pi
LcapA		.equ 4Dh	; A
LcapB		.equ 4Eh	; B
LcapC		.equ 4Fh	; C
LcapD		.equ 50h	; D
LcapE		.equ 51h	; E
LcapF		.equ 52h	; F
LcapG		.equ 53h	; G
LcapH		.equ 54h	; H
LcapI		.equ 55h	; I
LcapJ		.equ 56h	; J
LcapK		.equ 57h	; K
LcapL		.equ 58h	; L
LcapM		.equ 59h	; M
LcapN		.equ 5Ah	; N
LcapO		.equ 5Bh	; O
LcapP		.equ 5Ch	; P
LcapQ		.equ 5Dh	; Q
LcapR		.equ 5Eh	; R
LcapS		.equ 5Fh	; S
LcapT		.equ 60h	; T
LcapU		.equ 61h	; U
LcapV		.equ 62h	; V
LcapW		.equ 63h	; W
LcapX		.equ 64h	; X
LcapY		.equ 65h	; Y
LcapZ		.equ 66h	; Z
Ltheta		.equ 67h	; theta
La		.equ 68h	; a
Lb		.equ 69h	; b
Lc		.equ 6Ah	; c
Ld		.equ 6Bh	; d
Le 		.equ 6Ch	; e
Lf 		.equ 6Dh	; f
Lg 		.equ 6Eh	; g
Lh 		.equ 6Fh	; h
Li 		.equ 70h	; i
Ll 		.equ 71h	; l
Lm 		.equ 72h	; m
Ln 		.equ 73h	; n
Lo 		.equ 74h	; o
Lp 		.equ 75h	; p
Lq 		.equ 76h	; q
Lr 		.equ 77h	; r
Ls 		.equ 78h	; s
Lt 		.equ 79h	; t
Lu 		.equ 7Ah	; u
Lv 		.equ 7Bh	; v
Lw 		.equ 7Ch	; w
Lx 		.equ 7Dh	; x
Ly 		.equ 7Eh	; y
Lcross		.equ 7Fh	; cross
LcurO		.equ 80h	; normal (overwrite) cursor
LcurO2		.equ 81h	; 2nd overwrite cursor
LcurOcapA	.equ 82h	; Alpha overwrite cursor
LcurI		.equ 83h	; insert cursor
LcurI2		.equ 84h	; 2nd insert cursor
LcurIcapA	.equ 85h	; Alpha insert cursor


;================ "Small" Font Characters ================

SEQ		.equ LsupEQ
Sdash		.equ LsupDash
Speriod		.equ LsupPeriod
S0		.equ Lsup0
S1		.equ Lsup1
S2		.equ Lsup2
S3		.equ Lsup3
S4		.equ Lsup4
S5		.equ Lsup5
S6		.equ Lsup6
S7		.equ Lsup7
S8		.equ Lsup8
S9		.equ Lsup9
ScapE		.equ LsupE
Stheta		.equ LsupTheta
ScapR		.equ LsupR
ScapT		.equ LsupT
ScapX		.equ LsupX
ScapY		.equ LsupY


;================ Character Definitions ================

 .if _MINIASM >= 2

 .clearchars
 .defchar ' ', Lspace
 .defchar '<', LLT
 .defchar '=', LEQ
 .defchar '>', LGT
 .defchar '+', LplusSign
 .defchar '-', Ldash
 .defchar '*', Lasterisk
 .defchar '/', Lslash
 .defchar '^', Lcaret
 .defchar '(', LlParen
 .defchar ')', LrParen
 .defchar '[', LlBrack
 .defchar ']', LrBrack
 .defchar '{', LlBrace
 .defchar '}', LrBrace
 .defchar '?', Lquestion
 .defchar '!', Lexclam
 .defchar ':', Lcolon
 .defchar ',', Lcomma
 .defchar '\"', Lquote
 .defchar '.', Lperiod
 .defchar '0', L0
 .defchar '1', L1
 .defchar '2', L2
 .defchar '3', L3
 .defchar '4', L4
 .defchar '5', L5
 .defchar '6', L6
 .defchar '7', L7
 .defchar '8', L8
 .defchar '9', L9
 .defchar 'A', LcapA
 .defchar 'B', LcapB
 .defchar 'C', LcapC
 .defchar 'D', LcapD
 .defchar 'E', LcapE
 .defchar 'F', LcapF
 .defchar 'G', LcapG
 .defchar 'H', LcapH
 .defchar 'I', LcapI
 .defchar 'J', LcapJ
 .defchar 'K', LcapK
 .defchar 'L', LcapL
 .defchar 'M', LcapM
 .defchar 'N', LcapN
 .defchar 'O', LcapO
 .defchar 'P', LcapP
 .defchar 'Q', LcapQ
 .defchar 'R', LcapR
 .defchar 'S', LcapS
 .defchar 'T', LcapT
 .defchar 'U', LcapU
 .defchar 'V', LcapV
 .defchar 'W', LcapW
 .defchar 'X', LcapX
 .defchar 'Y', LcapY
 .defchar 'Z', LcapZ
 .defchar 'a', La
 .defchar 'b', Lb
 .defchar 'c', Lc
 .defchar 'd', Ld
 .defchar 'e', Le
 .defchar 'f', Lf
 .defchar 'g', Lg
 .defchar 'h', Lh
 .defchar 'i', Li
 .defchar 'l', Ll
 .defchar 'm', Lm
 .defchar 'n', Ln
 .defchar 'o', Lo
 .defchar 'p', Lp
 .defchar 'q', Lq
 .defchar 'r', Lr
 .defchar 's', Ls
 .defchar 't', Lt
 .defchar 'u', Lu
 .defchar 'v', Lv
 .defchar 'w', Lw
 .defchar 'x', Lx
 .defchar 'y', Ly
 .defchar '_', LcurI

 .endif


;================ Definitions for Unity ===============

customIntCall	.equ 0D3E5h
customIntAddr	.equ 0D3E6h

; Macro to define a custom interrupt routine that runs continuously
;
; Syntax:
;	DEFINE_CUSTOM_INT MyInterrupt
;	<your code goes here>
;	RET

DEFINE_CUSTOM_INT .macro name
_intstart_ ## name:
_intsp_ ## name .equ $ + 1
	ld hl,0
	sbc hl,sp
	jr nc,_intmain_ ## name
	CUSTOM_INT_STOP
	ret
_intmain_ ## name:
	.endm

; Macro to define a custom interrupt routine that runs only once
;
; Syntax:
;	DEFINE_CUSTOM_INT_NO_REPEAT MyInterrupt
;	<your code goes here>
;	RET

DEFINE_CUSTOM_INT_NO_REPEAT .macro name
_intstart_ ## name:
_intsp_ ## name .equ $ + 1
	ld hl,0
	sbc hl,sp
	CUSTOM_INT_STOP
	ret c
_intmain_ ## name:
	.endm

; Macro to set/enable custom interrupt routine
;
; Syntax:
;	CUSTOM_INT_START MyInterrupt

CUSTOM_INT_START .macro name
	ld (_intsp_ ## name),sp
	ld hl,_intstart_ ## name
	ld (customIntAddr),hl
	ld a,0CDh
	ld (customIntCall),a
	.endm

; Macro to disable custom interrupt routine
;
; Syntax:
;	CUSTOM_INT_STOP

CUSTOM_INT_STOP .macro
	ld a,21h
	ld (customIntCall),a
	.endm

; Macro to re-enable previous custom interrupt routine (use with caution!)
;
; Syntax:
;	CUSTOM_INT_CONTINUE

CUSTOM_INT_CONTINUE .macro
	ld a,0CDh
	ld (customIntCall),a
	.endm

